package com.floreantpos.util;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Future;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.json.JSONArray;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.floreantpos.PosException;
import com.floreantpos.model.DosageForms;
import com.floreantpos.model.InventoryStockUnit;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.InventoryUnitGroup;
import com.floreantpos.model.Manufacturer;
import com.floreantpos.model.MedGenerics;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.Recepie;
import com.floreantpos.model.RecepieItem;
import com.floreantpos.model.ReportGroup;
import com.floreantpos.model.Specimen;
import com.floreantpos.model.Store;
import com.floreantpos.model.Tag;
import com.floreantpos.model.Tax;
import com.floreantpos.model.TaxGroup;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemGroup;
import com.floreantpos.model.dao.DosageFormsDAO;
import com.floreantpos.model.dao.InventoryStockUnitDAO;
import com.floreantpos.model.dao.InventoryUnitDAO;
import com.floreantpos.model.dao.InventoryUnitGroupDAO;
import com.floreantpos.model.dao.ManufacturerDAO;
import com.floreantpos.model.dao.MedGenericsDAO;
import com.floreantpos.model.dao.MenuCategoryDAO;
import com.floreantpos.model.dao.MenuGroupDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.RecepieDAO;
import com.floreantpos.model.dao.RecepieItemDAO;
import com.floreantpos.model.dao.ReportGroupDAO;
import com.floreantpos.model.dao.SpecimenDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TagDAO;
import com.floreantpos.model.dao.TaxDAO;
import com.floreantpos.model.dao.TaxGroupDAO;
import com.floreantpos.model.dao.TestItemDAO;
import com.floreantpos.model.dao.TestItemGroupDAO;
import com.floreantpos.model.util.DataProvider;

public class MenuItemJsonManager {

	public byte[] doExportAndGetFile(ProductType productType, Future<?> currentTask) throws Exception {
		List<MenuItem> list = MenuItemDAO.getInstance().findByProductType(productType);

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		mapper.addMixIn(MenuItem.class, MenuItemMixIn.class);
		mapper.addMixIn(InventoryUnit.class, InventoryUnitMixIn.class);
		//mapper.addMixIn(Recepie.class, ReceipeMixIn.class);

		JSONArray jsonArray = new JSONArray();
		for (MenuItem menuItem : list) {
			if (currentTask != null && currentTask.isCancelled()) {
				break;
			}

			MenuItemDAO.getInstance().initialize(menuItem);
			JSONObject jsonObject = new JSONObject(mapper.writeValueAsString(menuItem));

			if (jsonObject.has("testItems")) {
				JSONArray testItems = jsonObject.getJSONArray("testItems");
				for (int i = 0; i < testItems.length(); i++) {
					Object object = testItems.get(i);
					if (object instanceof JSONObject) {
						JSONObject testItem = (JSONObject) object;
						if (testItem.has("testItemGroupId") && !testItem.isNull("testItemGroupId")) {
							String testItemGroupId = testItem.getString("testItemGroupId");
							if (currentTask != null && currentTask.isCancelled()) {
								break;
							}
							TestItemGroup testItemGroup = TestItemGroupDAO.getInstance().get(testItemGroupId);
							if (testItemGroup != null) {
								testItem.put("testItemGroup", new JSONObject(mapper.writeValueAsString(testItemGroup)));
							}
						}
						if (testItem.has("unit") && !testItem.isNull("unit")) {
							String unitId = testItem.getString("unit");
							if (currentTask != null && currentTask.isCancelled()) {
								break;
							}
							InventoryUnit unit = InventoryUnitDAO.getInstance().get(unitId);
							if (unit != null) {
								testItem.put("inventoryUnit", new JSONObject(mapper.writeValueAsString(unit)));
							}
						}
					}
				}
			}

			if (StringUtils.isNotBlank(menuItem.getMenuCategoryId())) {
				if (currentTask != null && currentTask.isCancelled()) {
					break;
				}
				MenuCategory menuCategory = MenuCategoryDAO.getInstance().get(menuItem.getMenuCategoryId());
				if (menuCategory != null) {
					jsonObject.put("menuCategory", new JSONObject(mapper.writeValueAsString(menuCategory)));
				}
			}

			if (StringUtils.isNotBlank(menuItem.getMenuGroupId())) {
				if (currentTask != null && currentTask.isCancelled()) {
					break;
				}
				MenuGroup menuGroup = MenuGroupDAO.getInstance().get(menuItem.getMenuGroupId());
				if (menuGroup != null) {
					jsonObject.put("menuGroup", new JSONObject(mapper.writeValueAsString(menuGroup)));
				}
			}

			if (StringUtils.isNotBlank(menuItem.getContainerId())) {
				Tag container = (Tag) DataProvider.get().getObjectOf(Tag.class, menuItem.getContainerId());
				if (container != null) {
					jsonObject.put("container", new JSONObject(mapper.writeValueAsString(container)));
				}
			}

			if (StringUtils.isNotBlank(menuItem.getSpecimenId())) {
				Specimen specimen = (Specimen) DataProvider.get().getObjectOf(Specimen.class, menuItem.getSpecimenId());
				if (specimen != null) {
					jsonObject.put("specimen", new JSONObject(mapper.writeValueAsString(specimen)));
				}
			}

			ReportGroup reportGroup = menuItem.getReportGroup();
			if (reportGroup != null) {
				jsonObject.put("reportGroup", new JSONObject(mapper.writeValueAsString(reportGroup)));
			}

			TaxGroup taxGroup = menuItem.getTaxGroup();
			if (taxGroup != null) {
				jsonObject.put("taxGroup", new JSONObject(mapper.writeValueAsString(taxGroup)));
			}

			if (StringUtils.isNotBlank(menuItem.getDefaultRecipeId())) {
				if (currentTask != null && currentTask.isCancelled()) {
					break;
				}
				Recepie recepie = RecepieDAO.getInstance().get(menuItem.getDefaultRecipeId());
				if (recepie != null && !recepie.isDeleted()) {
					recepie.setMenuItem(null);
					for (RecepieItem recepieItem : recepie.getRecepieItems()) {
						MenuItem inventoryItem = recepieItem.getInventoryItem();
						if (inventoryItem == null) {
							continue;
						}
						if (currentTask != null && currentTask.isCancelled()) {
							break;
						}
						MenuItemDAO.getInstance().initialize(inventoryItem);
					}
					jsonObject.put("recepie", new JSONObject(mapper.writeValueAsString(recepie)));
				}
			}

			jsonArray.put(jsonObject);
		}

		return IOUtils.toByteArray(IOUtils.toInputStream(jsonArray.toString(), "UTF-8"));
	}

	public boolean doImportAndSaveMenuItem(ProductType productType, File file) throws Exception {
		if (!file.exists()) {
			throw new PosException("File not found.");
		}

		ObjectMapper mapper = new ObjectMapper();
		mapper.addMixIn(MenuItem.class, MenuItemMixIn.class);
		mapper.addMixIn(RecepieItem.class, RecepieItemMixIn.class);

		JsonNode rootNode = mapper.readTree(file);

		if (rootNode.isArray()) {
			try (Session session = MenuItemDAO.getInstance().createNewSession()) {
				for (JsonNode menuItemJsonNode : rootNode) {
					Transaction beginTransaction = session.beginTransaction();

					Set<String> labTestMethods = new HashSet<>();

					convertMenuitemAndSave(mapper, session, menuItemJsonNode, labTestMethods);

					List<String> methods = StoreDAO.getRestaurant().getMethods();
					labTestMethods.removeIf(method -> methods.contains(method) && StringUtils.isBlank(method));

					if (!labTestMethods.isEmpty()) {
						Store restaurant = StoreDAO.getRestaurant();
						boolean isStoreSaveRequired = false;
						for (String method : labTestMethods) {
							if (restaurant.addToMethod(method)) {
								isStoreSaveRequired = true;
							}
						}
						if (isStoreSaveRequired) {
							StoreDAO.getInstance().saveOrUpdate(restaurant, session);
						}
					}

					beginTransaction.commit();
				}
			}
		}

		return true;
	}

	private void convertMenuitemAndSave(ObjectMapper mapper, Session session, JsonNode menuItemJsonNode, Set<String> labTestMethod) {
		MenuItem menuItem = mapper.convertValue(menuItemJsonNode, MenuItem.class);

		labTestMethod.add(menuItem.getLabTestMethod());

		List<InventoryStockUnit> stockUnits = menuItem.getStockUnits();
		menuItem.setStockUnits(null);

		List<TestItem> testItems = menuItem.getTestItems();
		menuItem.setTestItems(null);

		saveOrUpdateMenuItem(session, menuItem);

		if (stockUnits != null && !stockUnits.isEmpty()) {
			for (InventoryStockUnit inventoryStockUnit : stockUnits) {
				inventoryStockUnit.setMenuItemId(menuItem.getId());
				InventoryStockUnit existingItem = InventoryStockUnitDAO.getInstance().get(inventoryStockUnit.getId(), session);
				if (existingItem == null) {
					session.save(inventoryStockUnit);
				}
				else {
					//session.merge(inventoryStockUnit);
				}
			}
			menuItem.setStockUnits(stockUnits);
		}

		if (testItems != null && !testItems.isEmpty()) {
			for (TestItem testItem : testItems) {
				if (testItem == null || StringUtils.isBlank(testItem.getId())) {
					continue;
				}
				TestItem existingItem = TestItemDAO.getInstance().get(testItem.getId(), session);
				if (existingItem == null) {
					session.save(testItem);
				}
				else {
					//session.merge(testItem);
				}
			}
			menuItem.setTestItems(testItems);
		}

		if (menuItemJsonNode.has("testItems")) {
			JsonNode testItemsJsonNode = menuItemJsonNode.get("testItems");
			for (JsonNode testItemJson : testItemsJsonNode) {
				if (testItemJson.has("testItemGroup")) {
					JsonNode testItemGroupJson = testItemJson.get("testItemGroup");
					if (testItemGroupJson != null && testItemGroupJson.isObject() && !testItemGroupJson.isNull()) {
						TestItemGroup testItemGroup = mapper.convertValue(testItemGroupJson, TestItemGroup.class);
						TestItemGroup existingItem = StringUtils.isBlank(testItemGroup.getId()) ? null
								: TestItemGroupDAO.getInstance().get(testItemGroup.getId(), session);
						if (existingItem == null) {
							session.save(testItemGroup);
						}
						else {
							//session.merge(testItemGroup);
						}
					}
				}
				if (testItemJson.has("inventoryUnit")) {
					JsonNode inventoryUnitJson = testItemJson.get("inventoryUnit");
					convertInventoryUnitAndSave(mapper, session, inventoryUnitJson);
				}
			}
		}

		JsonNode menuCategoryJsonNode = menuItemJsonNode.get("menuCategory");
		if (menuCategoryJsonNode != null && menuCategoryJsonNode.isObject() && !menuCategoryJsonNode.isNull()) {
			MenuCategory menuCategory = mapper.convertValue(menuCategoryJsonNode, MenuCategory.class);
			MenuCategory existingItem = StringUtils.isBlank(menuCategory.getId()) ? null : MenuCategoryDAO.getInstance().get(menuCategory.getId(), session);
			if (existingItem == null) {
				session.save(menuCategory);
			}
			else {
				//session.merge(menuCategory);
			}
		}

		JsonNode menuGroupJsonNode = menuItemJsonNode.get("menuGroup");
		if (menuGroupJsonNode != null && menuGroupJsonNode.isObject() && !menuGroupJsonNode.isNull()) {
			MenuGroup menuGroup = mapper.convertValue(menuGroupJsonNode, MenuGroup.class);
			MenuGroup existingItem = StringUtils.isBlank(menuGroup.getId()) ? null : MenuGroupDAO.getInstance().get(menuGroup.getId(), session);
			if (existingItem == null) {
				session.save(menuGroup);
			}
			else {
				//session.merge(menuGroup);
			}
		}

		JsonNode containerJsonNode = menuItemJsonNode.get("container");
		if (containerJsonNode != null && containerJsonNode.isObject() && !containerJsonNode.isNull()) {
			Tag container = mapper.convertValue(containerJsonNode, Tag.class);
			Tag existingItem = StringUtils.isBlank(container.getId()) ? null : TagDAO.getInstance().get(container.getId(), session);
			if (existingItem == null) {
				session.save(container);
			}
			else {
				//session.merge(container);
			}
		}

		JsonNode specimenJsonNode = menuItemJsonNode.get("specimen");
		if (specimenJsonNode != null && specimenJsonNode.isObject() && !specimenJsonNode.isNull()) {
			Specimen specimen = mapper.convertValue(specimenJsonNode, Specimen.class);
			Specimen existingItem = StringUtils.isBlank(specimen.getId()) ? null : SpecimenDAO.getInstance().get(specimen.getId(), session);
			if (existingItem == null) {
				session.save(specimen);
			}
			else {
				//session.merge(specimen);
			}
		}

		JsonNode manufactureJsonNode = menuItemJsonNode.get("manufacture");
		if (manufactureJsonNode != null && manufactureJsonNode.isObject() && !manufactureJsonNode.isNull()) {
			Manufacturer manufacturer = mapper.convertValue(manufactureJsonNode, Manufacturer.class);
			Manufacturer existingItem = StringUtils.isBlank(manufacturer.getId()) ? null : ManufacturerDAO.getInstance().get(manufacturer.getId(), session);
			if (existingItem == null) {
				session.save(manufacturer);
			}
			else {
				//session.merge(manufacturer);
			}
		}

		JsonNode medGenericsJsonNode = menuItemJsonNode.get("medGenerics");
		if (medGenericsJsonNode != null && medGenericsJsonNode.isObject() && !medGenericsJsonNode.isNull()) {
			MedGenerics medGenerics = mapper.convertValue(medGenericsJsonNode, MedGenerics.class);
			MedGenerics existingItem = StringUtils.isBlank(medGenerics.getId()) ? null : MedGenericsDAO.getInstance().get(medGenerics.getId(), session);
			if (existingItem == null) {
				session.save(medGenerics);
			}
			else {
				//session.merge(medGenerics);
			}
		}

		JsonNode dosageFormsJsonNode = menuItemJsonNode.get("dosageForms");
		if (dosageFormsJsonNode != null && dosageFormsJsonNode.isObject() && !dosageFormsJsonNode.isNull()) {
			DosageForms dosageForms = mapper.convertValue(dosageFormsJsonNode, DosageForms.class);
			DosageForms existingItem = StringUtils.isBlank(dosageForms.getId()) ? null : DosageFormsDAO.getInstance().get(dosageForms.getId(), session);
			if (existingItem == null) {
				session.save(dosageForms);
			}
			else {
				//session.merge(dosageForms);
			}
		}

		JsonNode reportGroupJsonNode = menuItemJsonNode.get("reportGroup");
		if (reportGroupJsonNode != null && reportGroupJsonNode.isObject() && !reportGroupJsonNode.isNull()) {
			ReportGroup reportGroup = mapper.convertValue(reportGroupJsonNode, ReportGroup.class);
			ReportGroup existingItem = StringUtils.isBlank(reportGroup.getId()) ? null : ReportGroupDAO.getInstance().get(reportGroup.getId(), session);
			if (existingItem == null) {
				session.save(reportGroup);
			}
			else {
				//session.merge(reportGroup);
			}
		}

		JsonNode taxGroupJsonNode = menuItemJsonNode.get("taxGroup");
		if (taxGroupJsonNode != null && taxGroupJsonNode.isObject() && !taxGroupJsonNode.isNull()) {
			TaxGroup taxGroup = mapper.convertValue(taxGroupJsonNode, TaxGroup.class);

			String currentOutletId = DataProvider.get().getCurrentOutletId();

			List<Tax> taxes = taxGroup.getTaxes();
			if (taxes != null && !taxes.isEmpty()) {
				for (Tax tax : taxes) {
					tax.setOutletId(currentOutletId);
					Tax existingItem = StringUtils.isBlank(tax.getId()) ? null : TaxDAO.getInstance().get(new Tax(tax.getId(), currentOutletId), session);
					if (existingItem == null) {
						session.save(tax);
					}
					else {
						//session.merge(tax);
					}
				}
			}

			TaxGroup existingItem = StringUtils.isBlank(taxGroup.getId()) ? null
					: TaxGroupDAO.getInstance().get(new TaxGroup(taxGroup.getId(), currentOutletId), session);
			if (existingItem == null) {
				taxGroup.setOutletId(currentOutletId);
				session.save(taxGroup);
			}
			else {
				//session.merge(taxGroup);
			}
		}

		JsonNode unitNode = menuItemJsonNode.get("inventoryUnit");
		convertInventoryUnitAndSave(mapper, session, unitNode);

		JsonNode recepieNode = menuItemJsonNode.get("recepie");
		if (recepieNode != null && recepieNode.isObject() && !recepieNode.isNull()) {
			convertRecepieAndSave(mapper, session, recepieNode, labTestMethod);
		}

		JsonNode additionalItemsNode = menuItemJsonNode.get("additionalItems");
		if (additionalItemsNode != null && additionalItemsNode.isArray() && !additionalItemsNode.isNull()) {
			for (JsonNode additionalItemJson : additionalItemsNode) {
				JsonNode additionalItemJsonNode = additionalItemJson.get("inventoryItem");
				convertMenuitemAndSave(mapper, session, additionalItemJsonNode, labTestMethod);
			}
		}

	}

	private void convertInventoryUnitAndSave(ObjectMapper mapper, Session session, JsonNode unitNode) {
		if (unitNode != null && unitNode.isObject() && !unitNode.isNull()) {
			JsonNode unitGroupNode = unitNode.get("inventoryUnitGroup");
			if (unitGroupNode != null && unitGroupNode.isObject() && !unitGroupNode.isNull()) {
				InventoryUnitGroup unitGroup = mapper.convertValue(unitGroupNode, InventoryUnitGroup.class);
				InventoryUnitGroup existingItem = StringUtils.isBlank(unitGroup.getId()) ? null
						: InventoryUnitGroupDAO.getInstance().get(unitGroup.getId(), session);
				if (existingItem == null) {
					session.save(unitGroup);
				}
				else {
					//session.merge(unitGroup);
				}
			}

			InventoryUnit unit = mapper.convertValue(unitNode, InventoryUnit.class);
			if (StringUtils.isBlank(unit.getId())) {
				return;
			}
			InventoryUnit existingItem = StringUtils.isBlank(unit.getId()) ? null : InventoryUnitDAO.getInstance().get(unit.getId(), session);
			if (existingItem == null) {
				session.save(unit);
			}
			else {
				//session.merge(unit);
			}
		}
	}

	private void convertRecepieAndSave(ObjectMapper mapper, Session session, JsonNode recepieNode, Set<String> labTestMethod) {
		Recepie recepie = mapper.convertValue(recepieNode, Recepie.class);
		if (StringUtils.isNotBlank(recepie.getId())) {
			recepie.setRecepieItems(null);
			Recepie existingItem = RecepieDAO.getInstance().get(recepie.getId(), session);
			if (existingItem == null) {
				session.save(recepie);
			}
			else {
				//session.merge(recepie);
			}
		}

		JsonNode recepieItemsJsonNode = recepieNode.get("recepieItems");
		if (recepieItemsJsonNode != null && recepieItemsJsonNode.isArray() && !recepieItemsJsonNode.isNull()) {
			for (JsonNode recepieItemJson : recepieItemsJsonNode) {
				JsonNode menuItemJson = recepieItemJson.get("inventoryItem");
				convertMenuitemAndSave(mapper, session, menuItemJson, labTestMethod);

				RecepieItem recepieItem = mapper.convertValue(recepieItemJson, RecepieItem.class);
				recepieItem.setRecepie(recepie);
				recepie.addRecepieItem(recepieItem);
				RecepieItem existingItem = StringUtils.isBlank(recepieItem.getId()) ? null : RecepieItemDAO.getInstance().get(recepieItem.getId(), session);
				if (existingItem == null) {
					session.save(recepieItem);
				}
				else {
					//session.merge(recepieItem);
				}
			}
		}

	}

	private void saveOrUpdateMenuItem(Session session, MenuItem menuItem) {
		if (menuItem == null) {
			return;
		}

		MenuItem existingItem = StringUtils.isBlank(menuItem.getId()) ? null : MenuItemDAO.getInstance().get(menuItem.getId(), session);
		if (existingItem == null) {
			session.save(menuItem);
		}
		else {
			//session.merge(menuItem);
		}
	}

	public abstract class MenuItemMixIn {
		@JsonDeserialize(using = MenuItemBooleanDeserializer.class)
		public abstract Boolean getInventoryItem();

		@JsonIgnore(false)
		@JsonProperty("additionalItems")
		public abstract List<RecepieItem> getAdditionalItems();

		@JsonIgnore(false)
		@JsonProperty("inventoryUnit")
		public abstract InventoryUnit getUnit();

		@JsonIgnore(false)
		@JsonProperty("inventoryStockUnit")
		public abstract InventoryStockUnit getStockUnits();

		@JsonIgnore(false)
		@JsonProperty("manufacture")
		public abstract Manufacturer getManufacture();

		@JsonIgnore(false)
		@JsonProperty("medGenerics")
		public abstract MedGenerics getMedGenerics();

		@JsonIgnore(false)
		@JsonProperty("dosageForms")
		public abstract DosageForms getDosageForms();

	}

	public abstract class InventoryUnitMixIn {
		@JsonProperty("inventoryUnitGroup")
		public abstract InventoryUnitGroup getUnitGroup();
	}

	//public abstract class ReceipeMixIn {
	//	@JsonIgnore()
	//	public abstract List<ManufacturingItem> getManufacturingItems();
	//}

	public static class MenuItemBooleanDeserializer extends JsonDeserializer<Boolean> {

		public MenuItemBooleanDeserializer() {
		}

		@Override
		public Boolean deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
			JsonToken token = p.getCurrentToken();

			if (token == JsonToken.VALUE_TRUE)
				return true;
			if (token == JsonToken.VALUE_FALSE)
				return false;

			if (token == JsonToken.START_OBJECT) {
				p.skipChildren();
				return false;
			}

			return null;
		}
	}

	public abstract class RecepieItemMixIn {
		@JsonDeserialize(using = MenuItemDeserializer.class)
		public abstract MenuItem getInventoryItem();
	}

	public static class MenuItemDeserializer extends JsonDeserializer<MenuItem> {

		public MenuItemDeserializer() {
		}

		private final ObjectMapper mapper = new ObjectMapper();

		@Override
		public MenuItem deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
			JsonNode node = p.getCodec().readTree(p);
			return mapper.treeToValue(node, MenuItem.class);
		}
	}
}
