/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.GratuityPaymentHistory;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.util.POSUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class GratuityDAO extends BaseGratuityDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public GratuityDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public List<Gratuity> findByUser(User user) throws PosException {
		return findByUser(user, false);
	}

	public List<Gratuity> findByUser(User user, boolean includePaid) throws PosException {
		Session session = null;

		try {
			session = getSession();

			Criteria criteria = session.createCriteria(getReferenceClass());

			if (user != null)
				criteria.add(Restrictions.eq(Gratuity.PROP_OWNER_ID, user.getId()));

			if (!includePaid)
				criteria.add(Restrictions.eq(Gratuity.PROP_PAID, Boolean.FALSE));

			criteria.add(Restrictions.eq(Gratuity.PROP_REFUNDED, Boolean.FALSE));
			return criteria.list();
		} catch (Exception e) {
			throw new PosException("" + user.getFirstName() + " " + user.getLastName()); //$NON-NLS-1$ //$NON-NLS-2$
		} finally {
			closeSession(session);
		}
	}

	public void payGratuities(List<Gratuity> gratuities, List<GratuityPaymentHistory> gratuityPaymentList) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Gratuity gratuity : gratuities) {
				saveOrUpdate(gratuity, session);
			}
			for (GratuityPaymentHistory gratuityPayment : gratuityPaymentList) {
				GratuityPaymentHistoryDAO.getInstance().saveOrUpdate(gratuityPayment, session);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw new PosException(""); //$NON-NLS-1$
		} finally {
			closeSession(session);
		}
	}

	public double findTotalGratuityAmount(Date fromDate, Date toDate, User user) throws PosException {
		Session session = null;
		try {
			session = createNewSession();
			//DetachedCriteria detachedCriteria = DetachedCriteria.forClass(Ticket.class);
			Criteria criteria = session.createCriteria(getReferenceClass());
			//criteria.createAlias(Gratuity.PROP_TICKET_ID, alias)
			//			criteria.add(Restrictions.between(DeclaredTips.PROP_DECLARED_TIME, fromDate, toDate));
			if (user != null) {
				criteria.add(Restrictions.eq(Gratuity.PROP_OWNER_ID, user.getId()));
			}
			criteria.add(Restrictions.eq(Gratuity.PROP_REFUNDED, Boolean.FALSE));
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(Gratuity.PROP_AMOUNT), Gratuity.PROP_AMOUNT);

			criteria.setProjection(projectionList);
			Number totalAmount = (Number) criteria.uniqueResult();
			if (totalAmount != null) {
				return totalAmount.doubleValue();

			}
		} finally {
			closeSession(session);
		}
		return 0.0;
	}

	public void saveOrUpdateGratuityList(List<Gratuity> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			for (Iterator<Gratuity> iterator = dataList.iterator(); iterator.hasNext();) {
				Gratuity item = (Gratuity) iterator.next();
				Gratuity existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), "Gratuity " + item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	public double findTotalPaidGratuititesForStaffBank(CashDrawer staffBank) throws PosException {
		try (Session session = createNewSession()) {
			DetachedCriteria detachedCriteria = DetachedCriteria.forClass(PosTransaction.class);
			detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			detachedCriteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, staffBank.getId()));
			detachedCriteria.setProjection(Projections.property(PosTransaction.PROP_TICKET));

			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Property.forName(Gratuity.PROP_TICKET_ID).in(detachedCriteria));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(Gratuity.PROP_AMOUNT));
			projectionList.add(Projections.sum(Gratuity.PROP_TIPS_PAID_AMOUNT));
			criteria.setProjection(projectionList);

			List list = criteria.list();
			double total = 0;
			if (list.size() > 0) {
				Object[] objects = (Object[]) list.get(0);
				if (objects.length > 0) {
					total = POSUtil.getDoubleAmount(objects[0]);
				}
				if (objects.length > 1) {
					total -= POSUtil.getDoubleAmount(objects[1]);
				}

			}
			criteria = session.createCriteria(GratuityPaymentHistory.class);
			criteria.add(Restrictions.eq(GratuityPaymentHistory.PROP_CASH_DRAWER, staffBank));
			criteria.setProjection(Projections.sum(GratuityPaymentHistory.PROP_AMOUNT));
			return total + POSUtil.getDoubleAmount(criteria.uniqueResult());
		}
	}

}