package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.Challan;
import com.floreantpos.model.ChallanItemDto;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.ext.InvMapKey;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginatedListModel;

public class ChallanDAO extends BaseChallanDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		Challan challan = (Challan) obj;
		updateTime(obj);
		Serializable save = super.save(obj, s);
		performPostSaveOperations(s, challan);
		return save;
	}

	@Override
	protected void update(Object obj, Session s) {
		Challan challan = (Challan) obj;
		updateTime(obj);
		super.update(obj, s);
		performPostSaveOperations(s, challan);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		Challan challan = (Challan) obj;
		updateTime(obj);
		super.saveOrUpdate(obj, s);
		performPostSaveOperations(s, challan);
	}

	@Override
	public void delete(Object obj, Session s) throws HibernateException {
		Challan challan = (Challan) obj;
		challan.setDeleted(true);
		update(challan, s);
	}

	private void performPostSaveOperations(Session session, Challan challan) {
		updateStock(challan, session);
		challan.setShouldUpdateStock(false);
	}

	private void updateStock(Challan challan, Session session) {
		try {
			if (!challan.isShouldUpdateStock()) {
				return;
			}

			HashMap<InvMapKey, Double> itemMap = buildItemMapForInventoryAdjustment(challan, session);

			Ticket ticket = TicketDAO.getInstance().get(challan.getTicketId(), challan.getOutletId(), session);
			if (ticket == null) {
				return;
			}

			TicketDAO.getInstance().adjustInventory(ticket, itemMap, InventoryTransactionType.OUT, InventoryTransaction.REASON_TICKET_SALES,
					DataProvider.get().getDefaultOutLocation(), session);

		} catch (Exception e) {
			PosLog.error(getClass(), "Failed to update stock balance for challan: " + challan.getId(), e);
		}
	}

	private HashMap<InvMapKey, Double> buildItemMapForInventoryAdjustment(Challan challan, Session session) {
		HashMap<InvMapKey, Double> itemMap = new HashMap<>();
		for (ChallanItemDto challanItemDto : challan.getItems()) {
			String ticketItemId = challanItemDto.getTicketItemId();
			TicketItem ticketItem = session.get(TicketItem.class, ticketItemId);

			if (ticketItem.getMenuItemId() == null || ticketItem.isVoided() || ticketItem.isItemReturned() || ticketItem.isInventoryAdjusted()) {
				continue;
			}

			if (ticketItem.isComboItem()) {
				buldComboitemMapForInventoryAdjustment(itemMap, ticketItem, challanItemDto.getItemQuantity());
				continue;
			}

			InvMapKey key = new InvMapKey(ticketItem);
			Double toBeAdjustQty = challanItemDto.getItemQuantity();
			itemMap.put(key, toBeAdjustQty);

			ticketItem.setInventoryAdjustQty(ticketItem.getInventoryAdjustQty() + toBeAdjustQty);
		}
		return itemMap;
	}

	private void buldComboitemMapForInventoryAdjustment(HashMap<InvMapKey, Double> itemMap, TicketItem parentItem, Double challanItemQuantity) {
		List<TicketItem> comboItems = parentItem.getComboItems();
		if (comboItems == null) {
			return;
		}
		for (TicketItem ticketItem : comboItems) {
			InvMapKey key = new InvMapKey(ticketItem);
			Double toBeAdjustQty = (ticketItem.getQuantity() / parentItem.getQuantity()) * challanItemQuantity;
			itemMap.put(key, toBeAdjustQty);
			ticketItem.setInventoryAdjustQty(ticketItem.getInventoryAdjustQty() + toBeAdjustQty);
		}
	}

	public List<Challan> getChallansByTicketId(String ticketId) {
		if (StringUtils.isBlank(ticketId)) {
			return Collections.emptyList();
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Challan.class);
			criteria.add(Restrictions.eq(Challan.PROP_TICKET_ID, ticketId));
			return criteria.list();
		}
	}

	public double getShipmentQuantityByTicketItemId(String ticketItemId) {
		double qty = 0;
		if (StringUtils.isBlank(ticketItemId)) {
			return qty;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Challan.class);
			criteria.add(Restrictions.ilike(Challan.PROP_ITEM_DATA, ticketItemId, MatchMode.ANYWHERE));
			List<Challan> list = criteria.list();
			for (Challan challan : list) {
				qty += challan.getItems().stream().filter(t -> ticketItemId.equals(t.getTicketItemId())).mapToDouble(value -> value.getItemQuantity()).sum();
			}
		}
		return qty;
	}

	public void laodChallans(String projectId, String invoiceId, String customerId, PaginatedListModel<Challan> dataModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Challan.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(projectId)) {
				criteria.add(Restrictions.eq(Challan.PROP_PROJECT_ID, projectId));
			}

			if (StringUtils.isNotBlank(invoiceId)) {
				criteria.add(Restrictions.eq(Challan.PROP_TICKET_ID, invoiceId));
			}

			if (StringUtils.isNotBlank(customerId)) {
				criteria.add(Restrictions.eq(Challan.PROP_CUSTOMER_ID, customerId));
				//addCriteriaPropertyFilter(criteria, Challan.PROP_PROPERTIES, "customer.id", customerId); //$NON-NLS-1$
			}

			dataModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());

			criteria.addOrder(Order.desc(Challan.PROP_CREATE_DATE));
			dataModel.setData(criteria.list());
		}
	}

	public int rowCountByTicketId(String ticketId) {
		if (StringUtils.isBlank(ticketId)) {
			return 0;
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Challan.class);
			criteria.add(Restrictions.eq(Challan.PROP_TICKET_ID, ticketId));
			addDeletedFilter(criteria);
			return rowCount(criteria);
		}
	}

	public boolean isShippedAllChallan(String ticketId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Challan.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Challan.PROP_TICKET_ID, ticketId));
			criteria.add(Restrictions.eq(Challan.PROP_SHIPPED, false));
			List list = criteria.list();
			if (!list.isEmpty()) {
				return false;
			}
			return true;
		}

	}
}