/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;

import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.Challan;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.ExpenseTransaction;
import com.floreantpos.model.ImageResource;
import com.floreantpos.model.OnlineStore;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.SiiopaCustomer;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.BalanceUpdateTransactionDAO;
import com.floreantpos.model.dao.BookingInfoDAO;
import com.floreantpos.model.dao.ChallanDAO;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.ImageResourceDAO;
import com.floreantpos.model.dao.OnlineStoreDAO;
import com.floreantpos.model.dao.OutletDAO;
import com.floreantpos.model.dao.PosTransactionDAO;
import com.floreantpos.model.dao.SiiopaCustomerDAO;
import com.floreantpos.model.dao.TestItemDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.dao.TicketItemDAO;
import com.floreantpos.model.dao.UserDAO;
import com.floreantpos.util.POSUtil;

public class UpdateDBTo418 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo418(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void update() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		updateChallanFields();
		updateCustomerFields(1000);
		updateImageResourceFields();
		updateOnlineStoreFields();
		updateSiiopaCustomerFields();
		updateOutletFields();
		updateTestItemFields();
		updateTicketItemFields(1000);
		updateUserFields();

		updateTransactionFields(1000);

		updateBalanceUpdateTransactionFields();
		updateAdmissions();

		updateTicketFields(1000);

		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}

	public void updateChallanFields() {
		int totalRows = rowCount(Challan.class);
		PosLog.info(UpdateDBTo418.class, "Found total challan: " + totalRows); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<Challan> pagination = new Pagination<Challan>(count, increment);
				pagination.setNumRows(totalRows);
				findData(pagination, session, Challan.class, Challan.PROP_CREATE_DATE);

				List<Challan> challans = pagination.getDataList();
				for (Challan challan : challans) {

					String challanCustomerId = challan.getProperty("customer.id"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(challanCustomerId)) {
						challan.setCustomerId(challanCustomerId);

						PosLog.info(getClass(),
								"Challan " + challan.getId() + " is updating 'customer.id' from property to field. customerID: " + challanCustomerId);
						ChallanDAO.getInstance().saveOrUpdate(challan, session);
					}
				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "Challan update completed successfully"); //$NON-NLS-1$
	}

	public void updateCustomerFields(int increment) {
		int totalRows = rowCount(Customer.class);
		PosLog.info(UpdateDBTo418.class, "Found total customer: " + totalRows); //$NON-NLS-1$

		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<Customer> pagination = new Pagination<Customer>(count, increment);
				pagination.setNumRows(totalRows);
				findData(pagination, session, Customer.class, Customer.PROP_CREATE_DATE);

				List<Customer> customers = pagination.getDataList();
				for (Customer customer : customers) {

					String agentType = customer.getProperty("agent.type"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(agentType)) {
						customer.setAgentType(agentType);
					}

					String doctorAgentId = customer.getProperty("doctor.agent.id"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(doctorAgentId)) {
						customer.setDoctorAgentId(doctorAgentId);
					}

					String doctorDepartmentId = customer.getProperty("doctor.department.id"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(doctorDepartmentId)) {
						customer.setDoctorDepartmentId(doctorDepartmentId);
					}

					if (customer.hasProperty("doctor.indor")) {
						boolean isIndorDoctor = POSUtil.getBoolean(customer.getProperty("doctor.indor"), false); //$NON-NLS-1$//$NON-NLS-2$
						customer.setIndorDoctor(isIndorDoctor);
					}

					if (customer.hasProperty("doctor.lab")) {
						boolean isLabDoctor = POSUtil.getBoolean(customer.getProperty("doctor.lab"), false); //$NON-NLS-1$//$NON-NLS-2$
						customer.setLabDoctor(isLabDoctor);
					}

					//					PosLog.info(getClass(), "Customer " + customer.getName() + " is updating from property to field: " + message);
					CustomerDAO.getInstance().saveOrUpdate(customer, session);

				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "Customer update completed successfully"); //$NON-NLS-1$
	}

	public void updateImageResourceFields() {
		int totalRows = rowCount(ImageResource.class);
		PosLog.info(UpdateDBTo418.class, "Found total ImageResource: " + totalRows); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<ImageResource> pagination = new Pagination<ImageResource>(count, increment);
				pagination.setNumRows(totalRows);
				findData(pagination, session, ImageResource.class, ImageResource.PROP_ID);

				List<ImageResource> imageResources = pagination.getDataList();
				for (ImageResource imageResource : imageResources) {

					String imageFileName = imageResource.getProperty("image_file_name"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(imageFileName)) {
						imageResource.setImageFileName(imageFileName);

						PosLog.info(getClass(), "ImageResource " + imageResource.getId()
								+ " is updating 'image_file_name' from property to field. ImageFileName: " + imageFileName);
						ImageResourceDAO.getInstance().saveOrUpdate(imageResource, session);
					}
				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "ImageResource update completed successfully"); //$NON-NLS-1$
	}

	public void updateOnlineStoreFields() {
		int totalRows = rowCount(OnlineStore.class);
		PosLog.info(UpdateDBTo418.class, "Found total OnlineStore: " + totalRows); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<OnlineStore> pagination = new Pagination<OnlineStore>(count, increment);
				pagination.setNumRows(totalRows);
				findData(pagination, session, OnlineStore.class, OnlineStore.PROP_ID);

				List<OnlineStore> onlineStores = pagination.getDataList();
				for (OnlineStore onlineStore : onlineStores) {

					if (onlineStore.hasProperty("'MARKED_AS_TEST_DATA'")) {
						boolean markTestData = POSUtil.getBoolean(onlineStore.getProperty("'MARKED_AS_TEST_DATA'"), false); //$NON-NLS-1$//$NON-NLS-2$
						onlineStore.setTestStore(markTestData);

						PosLog.info(getClass(), "OnlineStore " + onlineStore.getId()
								+ " is updating \"'MARKED_AS_TEST_DATA'\" from property to field. markTestData: " + markTestData);
						OnlineStoreDAO.getInstance().saveOrUpdate(onlineStore, session);
					}
				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "OnlineStore update completed successfully"); //$NON-NLS-1$
	}

	public void updateSiiopaCustomerFields() {
		int totalRows = rowCount(SiiopaCustomer.class);
		PosLog.info(UpdateDBTo418.class, "Found total SiiopaCustomer: " + totalRows); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<SiiopaCustomer> pagination = new Pagination<SiiopaCustomer>(count, increment);
				pagination.setNumRows(totalRows);
				findData(pagination, session, SiiopaCustomer.class, SiiopaCustomer.PROP_ID);

				List<SiiopaCustomer> siiopaCustomers = pagination.getDataList();
				for (SiiopaCustomer siiopaCustomer : siiopaCustomers) {

					if (siiopaCustomer.hasProperty("'MARKED_AS_TEST_DATA'")) {
						boolean markTestData = POSUtil.getBoolean(siiopaCustomer.getProperty("'MARKED_AS_TEST_DATA'"), false); //$NON-NLS-1$//$NON-NLS-2$
						siiopaCustomer.setTestStore(markTestData);

						PosLog.info(getClass(), "SiiopaCustomer " + siiopaCustomer.getId()
								+ " is updating \"'MARKED_AS_TEST_DATA'\" from property to field. markTestData: " + markTestData);
						SiiopaCustomerDAO.getInstance().saveOrUpdate(siiopaCustomer, session);
					}
				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "SiiopaCustomer update completed successfully"); //$NON-NLS-1$
	}

	public void updateOutletFields() {
		int totalRows = rowCount(Outlet.class);
		PosLog.info(UpdateDBTo418.class, "Found total Outlets: " + totalRows); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<Outlet> pagination = new Pagination<Outlet>(count, increment);
				pagination.setNumRows(totalRows);
				findData(pagination, session, Outlet.class, Outlet.PROP_ID);

				List<Outlet> outlets = pagination.getDataList();
				for (Outlet outlet : outlets) {

					String previewImageId = outlet.getProperty("report.image_preview"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(previewImageId)) {
						outlet.setPreviewImageId(previewImageId);
					}

					String previewReceiptImageId = outlet.getProperty("receipt.image_preview"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(previewReceiptImageId)) {
						outlet.setPreviewReceiptImageId(previewReceiptImageId);
					}

					String previewPharmacyImageId = outlet.getProperty("pharmacy.image_preview"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(previewPharmacyImageId)) {
						outlet.setPreviewPharmacyImageId(previewPharmacyImageId);
					}

					String storeLogoResourceId = outlet.getProperty(AppConstants.PROP_HEADER_LOGO_IMAGE_ID);
					if (StringUtils.isBlank(storeLogoResourceId)) {
						storeLogoResourceId = outlet.getProperty("ticket.header.logo.imageid");
					}

					if (StringUtils.isNotBlank(storeLogoResourceId)) {
						outlet.setLogoImageId(storeLogoResourceId);
					}

					//					PosLog.info(getClass(), "Outlet " + outlet.getName() + " is updating from property to field: " + message);
					OutletDAO.getInstance().saveOrUpdate(outlet, session);

				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "Outlet update completed successfully"); //$NON-NLS-1$
	}

	public void updateTestItemFields() {
		int totalRows = rowCount(TestItem.class);
		PosLog.info(UpdateDBTo418.class, "Found total TestItem: " + totalRows); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<TestItem> pagination = new Pagination<TestItem>(count, increment);
				pagination.setNumRows(totalRows);
				findData(pagination, session, TestItem.class, TestItem.PROP_ID);

				List<TestItem> testItems = pagination.getDataList();
				for (TestItem testItem : testItems) {

					if (testItem.hasProperty("visible")) {
						boolean isVisible = POSUtil.getBoolean(testItem.getProperty("visible"), true);
						testItem.setVisible(isVisible);

						//						PosLog.info(getClass(), "TestItem " + testItem.getName() + " is updating 'visible' from property to field. visible: " + isVisible);
						TestItemDAO.getInstance().saveOrUpdate(testItem, session);
					}
				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "TestItem update completed successfully"); //$NON-NLS-1$
	}

	public void updateTicketItemFields(int increment) {
		int totalRows = rowCount(TicketItem.class);
		PosLog.info(UpdateDBTo418.class, "Found total TicketItem: " + totalRows); //$NON-NLS-1$

		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<TicketItem> pagination = new Pagination<TicketItem>(count, increment);
				pagination.setNumRows(totalRows);
				findData(pagination, session, TicketItem.class, TicketItem.PROP_CREATE_DATE);

				List<TicketItem> ticketItems = pagination.getDataList();
				for (TicketItem ticketItem : ticketItems) {

					//UpdateDBTo417
					String labDoctorId = ticketItem.getLabDoctorId();
					if (StringUtils.isNotBlank(labDoctorId)) {
						Doctor doctor = DoctorDAO.getInstance().get(labDoctorId);
						ticketItem.setLabDoctor(doctor);
					}
					//UpdateDBTo417

					boolean isReportDelivered = POSUtil.getBoolean(ticketItem.getProperty("report.delivered"), false); //$NON-NLS-1$//$NON-NLS-2$
					ticketItem.setReportDelivered(isReportDelivered);

					boolean allowDrFeeBeforeLabwork = POSUtil.getBoolean(ticketItem.getProperty("allow.dr.fee.before.labwork"), false); //$NON-NLS-1$//$NON-NLS-2$

					ticketItem.setAllowDrFeeBeforeLabwork(allowDrFeeBeforeLabwork);
					TicketItemDAO.getInstance().saveOrUpdate(ticketItem, session);

				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "TicketItem update completed successfully"); //$NON-NLS-1$
	}

	public void updateUserFields() {
		int totalRows = rowCount(User.class);
		PosLog.info(UpdateDBTo418.class, "Found total User: " + totalRows); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<User> pagination = new Pagination<User>(count, increment);
				pagination.setNumRows(totalRows);
				findData(pagination, session, User.class, User.PROP_ID);

				List<User> users = pagination.getDataList();
				for (User user : users) {

					StringBuilder sb = new StringBuilder();

					if (user.hasProperty("user.lab_stuff")) {
						boolean isLabStuff = POSUtil.getBoolean(user.getProperty("user.lab_stuff"), false); //$NON-NLS-1$//$NON-NLS-2$
						user.setLabStuff(isLabStuff);

						sb.append("LabStuff: " + isLabStuff);
						sb.append(", ");
					}

					if (user.hasProperty("user.accounts_manager")) {
						boolean isAccountsManager = POSUtil.getBoolean(user.getProperty("user.accounts_manager"), false); //$NON-NLS-1$//$NON-NLS-2$
						user.setAccountsManager(isAccountsManager);

						sb.append("AccountsManager: " + isAccountsManager);
						sb.append(".");
					}

					if (user.hasProperty("user.expense_officer")) {
						boolean isExpenseOfficer = POSUtil.getBoolean(user.getProperty("user.expense_officer"), false); //$NON-NLS-1$//$NON-NLS-2$
						user.setExpenseOfficer(isExpenseOfficer);

						sb.append("ExpenseOfficer: " + isExpenseOfficer);
						sb.append(".");
					}

					if (user.hasProperty("user.accounts_owner")) {
						boolean isAccountsOwner = POSUtil.getBoolean(user.getProperty("user.accounts_owner"), false); //$NON-NLS-1$//$NON-NLS-2$
						user.setAccountsOwner(isAccountsOwner);

						sb.append("AccountsOwner: " + isAccountsOwner);
						sb.append(".");
					}

					String message = sb.toString();
					if (StringUtils.isNotBlank(message)) {
						PosLog.info(getClass(), "User " + user.getFullName() + " is updating from property to field: " + message);
						UserDAO.getInstance().saveOrUpdate(user, session);
					}

				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "User update completed successfully"); //$NON-NLS-1$
	}

	public void updateTransactionFields(int increment) {

		int totalTransactions = rowCount(PosTransaction.class);
		PosLog.info(UpdateDBTo418.class, "Found total transactions: " + totalTransactions); //$NON-NLS-1$

		int count = 0;
		while (count < totalTransactions) {
			try (Session session = sessionFactory.openSession()) {
				Transaction transaction = session.beginTransaction();

				Pagination<PosTransaction> pagination = new Pagination<PosTransaction>(count, increment);
				pagination.setNumRows(totalTransactions);
				findPosTransactions(pagination, session);

				List<PosTransaction> posTransactions = pagination.getDataList();
				for (PosTransaction posTransaction : posTransactions) {

					//UpdateDBTo416 Start
					String accountManagerId = posTransaction.getProperty("user.AMId"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(accountManagerId)) {
						posTransaction.setAccountManagerId(accountManagerId);

						if ("1".equals(accountManagerId)) {
							posTransaction.setSourceType(ExpenseTransaction.EXPENSE_FROM_STORE);
						}
						else {
							posTransaction.setSourceType(ExpenseTransaction.EXPENSE_FROM_ACM);
						}
					}

					posTransaction.setBankAccountId(posTransaction.getProperty("bank_account_id"));

					posTransaction.setVendorId(posTransaction.getProperty("vendor_id"));

					posTransaction.putBalanceUpdateTransId(posTransaction.getProperty("source_id"));
					posTransaction.setSourceId(null);
					//UpdateDBTo416 Start

					posTransaction.setTransTicketIds(posTransaction.getProperty("trans.ticket_ids"));

					posTransaction.setConfirmPayment(POSUtil.getBoolean(posTransaction.getProperty("confirm.payment"), false));

					//PosLog.info(getClass(), "PosTransaction " + posTransaction.getId() + " is updating from property to field: " + message);
					PosTransactionDAO.getInstance().saveOrUpdate(posTransaction, session);

				}
				transaction.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "PosTransaction update completed successfully"); //$NON-NLS-1$
	}

	private void findPosTransactions(Pagination<PosTransaction> pagination, Session session) {
		Criteria criteria = createCriteria(session, PosTransaction.class);
		criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));
		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());

		PosLog.info(getClass(),
				String.format("Searching from: %s to %s", pagination.getCurrentRowIndex(), (pagination.getCurrentRowIndex() + pagination.getPageSize()))); //$NON-NLS-1$

		pagination.setRows(criteria.list());

	}

	public void updateBalanceUpdateTransactionFields() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.setProjection(Projections.rowCount());

			int totalTransactions = ((Long) criteria.uniqueResult()).intValue();

			PosLog.info(UpdateDBTo416.class, "Found total balance update transactions: " + totalTransactions); //$NON-NLS-1$

			criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.setMaxResults(totalTransactions);

			List<BalanceUpdateTransaction> posTransactions = criteria.list();

			Transaction transaction = session.beginTransaction();
			for (BalanceUpdateTransaction balanceUpdateTransaction : posTransactions) {

				String memoNo = balanceUpdateTransaction.getProperty("memo_no");
				String bankAccountId = balanceUpdateTransaction.getProperty("bank_account_id");

				if (StringUtils.isBlank(memoNo) && StringUtils.isBlank(bankAccountId)) {
					continue;
				}

				balanceUpdateTransaction.setMemoNo(memoNo);
				balanceUpdateTransaction.setBankAccountId(bankAccountId);

				BalanceUpdateTransactionDAO.getInstance().saveOrUpdate(balanceUpdateTransaction, session);
				PosLog.info(getClass(), "Updated balance update transaction " + balanceUpdateTransaction.getId()); //$NON-NLS-1$

			}
			transaction.commit();
		}

		PosLog.info(getClass(), "BalanceUpdateTransaction update completed successfully"); //$NON-NLS-1$
	}

	public void updateAdmissions() {

		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = session.createCriteria(BookingInfo.class);
			criteria.setProjection(Projections.rowCount());

			int totalRecords = ((Long) criteria.uniqueResult()).intValue();

			PosLog.info(getClass(), "Found total admissions: " + totalRecords); //$NON-NLS-1$

			criteria = session.createCriteria(BookingInfo.class);
			criteria.setMaxResults(totalRecords);

			List<BookingInfo> admissions = criteria.list();

			Transaction transaction = session.beginTransaction();
			for (BookingInfo admission : admissions) {

				admission.setCustomer(admission.getCustomer());
				if (admission.getDoctorId() != null) {
					admission.setDoctor(DoctorDAO.getInstance().get(admission.getDoctorId()));
				}

				BookingInfoDAO.getInstance().saveOrUpdate(admission, session);

			}
			transaction.commit();
		}

		PosLog.info(getClass(), "Admissions update completed successfully"); //$NON-NLS-1$
	}

	public void updateTicketFields(int increment) {
		int totalRows = rowCount(Ticket.class);
		PosLog.info(UpdateDBTo418.class, "Found total Ticket: " + totalRows); //$NON-NLS-1$

		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<Ticket> pagination = new Pagination<Ticket>(count, increment);
				pagination.setNumRows(totalRows);
				findData(pagination, session, Ticket.class, Ticket.PROP_CREATE_DATE);

				List<Ticket> tickets = pagination.getDataList();
				for (Ticket ticket : tickets) {

					String shipmentStatus = ticket.getProperty("shipmentStatus"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(shipmentStatus)) {
						ticket.setShipmentStatus(shipmentStatus);

						PosLog.info(getClass(),
								"Ticket " + ticket.getId() + " is updating 'shipmentStatus' from property to field. shipmentStatus: " + shipmentStatus);
						TicketDAO.getInstance().saveOrUpdate(ticket, session);
					}
				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), "Challan update completed successfully"); //$NON-NLS-1$
	}

	private void findData(Pagination pagination, Session session, Class clazz, String sortField) {
		Criteria criteria = createCriteria(session, clazz);
		criteria.addOrder(Order.asc(sortField));

		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());

		PosLog.info(getClass(), String.format("Searching " + clazz.getSimpleName() + " from: %s to %s out of %s", pagination.getCurrentRowIndex(), //$NON-NLS-1$
				(pagination.getCurrentRowIndex() + pagination.getPageSize()), pagination.getNumRows()));

		pagination.setRows(criteria.list());

	}

	public int rowCount(Class clazz) {
		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = createCriteria(session, clazz);
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}

	private Criteria createCriteria(Session session, Class clazz) {
		Criteria criteria = session.createCriteria(clazz);
		return criteria;
	}

}