/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.BalanceUpdateTransactionDAO;
import com.floreantpos.model.util.DataProvider;

public class UpdateDBTo443 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo443(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void update() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		int totalRows = rowCount();
		PosLog.info(UpdateDBTo416.class, "Found total balance update transaction: " + totalRows); //$NON-NLS-1$

		int increment = 10;
		int count = 0;
		while (count < totalRows) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<BalanceUpdateTransaction> pagination = new Pagination<BalanceUpdateTransaction>(count, increment);
				pagination.setNumRows(totalRows);
				findBalanceUpdateTransactions(pagination, session);

				List<BalanceUpdateTransaction> transactions = pagination.getDataList();
				for (Iterator iterator = transactions.iterator(); iterator.hasNext();) {
					BalanceUpdateTransaction balanceUpdateTransaction = (BalanceUpdateTransaction) iterator.next();

					StringBuilder sb = new StringBuilder();

					boolean hasPerformer = balanceUpdateTransaction.hasProperty("perform.by");
					if (hasPerformer) {
						String performerId = balanceUpdateTransaction.getProperty("perform.by", "");

						sb.append("Id: " + balanceUpdateTransaction.getId());
						sb.append(", ");
						sb.append("performerId: " + performerId);
						
						balanceUpdateTransaction.setPerformerId(performerId);

						String performerName = balanceUpdateTransaction.getPerformerName();
						if (StringUtils.isBlank(performerName)) {
							User performer = DataProvider.get().getUserById(performerId, balanceUpdateTransaction.getOutletId());
							if (performer != null) {
								balanceUpdateTransaction.putPerformerName(performer.getFullName());
								
								sb.append(", ");
								sb.append("performerName: " + performerName);
							}
						}

						sb.append(".");

						BalanceUpdateTransactionDAO.getInstance().saveOrUpdate(balanceUpdateTransaction, session);

						PosLog.info(getClass(), "BalanceUpdateTransaction (" + sb.toString() + ") is updating from field to property: ");

					}

				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}

	private void findBalanceUpdateTransactions(Pagination<BalanceUpdateTransaction> pagination, Session session) {
		Criteria criteria = createCommonCriteria(session);
		criteria.addOrder(Order.asc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());

		PosLog.info(getClass(),
				String.format("Searching from: %s to %s", pagination.getCurrentRowIndex(), (pagination.getCurrentRowIndex() + pagination.getPageSize()))); //$NON-NLS-1$

		pagination.setRows(criteria.list());

	}

	public int rowCount() {
		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = createCommonCriteria(session);
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}

	private Criteria createCommonCriteria(Session session) {
		Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
		criteria.add(Restrictions.ilike(BalanceUpdateTransaction.PROP_EXTRA_PROPERTIES, "\"perform.by\"", MatchMode.ANYWHERE)); //$NON-NLS-1$
		return criteria;
	}

}