/**
 * ************************************************************************

 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.beans.Transient;
import java.io.StringReader;
import java.lang.reflect.InvocationTargetException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonObject;
import javax.json.JsonReader;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.SerializationUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.main.Application;
import com.floreantpos.model.base.BaseTicket;
import com.floreantpos.model.dao.ActionHistoryDAO;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.PosTransactionDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.model.util.BookingUtil;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.model.util.ReferralCommissionType;
import com.floreantpos.model.util.ServiceChargeType;
import com.floreantpos.model.util.pricecalc.DataUtilCalcFactory;
import com.floreantpos.model.util.pricecalc.TicketCalcFactory;
import com.floreantpos.model.util.pricecalc.TicketItemCalcFactory;
import com.floreantpos.payment.common.Payable;
import com.floreantpos.util.JsonUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.floreantpos.util.ShiftUtil;
import com.floreantpos.util.TicketIdGenerator;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.orocube.common.util.TicketStatus;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "properties", "customer", "tables", "terminal", "createDateFormatted", "title", "outlet", "customerName",
		"customerEmail", "customerMobileNo", "bartabName", "gratuityAmount", "serviceChargePaidAmount", "serviceChargeDueAmount", "owner", "cashier", "shift",
		"customer", "department", "salesArea", "assignedDriver", "voidedBy", "terminal", "beverageCount", "beverageItemCount", "ticketStatus" })
@XmlSeeAlso({ PosTransaction.class, TicketItem.class })
@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement(name = "ticket")
public class Ticket extends BaseTicket implements Payable, TimedModel, PropertyContainer {
	public static final String CUSTOMER_TAX_EXEMPT = "customer.taxExempt"; //$NON-NLS-1$

	private static final long serialVersionUID = 1L;

	public static final String JSON_PROP_TICKET_DISC_AMOUNT = "ticketDiscAmount"; //$NON-NLS-1$
	public static final String JSON_PROP_ITEM_DISC_AMOUNT = "itemDiscAmount"; //$NON-NLS-1$

	public final static String PROP_TRANSACTIONS = "transactions"; //$NON-NLS-1$
	public final static String PROP_TICKET_ITEMS = "ticketItems"; //$NON-NLS-1$

	public final static String PROPERTY_ONLINE_ID = "onlineOrderId"; //$NON-NLS-1$
	public final static String PROPERTY_CARD_TRANSACTION_ID = "card_transaction_id"; //$NON-NLS-1$
	public final static String PROPERTY_CARD_TRACKS = "card_tracks"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_NAME = "card_name"; //$NON-NLS-1$
	public static final String PROPERTY_PAYMENT_METHOD = "payment_method"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_READER = "card_reader"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_NUMBER = "card_number"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_EXP_YEAR = "card_exp_year"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_EXP_MONTH = "card_exp_month"; //$NON-NLS-1$
	public static final String PROPERTY_ADVANCE_PAYMENT = "advance_payment"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_AUTH_CODE = "card_auth_code"; //$NON-NLS-1$
	public static final String PROPERTY_SUB_ORDER_TYPE = "ticket.sub_order_type"; //$NON-NLS-1$
	public static final String PROPERTY_LOYALTY_ADDED = "loyalty"; //$NON-NLS-1$
	public static final String PROPERTY_EXTRA_SEATS = "extra_seats"; //$NON-NLS-1$
	public static final String PROPERTY_OUTLET_SC_RATE = "storeScRate"; //$NON-NLS-1$
	private static final String PROPERTY_OUTLET_GRATUITY_RATE = "storeGratuityRate"; //$NON-NLS-1$

	public static final String CUSTOMER_PHONE = "CUSTOMER_MOBILE"; //$NON-NLS-1$
	public static final String CUSTOMER_NAME = "CUSTOMER_NAME"; //$NON-NLS-1$
	public static final String CUSTOMER_EMAIL = "CUSTOMER_EMAIL"; //$NON-NLS-1$
	public static final String CUSTOMER_AGE = "CUSTOMER_AGE"; //$NON-NLS-1$
	public static final String CUSTOMER_ADDRESS = "CUSTOMER_ADDRESS"; //$NON-NLS-1$
	public static final String JSON_PROP_BARTAB_NAME = "BARTAB_NAME"; //$NON-NLS-1$
	public static final String CUSTOMER_LAST_NAME = "CUSTOMER_LAST_NAME"; //$NON-NLS-1$
	public static final String CUSTOMER_ID = "CUSTOMER_ID"; //$NON-NLS-1$
	public static final String CUSTOMER_ZIP_CODE = "CUSTOMER_ZIP_CODE"; //$NON-NLS-1$
	public static final String MANAGER_INSTRUCTION = "MANAGER_INSTRUCTION"; //$NON-NLS-1$
	public static final String PHONE_EXTENSION = "PHONE_EXTENSION"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_ADDRESS = "CUSTOMER_ADDRESS"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_COUNTRY = "CUSTOMER_COUNTRY"; //$NON-NLS-1$

	public static final String CUSTOMER_TOTAL_LOYALTY_POINT = "CUSTOMER_LOYALTY_POINT"; //$NON-NLS-1$

	public static final String DRIVER_OUT_TIME = "OUT_AT"; //$NON-NLS-1$

	public static final String PROPERTY_DISCOUNT = "DISCOUNT"; //$NON-NLS-1$

	public static final int ORDER_PENDING = 0;
	public static final int ORDER_VERIFIED = 1;
	public static final int ORDER_FULLY_INVOICED = 2;
	public static final int ORDER_PARTIALLY_INVOICED = 3;
	public static final int ORDER_FULLY_SHIPMENT = 4;
	public static final int ORDER_PARTIALLY_SHIPMENT = 5;
	public static final int ORDER_PARTIALLY_SHIPMENT_AND_INVOICED = 6;
	public static final int ORDER_CLOSED = 7;
	public static final int ORDER_CANCELLED = 8;
	public static final String ADMISSION_CHARGE = "Admission charge"; //$NON-NLS-1$
	private transient Map<String, ActionHistory> events = new LinkedHashMap<>();
	private transient BookingInfo bookingInfo;

	//@formatter:off
	public static final String[] ORDER_STATUS = { "Pending", //$NON-NLS-1$
			"Verified", //$NON-NLS-1$
			"Fully Invoiced", //$NON-NLS-1$
			"Partially Invoiced", //$NON-NLS-1$
			"Fully Shipped", //$NON-NLS-1$
			"Partially Shipped", //$NON-NLS-1$
			"Partially Shipped and Invoiced", //$NON-NLS-1$
			"Closed", //$NON-NLS-1$
			"Cancelled" //$NON-NLS-1$
	};
	//@formatter:on

	public static final String ORDER_VERIFIED_BY = "Verified_By"; //$NON-NLS-1$
	public static final String ORDER_SENT_BY = "Sent_By"; //$NON-NLS-1$
	public static final String ORDER_SHIPMENT_BY = "Shipping_By"; //$NON-NLS-1$
	public static final String ORDER_INVOICED_BY = "Invoiced_By"; //$NON-NLS-1$
	public static final String ORDER_CLOSED_BY = "Closed_By"; //$NON-NLS-1$

	public static final String DEBIT = "DEBIT"; //$NON-NLS-1$
	public static final String CREDIT = "CREDIT"; //$NON-NLS-1$

	public static final String SPLIT = "split"; //$NON-NLS-1$
	public static final String SPLIT_TICKET_ID = "split_ticket_id"; //$NON-NLS-1$
	public static final String SPLIT_SEAT_NUMBER = "split_seat_number"; //$NON-NLS-1$
	public static final String SPLIT_TYPE = "SPLIT_TYPE"; //$NON-NLS-1$
	public static final int SPLIT_EQUALLY = 0;
	public static final int SPLIT_BY_SEAT = 1;
	public static final int SPLIT_MANUALLY = 2;

	/**
	 * This is the total discount amount applied to this ticket, not to any item.
	 * That is, this total of TicketDiscount, not TicketItemDiscount.
	 */
	private List deletedItems;
	private String sortOrder;
	private transient Customer customer;
	private transient Agent referrer;
	private transient Outlet outlet;
	private transient Department department;
	private transient SalesArea salesArea;
	private List<TicketDiscount> discounts;
	private List<Integer> tableNumbers;
	private boolean shouldUpdateTableStatus;
	private transient boolean shouldUpdateStock;

	private transient com.google.gson.JsonObject propertiesContainer;
	private boolean shouldPublishMqtt = true;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	private transient Project project;

	private static final String jSON_PROP_TABLE_NAMES = "tableNames"; //$NON-NLS-1$

	public static final String PROP_ID = "id"; //$NON-NLS-1$
	public static final String PROP_OUTLET_ID = "outletId"; //$NON-NLS-1$

	public Ticket() {
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	public Ticket(String ticketId, String outletId) {
		super(ticketId, outletId);
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	public Ticket(boolean populateDefaultProperties) {
		super();
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
		if (populateDefaultProperties) {
			populateDefaultProperties();
		}
	}

	public TicketType getTicketType() {
		return TicketType.getByTypeNo(super.getType());
	}

	public void setTicketType(TicketType type) {
		if (type != null) {
			super.setType(type.getTypeNo());
		}
	}

	@Override
	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	@Override
	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	@Override
	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	@Override
	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public void setCreateDate(Date createDate) {
		super.setCreateDate(createDate);
		super.setActiveDate(createDate);
	}

	@Override
	public List<TicketItem> getTicketItems() {
		List<TicketItem> items = super.getTicketItems();

		if (items == null) {
			items = new ArrayList<TicketItem>();
			super.setTicketItems(items);
		}
		return items;
	}

	@XmlElement(name = "transactionList")
	@Override
	public Set<PosTransaction> getTransactions() {
		Set<PosTransaction> items = super.getTransactions();

		if (items == null) {
			items = new HashSet<PosTransaction>();
			super.setTransactions(items);
		}
		return items;
	}

	@Override
	public Integer getNumberOfGuests() {
		Integer guests = super.getNumberOfGuests();
		if (guests == null || guests.intValue() == 0) {
			return Integer.valueOf(1);
		}
		return guests;
	}

	public String getCreateDateFormatted() {
		return DateUtil.formatDateWithTime(getCreateDate());
	}

	public String getTitle() {
		String title = ""; //$NON-NLS-1$
		if (getId() != null) {
			title += "#" + getId(); //$NON-NLS-1$
		}
		title += "" + ": " + getOwner(); //$NON-NLS-1$ //$NON-NLS-2$
		title += "" + ":" + getCreateDateFormatted(); //$NON-NLS-1$ //$NON-NLS-2$
		title += "" + ": " + NumberUtil.formatNumber(getTotalAmountWithTips()); //$NON-NLS-1$ //$NON-NLS-2$

		return title;
	}

	@XmlTransient
	@JsonIgnore
	public int getBeverageCount() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null)
			return 0;

		int count = 0;
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isBeverage()) {
				count += ticketItem.getQuantity();
			}
		}
		return count;
	}

	public void calculatePrice() {
		if (getTicketType() != null && getTicketType() == TicketType.IPD && !isClosed() && !isBookingDischarged()) {
			BookingUtil.calculateBedPrice(this);
		}
		TicketCalcFactory.getCalc(this).calculatePrice(this);
	}

	public double getRepricableAmount() {
		double repricableAmount = getTotalAmount() - getDeliveryCharge() - getFeeAmount();
		if (!isDiscountOnSerivceCharge()) {
			repricableAmount -= getServiceCharge();
		}
		return repricableAmount;
	}

	public double getToleranceAmount() {
		if (hasProperty("toleranceAmount")) { //$NON-NLS-1$
			return POSUtil.parseDouble(getProperty("toleranceAmount")); //$NON-NLS-1$
		}
		return 0;
	}

	public void setToleranceAmount(double toleranceAmount) {
		addProperty("toleranceAmount", String.valueOf(toleranceAmount)); //$NON-NLS-1$
	}

	public double getToleranceFactor() {
		if (hasProperty("toleranceFactor")) { //$NON-NLS-1$
			return POSUtil.parseDouble(getProperty("toleranceFactor")); //$NON-NLS-1$
		}
		Currency mainCurrency = DataProvider.get().getMainCurrency();
		if (mainCurrency != null) {
			return mainCurrency.getTolerance();
		}
		return 0;
	}

	public void setToleranceFactor(double tf) {
		addProperty("toleranceFactor", String.valueOf(tf)); //$NON-NLS-1$
	}

	public double getSubtotalAmountWithVoidItems() {
		return getSubtotalAmount() + getVoidSubtotal();
	}

	public static TicketDiscount convertToTicketDiscount(Discount discount, Ticket ticket) {
		return TicketCalcFactory.getCalc(ticket).convertToTicketDiscount(discount, ticket);
	}

	public static TicketDiscount buildLoyaltyDiscount(Ticket ticket) {
		return TicketCalcFactory.getCalc(ticket).buildLoyaltyDiscount(ticket);
	}

	public void removeLoyaltyDiscount() {
		List<TicketDiscount> discounts = getDiscounts();
		for (Iterator<TicketDiscount> iterator = discounts.iterator(); iterator.hasNext();) {
			TicketDiscount ticketDiscount = (TicketDiscount) iterator.next();
			if (ticketDiscount.getType().equals(Discount.DISCOUNT_TYPE_LOYALTY)) {
				iterator.remove();
				ActionHistoryDAO.addDiscountRemovedActionHistory(this, ticketDiscount);
				break;
			}
		}
	}

	public double calculateDiscountFromType(TicketDiscount coupon, double subtotal) {
		return TicketCalcFactory.getCalc(this).calculateDiscountFromType(this, coupon, subtotal);
	}

	public void addDeletedItems(Object o) {
		if (deletedItems == null) {
			deletedItems = new ArrayList();
		}

		deletedItems.add(o);
	}

	public List getDeletedItems() {
		return deletedItems;
	}

	public void clearDeletedItems() {
		if (deletedItems != null) {
			deletedItems.clear();
		}

		deletedItems = null;
	}

	public int countItem(TicketItem ticketItem) {
		List<TicketItem> items = getTicketItems();
		if (items == null) {
			return 0;
		}

		int count = 0;
		for (TicketItem ticketItem2 : items) {
			if (ticketItem.getMenuItemId().equals(ticketItem2.getMenuItemId())) {
				++count;
			}
		}

		return count;
	}

	public boolean needsKitchenPrint() {
		if (getDeletedItems() != null && getDeletedItems().size() > 0) {
			return true;
		}

		List<TicketItem> ticketItems = getTicketItems();
		for (TicketItem item : ticketItems) {
			if (item.isShouldPrintToKitchen() && !item.isPrintedToKitchen()) {
				return true;
			}
		}

		return false;
	}

	public boolean isCoreAmountPaid() {
		if (getPaidAmount() < getTotalAmount()) {
			return false;
		}
		return true;
	}

	@Override
	public void addProperty(String name, String value) {
		this.initPropertiesContainer();
		if (StringUtils.isEmpty(value)) {
			propertiesContainer.remove(name);
			return;
		}

		propertiesContainer.addProperty(name, value);
		super.setExtraProperties(propertiesContainer.toString());
	}

	@Override
	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	@Override
	public String getProperty(String key) {
		return this.getProperty(key, null);
	}

	private void initPropertiesContainer() {
		if (this.propertiesContainer == null) {
			String properties = super.getExtraProperties();
			this.propertiesContainer = StringUtils.isBlank(properties) ? new com.google.gson.JsonObject()
					: new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		}
	}

	@Override
	public String getProperty(String key, String defaultValue) {
		this.initPropertiesContainer();
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}

		return defaultValue;
	}

	@Override
	public void removeProperty(String propertyName) {
		this.initPropertiesContainer();
		propertiesContainer.remove(propertyName);
		super.setExtraProperties(propertiesContainer.toString());
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);
		return POSUtil.getBoolean(property);
	}

	public String toURLForm() {
		String s = "ticket_id=" + getId(); //$NON-NLS-1$

		List<TicketItem> items = getTicketItems();
		if (items == null || items.size() == 0) {
			return s;
		}

		for (int i = 0; i < items.size(); i++) {
			TicketItem ticketItem = items.get(i);
			s += "&items[" + i + "][id]=" + ticketItem.getId(); //$NON-NLS-1$ //$NON-NLS-2$
			s += "&items[" + i + "][name]=" + POSUtil.encodeURLString(ticketItem.getName()); //$NON-NLS-1$ //$NON-NLS-2$
			s += "&items[" + i + "][price]=" + ticketItem.getSubtotalAmount(); //$NON-NLS-1$ //$NON-NLS-2$
		}

		s += "&tax=" + getTaxAmount(); //$NON-NLS-1$
		s += "&subtotal=" + getSubtotalAmount(); //$NON-NLS-1$
		s += "&grandtotal=" + getTotalAmountWithTips(); //$NON-NLS-1$

		return s;
	}

	public void updateCustomer(Customer customer) {
		this.customer = customer;
	}

	public void setCustomer(Customer customer) {
		String oldCustomerId = this.getCustomerId();

		this.customer = customer;
		removeCustomer();
		if (customer == null) {
			return;
		}

		setCustomerId(customer.getId());
		addProperty(Ticket.CUSTOMER_ID, String.valueOf(customer.getId()));
		addProperty(Ticket.CUSTOMER_NAME, customer.getFirstName());
		addProperty(Ticket.CUSTOMER_LAST_NAME, customer.getLastName());
		addProperty(Ticket.CUSTOMER_PHONE, customer.getMobileNo());
		addProperty(Ticket.CUSTOMER_ZIP_CODE, customer.getZipCode());
		addProperty(Ticket.CUSTOMER_TAX_EXEMPT, customer.isTaxExempt().toString());
		addProperty(Ticket.CUSTOMER_EMAIL, customer.getEmail());

		setTaxExempt(customer.isTaxExempt());

		patientJournalLog(customer, oldCustomerId);
	}

	/**
	 * To be used in menugreat. If a customer does not want to register then this method to be called.
	 * 
	 * @param customer
	 */
	public void setCustomerInfo(String name, String email, String phone) {
		if (StringUtils.isNotBlank(name)) {
			name = name.trim();

			int firstSpaceIndex = name.indexOf(" "); //$NON-NLS-1$
			String firstName = name;
			String lastName = ""; //$NON-NLS-1$

			if (firstSpaceIndex > 0) {
				firstName = name.substring(0, firstSpaceIndex);
				lastName = name.substring(firstSpaceIndex);
			}
			addProperty(Ticket.CUSTOMER_NAME, firstName);
			addProperty(Ticket.CUSTOMER_LAST_NAME, lastName);
		}

		addProperty(Ticket.CUSTOMER_EMAIL, email);
		if (StringUtils.isBlank(phone)) {
			removeProperty(CUSTOMER_PHONE);
		}
		else {
			addProperty(Ticket.CUSTOMER_PHONE, phone);
		}
	}

	public void updateCustomerRef(Customer customer) {
		this.customer = customer;
	}

	@XmlTransient
	public Customer getCustomer() {
		if (this.customer != null) {
			return this.customer;
		}
		String customerId = getCustomerId();
		if (StringUtils.isEmpty(customerId)) {
			return null;
		}
		customer = DataProvider.get().getCustomer(customerId);
		return customer;
	}

	public void removeCustomer() {
		removeProperty(Ticket.CUSTOMER_ID);
		removeProperty(Ticket.CUSTOMER_NAME);
		removeProperty(Ticket.CUSTOMER_LAST_NAME);
		removeProperty(Ticket.CUSTOMER_PHONE);
		removeProperty(Ticket.CUSTOMER_ZIP_CODE);
		removeProperty(Ticket.CUSTOMER_TAX_EXEMPT);
		removeProperty(Ticket.CUSTOMER_EMAIL);
	}

	public String getSortOrder() {
		if (sortOrder == null) {
			return ""; //$NON-NLS-1$
		}
		return sortOrder;
	}

	public void setSortOrder(String sortOrder) {
		this.sortOrder = sortOrder;
	}

	public void setTicketStatus(TicketStatus ticketStatus) {
		super.setStatus(ticketStatus.name());
		String statusUpdateTimeKey = "delivery." + ticketStatus.name() + ".time"; //$NON-NLS-1$ //$NON-NLS-2$
		if (hasProperty(statusUpdateTimeKey)) {
			return;
		}
		addProperty(statusUpdateTimeKey, DateUtil.formatDateWithDefaultTimeAndSec(new Date()));
	}

	public Date getTicketStatusUpdatedTime(TicketStatus ticketStatus) {
		String statusUpdateTimeKey = "delivery." + ticketStatus.name() + ".time"; //$NON-NLS-1$ //$NON-NLS-2$
		if (!hasProperty(statusUpdateTimeKey)) {
			return null;
		}
		try {
			return DateUtil.formatDateWithDefaultTimeAndSec(getProperty(statusUpdateTimeKey));
		} catch (Exception e) {
		}
		return null;
	}

	@XmlTransient
	public TicketStatus getTicketStatus() {
		try {
			String status = super.getStatus();
			if (StringUtils.isBlank(status)) {
				return TicketStatus.Unknown;
			}

			return TicketStatus.valueOf(status);
		} catch (Exception e) {
			return TicketStatus.Unknown;
		}
	}

	public String getOrderStatusDisplay() {
		return ORDER_STATUS[getOrderStatus()];
	}

	public void setOrderStatusDisplay(String orderStatusDisplay) {
		//this.orderStatusDisplay = orderStatusDisplay;
	}

	public void consolidateTicketItems() {
		List<TicketItem> ticketItems = getTicketItems();

		Map<String, List<TicketItem>> itemMap = new LinkedHashMap<String, List<TicketItem>>();

		for (Iterator iterator = ticketItems.iterator(); iterator.hasNext();) {
			TicketItem newItem = (TicketItem) iterator.next();
			String menuItemId = newItem.getMenuItemId() == null ? newItem.getName() : newItem.getMenuItemId();
			if (StringUtils.isEmpty(menuItemId)) {
				continue;
			}
			List<TicketItem> itemListInMap = itemMap.get(menuItemId);

			if (itemListInMap == null) {
				List<TicketItem> list = new ArrayList<TicketItem>();
				list.add(newItem);

				itemMap.put(menuItemId.toString(), list);
			}
			else {
				boolean merged = false;
				for (TicketItem itemInMap : itemListInMap) {
					if (itemInMap.isMergable(newItem, false)) {
						itemInMap.merge(newItem);
						merged = true;
						break;
					}
				}

				if (!merged) {
					itemListInMap.add(newItem);
				}
			}
		}

		getTicketItems().clear();
		Collection<List<TicketItem>> values = itemMap.values();
		for (List<TicketItem> list : values) {
			if (list != null) {
				getTicketItems().addAll(list);
			}
		}

		calculatePrice();
	}

	public void consolidateKitchenTicketItems() {
		boolean isFilterKitchenPrintedItems = true;
		consolidateKitchenTicketItems(isFilterKitchenPrintedItems);
	}

	public void consolidateKitchenTicketItems(boolean isFilterTicket) {
		consolidateKitchenTicketItems(isFilterTicket, null);
	}

	public void consolidateKitchenTicketItems(boolean isFilterTicket, List<TicketItem> ticketItems) {
		if (isSourceOnline()) {
			return;
		}
		if (ticketItems == null) {
			ticketItems = getTicketItems();
		}
		Map<String, List<TicketItem>> itemMap = new LinkedHashMap<String, List<TicketItem>>();

		for (Iterator iterator = ticketItems.iterator(); iterator.hasNext();) {
			TicketItem newItem = (TicketItem) iterator.next();
			if ((!newItem.isShouldPrintToKitchen() && !newItem.isPrintKitchenSticker())) {
				continue;
			}
			if (isFilterTicket && newItem.isPrintedToKitchen()) {
				continue;
			}
			String menuItemId = newItem.getMenuItemId();
			if (StringUtils.isEmpty(menuItemId)) {
				menuItemId = newItem.getName();
			}
			List<TicketItem> itemListInMap = itemMap.get(menuItemId);

			if (itemListInMap == null) {
				List<TicketItem> list = new ArrayList<TicketItem>();
				list.add(newItem);

				itemMap.put(menuItemId.toString(), list);
			}
			else {
				itemListInMap.add(newItem);
			}
		}

		ticketItems.clear();
		Collection<List<TicketItem>> values = itemMap.values();
		for (List<TicketItem> list : values) {
			if (list != null) {
				ticketItems.addAll(list);
			}
		}

		calculatePrice();
	}

	/**
	 * Mark ticket items, modifiers, add-ons as printed to kitchen
	 */
	public void markPrintedToKitchen() {
		List<TicketItem> ticketItems = getTicketItems();
		markPrintedToKitchen(ticketItems);
	}

	public void markPrintedToKitchen(List<TicketItem> ticketItems) {
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isPrintedToKitchen() || !ticketItem.isShouldPrintToKitchen()) {
				continue;
			}

			List<Printer> printers = ticketItem.getPrinters(getOrderType());
			if (printers == null) {
				continue;
			}
			ticketItem.setPrintedToKitchen(true);
			ticketItem.setKitchenStatusValue(KitchenStatus.WAITING);
			List<TicketItemModifier> ticketItemModifiers = ticketItem.getTicketItemModifiers();
			if (ticketItemModifiers != null) {
				for (TicketItemModifier itemModifier : ticketItemModifiers) {
					itemModifier.setPrintedToKitchen(true);
				}
			}
			List<TicketItemCookingInstruction> cookingInstructions = ticketItem.getCookingInstructions();
			if (cookingInstructions != null) {
				for (TicketItemCookingInstruction ticketItemCookingInstruction : cookingInstructions) {
					ticketItemCookingInstruction.setPrintedToKitchen(true);
				}
				ticketItem.buildCoookingInstructions();
			}
		}
	}

	public TicketSource getTicketSource() {
		return TicketSource.fromName(super.getSource());
	}

	public void setTicketSource(TicketSource ticketSource) {
		super.setSource(ticketSource.name());
	}

	public boolean isSourceOnline() {
		return getTicketSource() == TicketSource.Online;
	}

	public boolean isReservation() {
		return getTicketSource() == TicketSource.Reservation;
	}

	public static Ticket clone(Ticket source) {
		return (Ticket) SerializationUtils.clone(source);
	}

	/**
	 * If clone fails return null else return New ticket.
	 * @param ticket
	 * @param newOutlet
	 * 
	 * @return New ticket
	 */
	public Ticket clone(Ticket ticket, Outlet newOutlet) {
		try {
			Ticket newTicket = new Ticket();
			PropertyUtils.copyProperties(newTicket, ticket);
			newTicket.setOutlet(newOutlet);
			newTicket.setId(TicketIdGenerator.generateTicketId(newTicket));
			newTicket.setCreateDate(StoreDAO.getServerTimestamp());

			//TicketItem
			List<TicketItem> newTicketItems = new ArrayList<TicketItem>();
			for (TicketItem ticketItem : ticket.getTicketItems()) {

				TicketItem newTicketItem = new TicketItem();
				if (ticketItem instanceof ModifiableTicketItem) {
					newTicketItem = new ModifiableTicketItem();
				}
				PropertyUtils.copyProperties(newTicketItem, ticketItem);
				newTicketItem.setId(null);
				newTicketItem.setCreateDate(StoreDAO.getServerTimestamp());
				newTicketItem.setTicket(newTicket);
				if (ticketItem.isHasModifiers()) {
					List<TicketItemModifier> newTicketItemModifiers = new ArrayList<TicketItemModifier>();
					for (TicketItemModifier ticketItemModifier : ticketItem.getTicketItemModifiers()) {
						TicketItemModifier newTicketItemModi = new TicketItemModifier();
						PropertyUtils.copyProperties(newTicketItemModi, ticketItemModifier);
						newTicketItemModi.setId(null);
						newTicketItemModi.setTicketItem(newTicketItem);
						newTicketItemModifiers.add(newTicketItemModi);
					}
					newTicketItem.setTicketItemModifiers(newTicketItemModifiers);
				}
				newTicketItems.add(newTicketItem);
			}
			newTicket.setTicketItems(newTicketItems);

			//Transaction
			Set<PosTransaction> transactionList = ticket.getTransactions();
			if (transactionList != null && transactionList.size() > 0) {
				ArrayList<PosTransaction> newTransactionList = new ArrayList<>();
				ArrayList<PosTransaction> transactions = new ArrayList<>(transactionList);
				for (PosTransaction posTransaction : transactions) {
					PosTransaction newPosTransaction = new PosTransaction();
					PropertyUtils.copyProperties(newPosTransaction, posTransaction);
					newPosTransaction.setId(null);
					newPosTransaction.setOutletId(newOutlet.getId());
					newTransactionList.add(newPosTransaction);
				}
				newTicket.setTransactions(transactionList);
			}

			return newTicket;
		} catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
			return null;
		}
	}

	public String getNumberToDisplay() {
		return Messages.getString("Ticket.11") + getTokenNo(); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public Outlet getOutlet() {
		if (this.outlet != null) {
			return this.outlet;
		}
		String outletId = getOutletId();
		if (outletId == null)
			return null;
		outlet = (Outlet) DataProvider.get().getObjectOf(Outlet.class, outletId);
		return outlet;
	}

	@Transient
	public void setOutlet(com.floreantpos.model.Outlet outlet) {
		this.outlet = outlet;
		if (outlet == null) {
			setOutletId(null);
		}
		else {
			setOutletId(outlet.getId());
		}
	}

	@XmlTransient
	public Department getDepartment() {
		String departmentId = getDepartmentId();
		if (departmentId == null)
			return null;
		if (department == null)
			department = (Department) DataProvider.get().getObjectOf(Department.class, departmentId);
		return department;
	}

	public void setDepartment(com.floreantpos.model.Department department) {
		this.department = department;
		if (department == null) {
			setDepartmentId(null);
		}
		else {
			setDepartmentId(department.getId());
		}
	}

	@XmlTransient
	public SalesArea getSalesArea() {
		if (this.salesArea != null) {
			return this.salesArea;
		}
		String salesAreaId = getSalesAreaId();
		if (salesAreaId == null)
			return null;
		salesArea = DataProvider.get().getSalesArea(salesAreaId);
		return salesArea;
	}

	public void setSalesArea(com.floreantpos.model.SalesArea salesArea) {
		this.salesArea = salesArea;
		if (salesArea == null) {
			setSalesAreaId(null);
		}
		else {
			setSalesAreaId(salesArea.getId());
		}
	}

	public boolean isSplited() {
		return Boolean.valueOf(getProperty(Ticket.SPLIT));
	}

	//	public void updateTicketItemPriceByOrderType(String orderTypeOption) {
	//		List<TicketItem> ticketItems = getTicketItems();
	//		OrderType orderType = getOrderType();
	//		TaxGroup taxGroup = null;
	//
	//		if (ticketItems == null) {
	//			return;
	//		}
	//		if (orderTypeOption.equals(SubOrderType.FOR_HERE.getName())) {
	//			taxGroup = orderType.getForHereTaxGroup();
	//			addProperty(Ticket.PROPERTY_SUB_ORDER_TYPE, SubOrderType.FOR_HERE.getName());
	//		}
	//		else if (orderTypeOption.equals(SubOrderType.TO_GO.getName())) {
	//			taxGroup = orderType.getToGoTaxGroup();
	//			addProperty(Ticket.PROPERTY_SUB_ORDER_TYPE, SubOrderType.TO_GO.getName());
	//		}
	//
	//		for (TicketItem ticketItem : ticketItems) {
	//			MenuItem.setItemTaxes(ticketItem, taxGroup, getOrderType());
	//		}
	//		calculatePrice();
	//	}

	public String getSplitId() {
		return getProperty(Ticket.SPLIT_TICKET_ID);
	}

	public void setSplitOrder(int order) {
		addProperty("SPLIT_ORDER", String.valueOf(order)); //$NON-NLS-1$
	}

	@JsonIgnore
	public String getDiffWithCrntTime() {
		if (getCreateDate() == null) {
			return null;
		}
		return DateUtil.getElapsedTime(getCreateDate(), new Date());
	}

	public void setDiffWithCrntTime(String diffWithCrntTime) {
	}

	public String getTableNames() {
		return getProperty(jSON_PROP_TABLE_NAMES);
	}

	@XmlTransient
	public String getBartabName() {
		if (getOrderType() != null && getOrderType().isBarTab()) {
			String barTabName = getProperty(Ticket.JSON_PROP_BARTAB_NAME);
			if (StringUtils.isBlank(barTabName)) {
				barTabName = getProperty(Ticket.CUSTOMER_NAME);
			}
			if (StringUtils.isNotBlank(barTabName)) {
				return barTabName;
			}
		}
		return null;
	}

	public String getTableNameDisplay() {
		String tableNames = getTableNames();
		if (StringUtils.isBlank(tableNames)) {
			return POSUtil.toFormattedString(getTableNumbers());
		}
		return tableNames;
	}

	public String getTableNameOrNumberDisplay() {
		Store store = DataProvider.get().getStore();
		if (store.isShowTableNameOnTable()) {
			return this.getTableNameDisplay();
		}
		else {
			return this.getTableNumbersDisplay();
		}
	}

	public Double getTotalAmountWithTips() {
		return super.getTotalAmount();
	}

	public void calculateRefundAmount() {
		double refundAmount = 0;
		double voidAmount = 0;
		Set<PosTransaction> transactions = getTransactions();
		if (transactions != null) {
			for (PosTransaction t : transactions) {
				if (t instanceof RefundTransaction && !t.isVoided()) {
					refundAmount += t.getAmount();
				}
			}
		}
		setRefundAmount(refundAmount + voidAmount);
	}

	public String getTableNumbersDisplay() {
		List<Integer> numbers = getTableNumbers();
		if (numbers == null || numbers.size() <= 0) {
			return ""; //$NON-NLS-1$
		}
		String tableNumbers = numbers.toString().replaceAll("[\\[\\]]", ""); //$NON-NLS-1$ //$NON-NLS-2$
		return tableNumbers;
	}

	public boolean isPrintedToKitchenOrInventoryAdjusted() {
		for (TicketItem ticketItem : getTicketItems()) {
			if (ticketItem.isPrintedToKitchen() || ticketItem.isInventoryAdjusted() || (!ticketItem.isReturned() && ticketItem.isVoided())) {
				return true;
			}
			if (ticketItem.isHasModifiers()) {
				for (TicketItemModifier modifier : ticketItem.getTicketItemModifiers()) {
					if (modifier.isPrintedToKitchen())
						return true;
				}
			}
		}
		return false;
	}

	public boolean hasRepriceDiscount() {
		List<TicketDiscount> discounts = getDiscounts();
		if (discounts == null || discounts.isEmpty()) {
			return false;
		}

		for (TicketDiscount ticketDiscount : discounts) {
			if (ticketDiscount.getOriginalType() == Discount.DISCOUNT_TYPE_REPRICE) {
				return true;
			}
		}

		return false;
	}

	public double getVoidSubtotal() {
		if (hasProperty("voidSubtotal")) { //$NON-NLS-1$
			return POSUtil.parseDouble("voidSubtotal"); //$NON-NLS-1$
		}
		return 0.0;
	}

	public void setVoidSubtotal(double voidSubtotal) {
		addProperty("voidSubtotal", String.valueOf(voidSubtotal)); //$NON-NLS-1$
	}

	public double getVoidTotal() {
		if (hasProperty("voidTotal")) { //$NON-NLS-1$
			return POSUtil.parseDouble("voidTotal"); //$NON-NLS-1$
		}
		return 0.0;
	}

	public void setVoidTotal(double voidTotal) {
		addProperty("voidTotal", String.valueOf(voidTotal)); //$NON-NLS-1$
	}

	@Override
	public void setTotalAmount(Double totalAmount) {
		super.setTotalAmount(totalAmount + 0.0);
	}

	@Override
	public void setDueAmount(Double dueAmount) {
		super.setDueAmount(dueAmount + 0.0);
	}

	@Override
	public void setDiscountAmount(Double discountAmount) {
		super.setDiscountAmount(discountAmount + 0.0);
	}

	@Override
	public void setTaxAmount(Double taxAmount) {
		super.setTaxAmount(taxAmount + 0.0);
	}

	@XmlTransient
	public OrderType getOrderType() {
		return DataProvider.get().getOrderTypeById(getOrderTypeId(), getOutletId());
	}

	public void setOrderType(OrderType orderType) {
		if (orderType != null) {
			setServiceChargeApplicable(orderType.isServiceChargeApplicable());
			setServiceChargeType(orderType.getServiceChargeType());
			setOutletServiceChargeRate(orderType.getServiceChargeRate());
			super.setOrderTypeId(orderType.getId());
		}
		else {
			try {
				throw new PosException("Order type is null!");
			} catch (Exception e) {
				PosLog.error(getClass(), e);
			}
		}
	}

	@XmlTransient
	public Shift getShift() {
		return DataProvider.get().getShiftById(getShiftId(), getOutletId());
	}

	public void setShift(Shift shift) {
		String shiftId = null;
		if (shift != null) {
			shiftId = shift.getId();
		}
		super.setShiftId(shiftId);
	}

	@XmlTransient
	public User getOwner() {
		return DataProvider.get().getUserById(getOwnerId(), getOutletId());
	}

	public void setOwner(User owner) {
		String ownerId = null;
		String ownerTypeId = null;
		if (owner != null) {
			ownerId = owner.getId();
			ownerTypeId = owner.getUserTypeId();

			if (getCreatedByUserId() == null) {
				setCreatedByUserId(ownerId);
			}

			addProperty("owner.firstname", owner.getFirstName()); //$NON-NLS-1$
			addProperty("owner.lastname", owner.getLastName()); //$NON-NLS-1$
		}
		super.setOwnerId(ownerId);
		super.setOwnerTypeId(ownerTypeId);
	}

	@XmlTransient
	public User getCashier() {
		return DataProvider.get().getUserById(getCashierId(), getOutletId());
	}

	public void setCashier(User cashier) {
		String cashierId = null;
		if (cashier != null) {
			cashierId = cashier.getId();
		}
		super.setCashierId(cashierId);
	}

	@XmlTransient
	public User getAssignedDriver() {
		return DataProvider.get().getUserById(getAssignedDriverId(), getOutletId());
	}

	public void setAssignedDriver(User assignedDriver) {
		String assignedDriverId = null;
		if (assignedDriver != null) {
			assignedDriverId = assignedDriver.getId();
			addProperty("driver.firstname", assignedDriver.getFirstName()); //$NON-NLS-1$
			addProperty("driver.lastname", assignedDriver.getLastName()); //$NON-NLS-1$
		}
		else {
			removeProperty("driver.firstname"); //$NON-NLS-1$
			removeProperty("driver.lastname"); //$NON-NLS-1$
		}
		super.setAssignedDriverId(assignedDriverId);
	}

	@XmlTransient
	public String getDriverNameFromProperty() {
		String firstName = getProperty("driver.firstname"); //$NON-NLS-1$
		String lastName = getProperty("driver.lastname"); //$NON-NLS-1$

		return firstName + " " + lastName; //$NON-NLS-1$
	}

	@XmlTransient
	public User getVoidedBy() {
		return DataProvider.get().getUserById(getVoidedById(), getOutletId());
	}

	public void setVoidedBy(User voidBy) {
		String voidById = null;
		if (voidBy != null) {
			voidById = voidBy.getId();
		}
		super.setVoidedById(voidById);
	}

	@XmlTransient
	public Terminal getTerminal() {
		return DataProvider.get().getTerminalById(getTerminalId(), getOutletId());
	}

	public void setTerminal(Terminal terminal) {
		Integer terminalId = null;
		if (terminal != null) {
			terminalId = terminal.getId();
		}
		super.setTerminalId(terminalId);
	}

	/*
	 * Discount part build as json and put in properties
	 */
	public void setDiscounts(List<TicketDiscount> discounts) {
		this.discounts = discounts;
		if (this.discounts == null) {
			this.discounts = new ArrayList<>();
		}
	}

	public List<TicketDiscount> getDiscounts() {
		if (discounts == null) {
			discounts = new ArrayList<TicketDiscount>();
			convertDiscountPropertyToList();
		}
		return discounts;
	}

	public void convertDiscountPropertyToList() {
		String property = super.getDiscountsProperty();
		if (StringUtils.isNotEmpty(property)) {
			JsonReader jsonParser = Json.createReader(new StringReader(property));
			JsonArray jsonArray = jsonParser.readArray();
			jsonParser.close();
			for (int i = 0; i < jsonArray.size(); i++) {
				JsonObject dObj = (JsonObject) jsonArray.get(i);
				TicketDiscount td = new TicketDiscount();
				td.setTicket(this);
				td.setId(JsonUtil.getString(dObj, TicketDiscount.PROP_ID));
				td.setLoyaltyPoint(JsonUtil.getInt(dObj, TicketDiscount.PROP_LOYALTY_POINT));
				td.setLoyaltyCharged(JsonUtil.getBoolean(dObj, TicketDiscount.PROP_LOYALTY_CHARGED));
				td.setDiscountId(JsonUtil.getString(dObj, TicketDiscount.PROP_DISCOUNT_ID));
				td.setName(JsonUtil.getString(dObj, TicketDiscount.PROP_NAME));
				td.setType(JsonUtil.getInt(dObj, TicketDiscount.PROP_TYPE));
				td.setAutoApply(JsonUtil.getBoolean(dObj, TicketDiscount.PROP_AUTO_APPLY));
				td.setCouponQuantity(JsonUtil.getDouble(dObj, TicketDiscount.PROP_COUPON_QUANTITY));
				td.setMinimumAmount(JsonUtil.getDouble(dObj, TicketDiscount.PROP_MINIMUM_AMOUNT));
				td.setValue(JsonUtil.getDouble(dObj, TicketDiscount.PROP_VALUE));
				td.setTotalDiscountAmount(JsonUtil.getDouble(dObj, TicketDiscount.PROP_TOTAL_DISCOUNT_AMOUNT));
				td.setCouponBarcode(JsonUtil.getString(dObj, "couponBarcode")); //$NON-NLS-1$
				Integer originalType = JsonUtil.getInt(dObj, TicketDiscount.PROP_ORIGINAL_TYPE);
				if (originalType == null) {
					originalType = td.getType();
				}
				td.setOriginalType(originalType);
				Double originalValue = JsonUtil.getDouble(dObj, TicketDiscount.PROP_ORIGINAL_VALUE);
				if (originalValue == null) {
					originalValue = td.getValue();
				}
				td.setOriginalValue(originalValue);
				td.putDiscountGivenByUserId(JsonUtil.getString(dObj, TicketDiscount.JSON_PROP_GIVEN_BY_USER_ID));
				td.putDiscountGivenByUserName(JsonUtil.getString(dObj, TicketDiscount.JSON_PROP_GIVEN_BY_USER_NAME));
				discounts.add(td);
			}
		}
	}

	public void addTodiscounts(TicketDiscount ticketDiscount) {
		discounts = getDiscounts();
		discounts.add(ticketDiscount);
		buildDiscounts();
		ActionHistoryDAO.addDiscountAddedActionHistory(this, ticketDiscount);
	}

	public void buildDiscounts() {
		JSONArray jsonArray = new JSONArray();
		for (TicketDiscount ticketDiscount : discounts) {
			jsonArray.put(ticketDiscount.toJson());
		}
		setDiscountsProperty(jsonArray.toString());
	}

	public String getAppliedCouponNumberDisplay() {
		if (getDiscountAmount() == 0 || getDiscounts() == null || getDiscounts().isEmpty()) {
			return ""; //$NON-NLS-1$
		}
		TicketDiscount ticketDiscount = getDiscounts().get(0);
		String couponBarcode = ticketDiscount.getCouponBarcode();
		if (StringUtils.isBlank(couponBarcode)) {
			couponBarcode = ticketDiscount.getNameDisplay();
		}
		return " (" + couponBarcode + ")"; //$NON-NLS-1$ //$NON-NLS-2$
	}

	//	@XmlTransient
	//	public Map<String, String> getProperties() {
	//		Map<String, String> properties = super.getProperties();
	//		if (properties == null) {
	//			properties = new HashMap<>();
	//			super.setProperties(properties);
	//		}
	//		return properties;
	//	}

	@Override
	public String getTicketTableNumbers() {
		String ticketTableNumbers = ""; //$NON-NLS-1$
		List<Integer> tableNumbers = getTableNumbers();
		if (tableNumbers != null && tableNumbers.size() > 0) {
			for (Iterator iterator = tableNumbers.iterator(); iterator.hasNext();) {
				Integer tableNo = (Integer) iterator.next();
				ticketTableNumbers += tableNo;
				if (iterator.hasNext())
					ticketTableNumbers += ","; //$NON-NLS-1$
			}
		}
		return ticketTableNumbers;
	}

	@Override
	public void setTicketTableNumbers(String ticketTableNumbers) {
		tableNumbers = new ArrayList<Integer>();
		if (StringUtils.isNotEmpty(ticketTableNumbers)) {
			String[] tableNums = ticketTableNumbers.split(","); //$NON-NLS-1$
			for (String tbl : tableNums) {
				int tableNo = POSUtil.parseInteger(tbl);
				if (tableNo > 0)
					tableNumbers.add(tableNo);
			}
		}
		super.setTicketTableNumbers(ticketTableNumbers);
	}

	public List<Integer> getTableNumbers() {
		return tableNumbers;
	}

	public void voidItem(TicketItem ticketItem, String voidReason, boolean itemWasted) {
		voidItem(ticketItem, voidReason, itemWasted, ticketItem.getQuantity());
	}

	public void voidItem(TicketItem ticketItem, String voidReason, boolean itemWasted, double quantity) {
		TicketCalcFactory.getCalc(this).voidItem(this, ticketItem, voidReason, itemWasted, quantity);
	}

	public void voidReturnedItem(TicketItem voidTicketItem, String voidReason, boolean itemWasted) {
		TicketCalcFactory.getCalc(this).voidReturnedItem(voidTicketItem, voidReason, itemWasted);
	}

	public TicketItem undoVoidItem(TicketItem voidedTicketItem) {
		return TicketCalcFactory.getCalc(this).undoVoidItem(this, voidedTicketItem);
	}

	public TicketItem getTicketItem(String ticketItemId) {
		return getTicketItem(ticketItemId, false);
	}

	public TicketItem getTicketItem(String ticketItemId, boolean isIncludeComboChild) {
		if (StringUtils.isBlank(ticketItemId)) {
			return null;
		}
		for (TicketItem ticketItem : getTicketItems()) {
			if (ticketItem.getId() == null) {
				continue;
			}
			if (ticketItem.getId().equals(ticketItemId)) {
				return ticketItem;
			}

			if (isIncludeComboChild) {
				if (ticketItem instanceof ComboTicketItem) {
					ComboTicketItem comboTicketItem = (ComboTicketItem) ticketItem;
					List<TicketItem> comboItems = comboTicketItem.getComboItems();
					if (comboItems != null && !comboItems.isEmpty()) {
						for (TicketItem comboChild : comboItems) {
							if (comboChild.getId().equals(ticketItemId)) {
								return comboChild;
							}
						}
					}
				}
			}
		}
		return null;
	}

	private void populateDefaultProperties() {
		Terminal terminal = DataProvider.get().getCurrentTerminal();
		setTerminal(terminal);

		setTaxIncluded(Application.getInstance().isPriceIncludesTax());
		//		StoreSession storeSession = DataProvider.get().getStoreSession();
		//		if (storeSession != null) {
		//			setStoreSessionId(storeSession.getId());
		//		}

		Department department = terminal.getDepartment();
		if (department != null)
			setDepartmentId(department.getId());

		Outlet outlet = terminal.getOutlet();
		if (outlet != null) {
			setOutletId(outlet.getId());
			setOutletServiceChargeRate(outlet.getServiceChargePercentage());
			setOutletGratuityRate(outlet.getDefaultGratuityPercentage());
		}

		setOwner(DataProvider.get().getCurrentUser());
		setShift(ShiftUtil.getCurrentShift());
		setShouldIncludeInSales(true);
		Currency mainCurrency = DataProvider.get().getMainCurrency();
		setToleranceFactor(mainCurrency.getTolerance());

		//setApplyFloridaTaxRule(DataProvider.get().getStore().isEnableFloridaTaxRule());
		String action = ActionHistory.NEW_CHECK;
		ActionHistory actionHistory = ActionHistory.create(this, action, "", getOwner(), DataProvider.get().getCurrentOutletId()); //$NON-NLS-1$
		addToEvents(action, actionHistory);

		setDataVersion(2);
	}

	public boolean isShouldUpdateTableStatus() {
		return shouldUpdateTableStatus;
	}

	public void setShouldUpdateTableStatus(boolean isUpdateTable) {
		this.shouldUpdateTableStatus = isUpdateTable;
	}

	public PosTransaction getBartabTransaction() {
		String bartabTransactionId = getProperty("bartab.transaction.id"); //$NON-NLS-1$
		if (StringUtils.isNotEmpty(bartabTransactionId)) {
			for (PosTransaction transaction : getTransactions()) {
				if (!transaction.isVoided() && bartabTransactionId.equals(String.valueOf(transaction.getId()))) {
					return transaction;
				}
			}
		}

		return null;
	}

	@Override
	public String getExtraProperties() {
		this.initPropertiesContainer();
		return this.propertiesContainer.toString();
	}

	@Override
	public void setExtraProperties(String properties) {
		super.setExtraProperties(properties);
		this.propertiesContainer = null;
		this.initPropertiesContainer();
	}

	public void addExtraProperty(String key, String value) {
		this.initPropertiesContainer();
		propertiesContainer.addProperty(key, value);
		super.setExtraProperties(propertiesContainer.toString());
	}

	public String getExtraProperty(String key) {
		this.initPropertiesContainer();
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	//	@Override
	//	public String toString() {
	//		return "Ticket [ticketDiscountAmount=" + ", itemDiscountAmount=" + getItemDiscountAmount() //$NON-NLS-1$ //$NON-NLS-2$
	//				+ ", deletedItems=" + deletedItems + ", sortOrder=" + sortOrder //$NON-NLS-1$ //$NON-NLS-2$
	//				+ ", customer=" + customer + ", outlet=" + outlet + ", department=" + department + ", salesArea=" + salesArea + ", discounts=" + discounts //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
	//				+ ", tableNumbers=" + tableNumbers + ", tables=" + tables + ", shouldUpdateTableStatus=" + shouldUpdateTableStatus + ", roundedAmount=" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	//				+ getRoundedAmount() + ", toleranceAmount=" + getToleranceAmount() + ", getTicketItems()=" //$NON-NLS-1$//$NON-NLS-2$
	//				+ getTicketItems() + "]"; //$NON-NLS-1$
	//	}

	public boolean isShouldPublishMqtt() {
		return shouldPublishMqtt;
	}

	public void setShouldPublishMqtt(boolean shouldPublishMqtt) {
		this.shouldPublishMqtt = shouldPublishMqtt;
	}

	public String getOwnerName() {
		if (getOwner() != null) {
			return getOwner().getFullName();
		}
		return ""; //$NON-NLS-1$
	}

	@XmlTransient
	public String getOwnerNameFromProperty() {
		String firstName = getProperty("owner.firstname"); //$NON-NLS-1$
		String lastName = getProperty("owner.lastname"); //$NON-NLS-1$
		if (StringUtils.isBlank(lastName)) {
			return firstName;
		}

		return firstName + " " + lastName; //$NON-NLS-1$
	}

	public void setOwnerName(String ownerName) {
	}

	@XmlTransient
	public String getCustomerName() {
		if (this.customer != null) {
			return this.customer.getName();
		}
		String customerName = getProperty(CUSTOMER_NAME);
		if (propertiesContainer.has(CUSTOMER_LAST_NAME)) {
			customerName = customerName + " " + getProperty(CUSTOMER_LAST_NAME); //$NON-NLS-1$
		}
		return customerName;
	}

	public void setCustomerName(String customerName) {
		addProperty(CUSTOMER_NAME, customerName);
	}

	@XmlTransient
	public String getCustomerEmail() {
		if (getCustomer() != null) {
			return getCustomer().getEmail();
		}
		return getProperty(Ticket.CUSTOMER_EMAIL);
	}

	public void setCustomerEmail(String customerEmail) {
		addProperty(Ticket.CUSTOMER_EMAIL, customerEmail);
	}

	@XmlTransient
	public String getCustomerMobileNo() {
		String phone = getProperty(CUSTOMER_PHONE, "");//$NON-NLS-1$

		if (StringUtils.isBlank(phone) && getCustomer() != null) {
			return getCustomer().getMobileNo();
		}

		return phone;
	}

	public void setCustomerMobileNo(String customerMobileNo) {
		addProperty(CUSTOMER_PHONE, customerMobileNo);
	}

	public void makeXmlTransient() {
		//		setProperties(null);
		setDiscounts(null);
		for (TicketItem ticketItem : getTicketItems()) {
			ticketItem.setTicket(null);
		}
		if (getTransactions() != null) {
			for (PosTransaction transaction : getTransactions()) {
				transaction.setTicket(null);
			}
		}
	}

	public void mergeTicket(Ticket otherTicket) {
		TicketCalcFactory.getCalc(this).mergeTicket(this, otherTicket);
	}

	public boolean isPriceApplicable(TicketItem ticketItem) {
		if (ticketItem.isVoided() && !ticketItem.isItemReturned()) {
			return false;
		}
		return true;
	}

	public void setHasGiftCard(boolean hasGiftCard) {
		addProperty(AppConstants.HAS_GIFT_CARD, String.valueOf(hasGiftCard));
	}

	public boolean hasGiftCard() {
		return Boolean.valueOf(getProperty(AppConstants.HAS_GIFT_CARD));
	}

	public void setSubOrderType(SubOrderType subOrderType) {
		if (subOrderType != null) {
			addProperty(PROPERTY_SUB_ORDER_TYPE, subOrderType.getName());
		}
	}

	public SubOrderType getSubOrderType() {
		return SubOrderType.fromName(getExtraProperty(Ticket.PROPERTY_SUB_ORDER_TYPE));
	}

	public boolean isKitchenPrintable() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null) {
			return false;
		}
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isKitchenPrintable()) {
				return true;
			}
		}
		return false;
	}

	public boolean isDiscountable() {
		List<TicketItem> ticketItems = this.getTicketItems();
		if (ticketItems == null || ticketItems.isEmpty()) {
			return Boolean.FALSE;
		}

		return Boolean.TRUE;
	}

	public boolean isVoidable() {
		List<TicketItem> ticketItems = this.getTicketItems();
		if (ticketItems == null || ticketItems.isEmpty()) {
			return Boolean.FALSE;
		}
		return Boolean.TRUE;
	}

	public Map<String, ActionHistory> getEvents() {
		if (this.events == null) {
			this.events = new LinkedHashMap<>();
		}
		return this.events;
	}

	public void addToEvents(String key, ActionHistory history) {
		if (this.events == null) {
			this.events = new LinkedHashMap<>();
		}
		this.events.put(key, history);
	}

	public void addRandomEvent(String eventName, String description) {
		ActionHistory actionHistory = ActionHistory.create(this, eventName, description, DataProvider.get().getCurrentUser(),
				DataProvider.get().getCurrentOutletId());
		getEvents().put(actionHistory.getActionName() + Math.random(), actionHistory);
	}

	public boolean hasLoyaltyDiscount() {
		List<TicketDiscount> discounts = getDiscounts();
		if (discounts == null) {
			return false;
		}
		for (TicketDiscount ticketDiscount : discounts) {
			String discountId = ticketDiscount.getDiscountId();
			if (StringUtils.isNotBlank(discountId) && discountId.equals("loyalty_discount")) {//$NON-NLS-1$
				return true;
			}
		}
		return false;
	}

	public boolean hasFixedDiscount() {
		List<TicketDiscount> discounts = getDiscounts();
		return discounts != null && discounts.stream().anyMatch(d -> d.getType() == Discount.DISCOUNT_TYPE_AMOUNT);
	}

	@XmlTransient
	public int getBeverageItemCount() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null) {
			return 0;
		}

		int beverageCount = 0;
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isBeverage()) {
				beverageCount += Math.abs(ticketItem.getQuantity());
			}
		}

		return beverageCount;
	}

	public void validateTicketBeforeSave() {
		if (getTicketItems() == null || getTicketItems().size() == 0) {
			throw new PosException(POSConstants.TICKET_IS_EMPTY_);
		}

		//		if (orderType.isBeverageMandatory()) {
		//			Integer numberOfGuests = getNumberOfGuests();
		//			int beverageItemCount = getBeverageItemCount();
		//			if (beverageItemCount < numberOfGuests) {
		//				throw new PosException(String.format(Messages.getString("Ticket.2"), numberOfGuests)); //$NON-NLS-1$
		//			}
		//		}
	}

	public double getTaxAmountFromProperty() {
		return POSUtil.parseDouble(this.getProperty(AppConstants.TAX_EXEMPT_AMOUNT));
	}

	public String getCustomerNameByProperty() {
		String firstName = this.getProperty(Ticket.CUSTOMER_NAME) == null ? "" : this.getProperty(Ticket.CUSTOMER_NAME); //$NON-NLS-1$
		String lastName = this.getProperty(Ticket.CUSTOMER_LAST_NAME) == null ? "" : this.getProperty(Ticket.CUSTOMER_LAST_NAME); //$NON-NLS-1$
		if (StringUtils.isBlank(lastName)) {
			return firstName;
		}

		return firstName + " " + lastName; //$NON-NLS-1$
	}

	public Double getOutletServiceChargeRate() {
		String scRateStr = getProperty(PROPERTY_OUTLET_SC_RATE);
		if (StringUtils.isBlank(scRateStr)) {
			return 0.0;
		}
		try {
			return NumberUtil.parse(scRateStr).doubleValue();
		} catch (ParseException e) {
			//PosLog.error(getClass(), e);
		}
		return 0.0;
	}

	public void setOutletServiceChargeRate(Double rate) {
		addProperty(PROPERTY_OUTLET_SC_RATE, String.valueOf(rate));
	}

	public Double getOutletGratuityRate() {
		String gratuityRate = getProperty(PROPERTY_OUTLET_GRATUITY_RATE);
		if (StringUtils.isBlank(gratuityRate)) {
			return 0.0;
		}
		try {
			return NumberUtil.parse(gratuityRate).doubleValue();
		} catch (ParseException e) {
			PosLog.error(getClass(), e);
		}
		return 0.0;
	}

	public void setOutletGratuityRate(Double rate) {
		addProperty(PROPERTY_OUTLET_GRATUITY_RATE, String.valueOf(rate));
	}

	public Double getSubtotalWithoutIncludedTax() {
		if (isTaxIncluded()) {
			return getSubtotalAmount() - getTaxAmount();
		}
		return getSubtotalAmount();
	}

	public void closeIfApplicable() {
		//if (NumberUtil.isZero(getDueAmount())) {
		//	OrderType orderType = getOrderType();

		//	if (orderType != null && (orderType.isCloseOnPaid() || orderType.isBarTab())) {//fix
		//		setClosed(true);
		//		setClosingDate(StoreDAO.getServerTimestamp());
		//	}
		//}
	}

	public boolean isShouldUpdateStock() {
		return shouldUpdateStock;
	}

	public void setShouldUpdateStock(boolean shouldUpdateStock) {
		this.shouldUpdateStock = shouldUpdateStock;
	}

	@JsonIgnore
	public double getServiceChargePaidAmount() {
		Set<PosTransaction> transactions = getTransactions();
		if (transactions == null || transactions.isEmpty()) {
			return 0;
		}

		double scPaidAmount = 0.0;
		for (PosTransaction posTransaction : transactions) {
			if (posTransaction.isVoided() || (posTransaction instanceof RefundTransaction)) {
				continue;
			}
			scPaidAmount += posTransaction.getServiceChargeAmount();
		}

		return scPaidAmount;
	}

	@JsonIgnore
	public double getServiceChargeDueAmount() {
		return getServiceCharge() - getServiceChargePaidAmount();
	}

	//	public double getTotalDiscountPercentageRate() {
	//		List<TicketDiscount> discounts = getDiscounts();
	//		if (discounts == null || discounts.isEmpty()) {
	//			return 0.0;
	//		}
	//
	//		double percentage = 0.0;
	//		for (TicketDiscount ticketDiscount : discounts) {
	//			if (ticketDiscount.getType() == Discount.DISCOUNT_TYPE_LOYALTY || ticketDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
	//				Double totalAmount = getSubtotalAmount() - getItemDiscountAmount();
	//				Double discountValue = ticketDiscount.getValue() * ticketDiscount.getCouponQuantity();
	//				double discountPercent = (100.0 * discountValue) / totalAmount;
	//				percentage += discountPercent;
	//			}
	//			else {
	//				percentage += (ticketDiscount.getValue() * ticketDiscount.getCouponQuantity());
	//			}
	//		}
	//		if (percentage > 100) {
	//			percentage = 100;
	//		}
	//		return percentage / 100.0;
	//	}
	//
	//	public double getDiscountPercentageRate(TicketDiscount ticketDiscount, double ticketAmountAfterDiscount) {
	//		Double minimumAmount = ticketDiscount.getMinimumAmount();
	//		double subtotalAfterItemDiscount = getSubtotalAmount() - getItemDiscountAmount();
	//		if (minimumAmount > 0 && subtotalAfterItemDiscount < minimumAmount) {
	//			return 0;
	//		}
	//		double percentage = 0.0;
	//		if (ticketDiscount.getType() == Discount.DISCOUNT_TYPE_LOYALTY || ticketDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
	//			Double discountValue = ticketDiscount.getValue() * ticketDiscount.getCouponQuantity();
	//			double discountPercent = (100.0 * discountValue) / ticketAmountAfterDiscount;
	//			percentage += discountPercent;
	//		}
	//		else {
	//			percentage += (ticketDiscount.getValue() * ticketDiscount.getCouponQuantity());
	//		}
	//		if (percentage > 100) {
	//			percentage = 100;
	//		}
	//		return percentage / 100.0;
	//	}

	public void setItemDiscountAmount(double amount) {
		addProperty(JSON_PROP_ITEM_DISC_AMOUNT, String.valueOf(amount));
	}

	public double getItemDiscountAmount() {
		return getDoubleProperty(JSON_PROP_ITEM_DISC_AMOUNT);
	}

	public double getTicketDiscountAmount() {
		return getDoubleProperty(JSON_PROP_TICKET_DISC_AMOUNT);
	}

	public void setTicketDiscountAmount(double amount) {
		addProperty(JSON_PROP_TICKET_DISC_AMOUNT, String.valueOf(amount));
	}

	public double getTicketDiscountAmountWithoutReturnedItems() {
		return getDoubleProperty("tdawri");
	}

	public void setTicketDiscountAmountWithoutReturnedItems(double amount) {
		addProperty("tdawri", String.valueOf(amount));
	}

	public double getOldDataTolerance() {
		return getDoubleProperty("oldDataTolerance"); //$NON-NLS-1$
	}

	public void setOldDataTolerance(double amount) {
		addProperty("oldDataTolerance", String.valueOf(amount)); //$NON-NLS-1$
	}

	@Override
	public com.google.gson.JsonObject getPropertyStore() {
		return propertiesContainer;
	}

	public void setDiscountOnSerivceCharge(boolean value) {
		addProperty("discOnSc", String.valueOf(value)); //$NON-NLS-1$
	}

	public boolean isDiscountOnSerivceCharge() {
		return POSUtil.getBoolean(getProperty("discOnSc")); //$NON-NLS-1$
	}

	public void setServiceChargeApplicable(boolean value) {
		addProperty("scApplicable", String.valueOf(value)); //$NON-NLS-1$
	}

	public boolean isServiceChargeApplicable() {
		return POSUtil.getBoolean(getProperty("scApplicable")); //$NON-NLS-1$
	}

	public void setApplyFloridaTaxRule(boolean value) {
		addProperty("applyFlTaxRule", String.valueOf(value)); //$NON-NLS-1$
	}

	public boolean isApplyFloridaTaxRule() {
		return POSUtil.getBoolean(getProperty("applyFlTaxRule")); //$NON-NLS-1$
	}

	public boolean isItemReturnable(TicketItem originalItem, double toBeReturnedQuantity) {
		double totalItemCount = originalItem.getQuantity();
		double totalReturnedCount = 0;
		List<TicketItem> ticketItems = getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			String voidedItemId = ticketItem.getVoidedItemId();
			if (StringUtils.isNotBlank(voidedItemId) && voidedItemId.equals(originalItem.getId()) && ticketItem.isItemReturned()) {
				totalReturnedCount += ticketItem.getQuantity();
			}
		}
		return Math.abs(totalReturnedCount) + toBeReturnedQuantity <= totalItemCount;
	}

	public boolean hasReturnedItem() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				if (ticketItem.isItemReturned()) {
					return Boolean.TRUE;
				}
			}
		}
		return Boolean.FALSE;
	}

	public String getDeliveryZipCode() {
		return getProperty("deliAddressZipCode"); //$NON-NLS-1$
	}

	public void setDeliveryAddressZipCode(String addressDetail) {
		addProperty("deliAddressZipCode", addressDetail); //$NON-NLS-1$
	}

	public String getDeliveryFlatNo() {
		return getProperty("flatNo", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void setDeliveryFlatNo(String flatNo) {
		addProperty("flatNo", flatNo); //$NON-NLS-1$
	}

	public double getMinimumDeliveryAmount() {
		return getDoubleProperty(Store.PROP_DELIVERY_MINIMUM_AMOUNT);
	}

	public void setMinimumDeliveryAmount(double deliveryMinimumAmount) {
		addProperty(Store.PROP_DELIVERY_MINIMUM_AMOUNT, String.valueOf(deliveryMinimumAmount));
	}

	public String getStoreId() {
		return getProperty("storeId"); //$NON-NLS-1$
	}

	public void setStoreId(String storeId) {
		addProperty("storeId", storeId); //$NON-NLS-1$
	}

	public String getEstimatedDeliveryTime() {
		return getProperty("estimated_delivery_time"); //$NON-NLS-1$
	}

	public void setEstimatedDeliveryTime(String deliveryTimeInMin) {
		addProperty("estimated_delivery_time", deliveryTimeInMin); //$NON-NLS-1$
	}

	public boolean isDeliveryASAP() {
		return getBooleanProperty("delivery_asap", Boolean.FALSE); //$NON-NLS-1$
	}

	public void setDeliveryASAP(boolean delivery_asap) {
		addProperty("delivery_asap", String.valueOf(delivery_asap)); //$NON-NLS-1$
	}

	public void setServiceChargeType(ServiceChargeType serviceChargeType) {
		addProperty("serviceChargeType", serviceChargeType.name()); //$NON-NLS-1$
	}

	@JsonIgnore
	public ServiceChargeType getServiceChargeType() {
		String property = getProperty("serviceChargeType"); //$NON-NLS-1$
		if (StringUtils.isNotBlank(property)) {
			try {
				return ServiceChargeType.valueOf(property);
			} catch (Exception e) {
				return ServiceChargeType.EMPTY;
			}
		}
		return ServiceChargeType.EMPTY;
	}

	public void setOrginalDeliveryDate(Date orginalDeliveryDate) {
		if (orginalDeliveryDate == null) {
			removeProperty("OrginalDeliveryDate"); //$NON-NLS-1$
			return;
		}
		addProperty("OrginalDeliveryDate", DateUtil.formatDateWithDefaultTimeAndSec(orginalDeliveryDate)); //$NON-NLS-1$
	}

	@JsonIgnore
	public Date getOrginalDeliveryDate() {
		String orginalDate = getProperty("OrginalDeliveryDate"); //$NON-NLS-1$
		try {
			return DateUtil.formatDateWithDefaultTimeAndSec(orginalDate);
		} catch (ParseException e) {

		}
		return null;
	}

	public void checkVoidable() throws PosException {
		List<TicketItem> ticketItems = this.getTicketItems();
		if (ticketItems == null || ticketItems.isEmpty()) {
			throw new PosException(POSConstants.TICKET_IS_EMPTY_);
		}
		if (hasReturnedItem()) {
			throw new PosException(Messages.getString("POSConstants.27")); //$NON-NLS-1$
		}
		if (this.getBooleanProperty("split", false)) { //$NON-NLS-1$
			throw new PosException(Messages.getString("Ticket.4")); //$NON-NLS-1$
		}
		if (TicketType.OT == getTicketType() && OTStatus.COMPLETED == getOTRequestStatus()) {
			throw new PosException("This ticket cannot be voided because the OT status is already marked as completed.");
		}
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isGiftCard() && ticketItem.getGiftCardPaidAmount() > 0) {
				throw new PosException(Messages.getString("Ticket.3")); //$NON-NLS-1$
			}
		}
	}

	/**
	 * How many of the given menu item is added in the ticket?
	 * 
	 * @param menuItem
	 * @return Quantity of the menuitem that is addedd in the ticket.
	 */
	public double countMenuItemQuantity(MenuItem menuItem) {
		double qty = 0;
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems != null) {
			for (TicketItem item : ticketItems) {
				if (item.getMenuItemId().equals(menuItem.getId())) {
					qty += item.getQuantity();
				}
				else {
					if (item.isComboItem()) {
						List<TicketItem> comboItems = item.getComboItems();
						if (comboItems != null) {
							for (TicketItem comboTicketItem : comboItems) {
								if (comboTicketItem.getMenuItemId().equals(menuItem.getId())) {
									qty += comboTicketItem.getQuantity();
								}
							}
						}
					}
				}
			}
		}

		return qty;
	}

	public void setCustomer(Customer selectedCustomer, boolean updateItemsPrice) {
		setCustomer(selectedCustomer);
		if (updateItemsPrice) {
			if (getTicketItems().size() > 0) {
				for (TicketItem ticketItem : getTicketItems()) {
					if (StringUtils.isBlank(ticketItem.getUnitName())) {
						continue;
					}
					MenuItem menuItem = ticketItem.getMenuItem();
					if (menuItem == null) {
						continue;
					}
					if (ticketItem.isComboItem()) {
						TicketItemCalcFactory.getCalc(ticketItem).doCalculateComboItemPrice(ticketItem, true);
					}
					else {
						menuItem.setTicketItemUnitPriceAndCost(ticketItem, menuItem, DataProvider.get().getInventoryUnitById(ticketItem.getUnitName()), this);
					}
				}
			}
		}
	}

	public void removeDoctor() {
		setDoctorId(null);
		removeProperty("doctor.name"); //$NON-NLS-1$
		removeProperty("doctor.agent.id"); //$NON-NLS-1$
	}

	public void putDoctor(Doctor doctor) {
		String oldDoctorId = this.getDoctorId();
		if (doctor == null) {
			removeDoctor();
			return;
		}
		setDoctorId(doctor.getId());
		putDoctorName(doctor.getName());

		putDoctorAgentId(doctor.getDoctorAgentId());

		doctorJournalLog(doctor, oldDoctorId);
	}

	public String getDoctorName() {
		return getProperty("doctor.name", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorName(String doctorName) {
		addProperty("doctor.name", doctorName); //$NON-NLS-1$
	}

	public void putDoctorAgentId(String agentId) {
		addProperty("doctor.agent.id", agentId); //$NON-NLS-1$
	}

	public String getDoctorAgentId() {
		return getProperty("doctor.agent.id", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void removeAgent() {
		setReferrerId(null);
		removeProperty("agent.name"); //$NON-NLS-1$
		removeProperty("agent.salary"); //$NON-NLS-1$
		removeProperty("rf.rate_on_net_sales"); //$NON-NLS-1$
		removeProperty("rf.type_on_net_sales"); //$NON-NLS-1$
		removeProperty("ApplyDiscountOnReferral"); //$NON-NLS-1$

		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				ticketItem.removeProperty("rf.type_on_report"); //$NON-NLS-1$
				ticketItem.removeProperty("rf.rate_on_report"); //$NON-NLS-1$

				MenuItem menuItem = ticketItem.getMenuItem();
				if (menuItem == null || !menuItem.isLabTest()) {
					continue;
				}
				Double unitPrice = menuItem.getPrice();
				if (ticketItem.isUrgentDelivery()) {
					unitPrice = menuItem.getUrgentPrice();
				}
				if (!menuItem.isEditablePrice()) {
					ticketItem.setUnitPrice(unitPrice);
				}
			}
		}

	}

	public void putAgent(Customer customer) {
		if (customer == null) {
			removeAgent();
			return;
		}

		removeAgent();

		if (!(customer instanceof Agent)) {
			return;
		}

		Agent referrer = (Agent) customer;
		String oldReferrerId = this.getReferrerId();

		setReferrerId(referrer.getId());
		putAgentName(referrer.getName());

		boolean isPriceUpdateRequired = false;
		if (StringUtils.isNotBlank(oldReferrerId)) {
			Agent customerObject = (Agent) DataProvider.get().getObjectOf(Agent.class, oldReferrerId);
			isPriceUpdateRequired = !(referrer.getAgentType().equals(customerObject.getAgentType()));
		}
		else {
			if (AgentTypeEnum.B2B == AgentTypeEnum.fromString(referrer.getAgentType())) {
				isPriceUpdateRequired = true;
			}
		}

		putRfRateOnNetSales(referrer.getRfRateOnNetSales());
		putRfOnNetSalesType(referrer.getRfOnNetSalesType());

		if (referrer != null) {
			putApplyDiscountOnReferral(referrer.isApplyDiscountOnReferralCommission());
		}

		updateItemReferrerProperty(referrer, POSUtil.parseDouble(referrer.getRfRateOnReport()), isPriceUpdateRequired);

		referrerJournalLog(referrer, oldReferrerId);
	}

	private void referrerJournalLog(Customer referrer, String oldReferrerId) {
		if (this.getId() == null) {
			addRandomEvent(ActionHistory.REFERRER_ADDED, "Referrer id: " + referrer.getId());
			return;
		}

		String action = ActionHistory.REFERRER_CHANGED;
		if (oldReferrerId != null && referrer != null) {
			ActionHistoryDAO.addActionHistory(this, action, String.format("Old referrer : %s, New referrer : %s", oldReferrerId, referrer.getId())); //$NON-NLS-1$
		}
	}

	private void doctorJournalLog(Doctor doctor, String oldDoctorId) {
		if (this.getId() == null) {
			addRandomEvent(ActionHistory.DOCTOR_ADDED, "Doctor id: " + doctor.getId());
			return;
		}

		String action = "Changed doctor";
		if (oldDoctorId != null && doctor != null) {
			ActionHistoryDAO.addActionHistory(this, action, String.format("Old doctor : %s, New doctor : %s", oldDoctorId, doctor.getId())); //$NON-NLS-1$
		}
	}

	private void patientJournalLog(Customer customer, String oldCustomerId) {
		if (this.getId() == null) {
			addRandomEvent(ActionHistory.PATIENT_ADDED, "Patient id: " + customer.getId());
			return;
		}
		String action = ActionHistory.CUSTOMER_REMOVED;
		String customerId = null;
		if (customer != null) {
			action = "Changed customer";
			customerId = customer.getId();
		}
		if (customer.getId() != null) {
			ActionHistoryDAO.addActionHistory(this, action, String.format("Old patient : %s, New patient : %s", oldCustomerId, customerId)); //$NON-NLS-1$
		}
	}

	private void updateItemReferrerProperty(Customer referrer, double referrerRate, boolean isPriceUpdateRequired) {
		String commissionTypeName = referrer.getRfOnReportType();
		ReferralCommissionType commissionType = ReferralCommissionType.fromName(commissionTypeName);
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				ticketItem.putRfOnReportType(commissionTypeName);

				MenuItem menuItem = ticketItem.getMenuItem();
				if (menuItem == null) {
					continue;
				}

				if (commissionType == ReferralCommissionType.STANDARD) {
					double referrerCommissionRate = DataUtilCalcFactory.getCalc().calculateReferrerCommissionRate(ticketItem, commissionType, menuItem);
					ticketItem.putRfRateOnReport(referrerCommissionRate);
				}
				else {
					ticketItem.putRfRateOnReport(referrerRate);
				}

				AgentTypeEnum agentType = AgentTypeEnum.fromString(referrer.getAgentType());
				Double unitPrice = menuItem.getPrice();
				if (ticketItem.isUrgentDelivery()) {
					unitPrice = menuItem.getUrgentPrice();
				}
				if (AgentTypeEnum.B2B == agentType && ticketItem.isLabTest()) {
					unitPrice = menuItem.getB2bPrice();
					ticketItem.putRfOnReportType(ReferralCommissionType.GREATER_THAN_SALES_PRICE.name());
				}
				if (isPriceUpdateRequired && !menuItem.isEditablePrice() && ProductType.match(ticketItem.getProductType(), ProductType.PATHOLOGY)) {
					ticketItem.setUnitPrice(unitPrice);
				}
			}
		}
	}

	public void putAgentName(String agentName) {
		addProperty("agent.name", agentName); //$NON-NLS-1$
	}

	public String getAgentName() {
		return getProperty("agent.name", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putRfOnNetSalesType(String chargeType) {
		addProperty("rf.type_on_net_sales", chargeType); //$NON-NLS-1$
	}

	public String getRfOnNetSalesType() {
		return getProperty("rf.type_on_net_sales", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putRfRateOnNetSales(String commissionValue) {
		addProperty("rf.rate_on_net_sales", commissionValue); //$NON-NLS-1$
	}

	public String getRfRateOnNetSales() {
		return getProperty("rf.rate_on_net_sales", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public double getDiscountableSubtotal() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null) {
			return 0;
		}
		double s = 0;
		for (TicketItem ticketItem : ticketItems) {
			s += ticketItem.getDiscountableSubtotal();
		}
		return s;
	}

	public void putApplyDiscountOnReferral(boolean apply) {
		addProperty("ApplyDiscountOnReferral", String.valueOf(apply)); //$NON-NLS-1$
	}

	public boolean isApplyDiscountOnReferral() {
		return getBooleanProperty("ApplyDiscountOnReferral", true); //$NON-NLS-1$
	}

	public boolean isPatientRequiredOnOrder() {
		for (TicketItem ticketItem : getTicketItems()) {
			if (ticketItem.isLabTest()) {
				return true;
			}
			if (ticketItem.isPatientRequired()) {
				return true;
			}
		}
		return false;
	}

	public double getCommissionOnReport() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null) {
			return 0;
		}
		double s = 0;
		for (TicketItem ticketItem : ticketItems) {
			s += ticketItem.getExpectedReferrerFeeOnReport();
		}
		return s;
	}

	public void putImportDataSource(String source) {
		addProperty("import_data_source", source);
	}

	public String getImportDataSource() {
		return getProperty("import_data_source");
	}

	public void putRfExtraPayment(double amount) {
		addProperty("rf_extra_payment", String.valueOf(amount));
	}

	public double getRfExtraPayment() {
		return getDoubleProperty("rf_extra_payment");
	}

	public void putPrescriptionNo(String prescriptionNo) {
		addProperty("prescription_No", prescriptionNo);
	}

	public String getPrescriptionNo() {
		return getProperty("prescription_No", "");
	}

	public Double getPaidAmountAfterRefund() {
		return getPaidAmount() - getRefundAmount();
	}

	@Override
	public String getEntityId() {
		return getId();
	}

	@Override
	public double getSubTotal() {
		return getSubtotalAmount();
	}

	@Override
	public Double getDueValue() {
		return getDueAmount();
	}

	@Override
	public void setDueValue(double d) {
		setDueAmount(d);
		setPaidAmount(getSubtotalAmount() - d);
	}

	@Override
	public PosTransaction createTransaction() {
		return null;
	}

	@Override
	public PosTransaction createTransaction(PaymentType paymentType) {
		PosTransaction transaction = Payable.super.createTransaction(paymentType);
		transaction.setTransactionTime(StoreDAO.getServerTimestamp());
		transaction.setTicket(this);
		transaction.setServer(this.getOwner());
		transaction.setCustomerName(getCustomerName());
		return transaction;
	}

	public void putCalculateCommissionForGoods(boolean calculate) {
		addProperty("calculate_commission_for_goods", String.valueOf(calculate));
	}

	public boolean isCalculateCommissionForGoods() {
		return getBooleanProperty("calculate_commission_for_goods", false);
	}

	public TicketItem getAdmissionFeeTicketItem() {
		for (Iterator<TicketItem> iterator = getTicketItems().iterator(); iterator.hasNext();) {
			TicketItem ticketItem = (TicketItem) iterator.next();
			if (ticketItem.isComboItem() && ticketItem.getComboItems() != null && ticketItem.getComboItems().size() > 0) {
				List<TicketItem> comboTicketItems = ticketItem.getComboItems();
				for (TicketItem comboTicketItem : comboTicketItems) {
					if (!comboTicketItem.isVoided() && comboTicketItem.isAdmissionChargeItem()) {
						return comboTicketItem;
					}
				}
			}
			if (!ticketItem.isVoided() && ticketItem.isAdmissionChargeItem()) {
				return ticketItem;
			}
		}
		return null;
	}

	public Double getAdmissionFee() {
		TicketItem admissionFeeItem = getAdmissionFeeTicketItem();
		return admissionFeeItem == null ? 0D : admissionFeeItem.getSubtotalAmount();
	}

	public String getQuotationId() {
		return getProperty("quotationId"); //$NON-NLS-1$
	}

	public void setDueDate(Date dueDate) {
		if (dueDate == null) {
			removeProperty("dueDate"); //$NON-NLS-1$
			return;
		}
		addProperty("dueDate", DateUtil.formatDateWithDefaultTimeAndSec(dueDate)); //$NON-NLS-1$
	}

	@JsonIgnore
	public Date getDueDate() {
		String dueDateStr = getProperty("dueDate"); //$NON-NLS-1$
		try {
			Date dueDate = DateUtil.formatDateWithDefaultTimeAndSec(dueDateStr);
			if (dueDate != null) {
				return DateUtil.endOfDay(dueDate);
			}
			return dueDate;
		} catch (ParseException e) {
		}
		return null;
	}

	@XmlTransient
	public Agent getReferrer() {
		if (StringUtils.isBlank(super.getReferrerId())) {
			return referrer = null;
		}
		if (referrer != null && referrer.getId().equals(super.getReferrerId())) {
			return referrer;
		}
		Customer customer = CustomerDAO.getInstance().get(super.getReferrerId());
		if (customer instanceof Agent) {
			return referrer = (Agent) customer;
		}

		return null;
	}

	public void setReferrer(Agent referrer) {
		this.referrer = referrer;
	}

	@Override
	public Boolean isShipped() {
		Boolean shipped = super.isShipped();
		if (shipped) {
			return shipped;
		}

		TicketItem shippableItem = getTicketItems().stream().filter(t -> t.isRequiredShipping()).findFirst().orElse(null);

		return shippableItem == null;
	}

	public void putCustomerEditable(boolean isCustomerEditable) {
		addProperty("customer.editable", String.valueOf(isCustomerEditable));
	}

	public boolean isCustomerEditable() {
		return getBooleanProperty("customer.editable", Boolean.TRUE);
	}

	public String getProjectNameDisplay() {
		if (StringUtils.isNotBlank(getProjectId())) {
			Project project = (Project) DataProvider.get().getObjectOf(Project.class, getProjectId());
			if (project != null) {
				return project.getNameDisplay();
			}
		}
		return null;
	}

	public String getProjectNote() {
		if (StringUtils.isNotBlank(getProjectId())) {
			Project project = (Project) DataProvider.get().getObjectOf(Project.class, getProjectId());
			if (project != null) {
				return project.getNote();
			}
		}
		return null;
	}

	public String getTermsAndConditionsId() {
		return getProperty("termsAndConditionsId"); //$NON-NLS-1$
	}

	public void putTermsAndConditionsId(String id) {
		addProperty("termsAndConditionsId", id); //$NON-NLS-1$
	}

	public void putOfficeReceived(double amount) {
		addProperty("office_received", String.valueOf(amount));
	}

	public double getOfficeReceived() {
		return getDoubleProperty("office_received");
	}

	public void putSubtotalWithoutReturnedItems(double amount) {
		addProperty("stwri", String.valueOf(amount));
	}

	public double getSubtotalWithoutReturnedItems() {
		return getDoubleProperty("stwri");
	}

	public void putItemDiscountWithoutReturnedItems(double amount) {
		addProperty("idwri", String.valueOf(amount));
	}

	public double getItemDiscountWithoutReturnedItems() {
		return getDoubleProperty("idwri");
	}

	public BookingInfo getBookingInfo() {
		if (bookingInfo == null && StringUtils.isNotBlank(getAdmissionId())) {
			bookingInfo = DataProvider.get().getBookingInfoByBookingId(getAdmissionId());
		}
		return bookingInfo;
	}

	public Date getOtDate() {
		String dateString = getProperty("ot_date");
		if (StringUtils.isNotBlank(dateString)) {
			try {
				return DateUtil.parseUTCString(dateString);
			} catch (Exception ignored) {
			}
		}
		return null;
	}

	public void putOtDate(Date otDate) {
		addProperty("ot_date", otDate == null ? null : DateUtil.formatDateAsUTCString(otDate));
	}

	public double getSubtotalAmountWithoutItemDiscount() {
		return (getSubtotalAmount() - getItemDiscountAmount());
	}

	public double getLabDoctorFeeDueAmount() {
		return getLabDoctorFee() - getLabDoctorFeePaidAmount();
	}

	public double getReferrerFeeDueAmount() {
		return getTotalReferrerFee() - getReferrerFeePaidAmount();
	}

	public void setProject(Project project) {
		this.project = project;
		if (project == null) {
			removeProperty("project_name");
			setProjectId(null);
		}
		else {
			addProperty("project_name", project.getName());
			setProjectId(project.getId());
		}
	}

	public Project getProject() {
		if (this.project != null) {
			return this.project;
		}
		String projectId = getProjectId();
		if (StringUtils.isNotBlank(projectId)) {
			this.project = (Project) DataProvider.get().getObjectOf(Project.class, projectId);
		}

		return this.project;
	}

	public TicketItem getOTTicketItem() {
		for (TicketItem ticketItem : getTicketItems()) {
			String productType = ticketItem.getProductType();
			if (StringUtils.isBlank(productType)) {
				continue;
			}
			if (productType.equals(ProductType.OT.name())) {
				return ticketItem;
			}
		}
		return null;
	}

	public TicketItem getTicketItemByMenuItemId(String menuItemId) {
		if (StringUtils.isBlank(menuItemId)) {
			return null;
		}
		for (TicketItem ticketItem : getTicketItems()) {
			String ticketItem_menuItemId = ticketItem.getMenuItemId();
			if (StringUtils.isBlank(ticketItem_menuItemId)) {
				continue;
			}
			if (ticketItem_menuItemId.equals(menuItemId)) {
				return ticketItem;
			}
		}
		return null;
	}

	public TicketItem getTicketItemByBedId(String bedId) {
		if (StringUtils.isBlank(bedId)) {
			return null;
		}
		for (TicketItem ticketItem : getTicketItems()) {
			if (ticketItem.isComboItem() && ticketItem.getComboItems() != null && ticketItem.getComboItems().size() > 0) {
				List<TicketItem> comboTicketItems = ticketItem.getComboItems();
				for (TicketItem comboTicketItem : comboTicketItems) {
					if (!comboTicketItem.isBedItem()) {
						continue;
					}
					String comboTicketItemBedId = comboTicketItem.getBedId();
					if (StringUtils.isBlank(comboTicketItemBedId)) {
						continue;
					}
					if (comboTicketItemBedId.equals(bedId)) {
						return comboTicketItem;
					}
				}
			}
			if (!ticketItem.isBedItem()) {
				continue;
			}

			String ticketItem_bedId = ticketItem.getBedId();
			if (StringUtils.isBlank(ticketItem_bedId)) {
				continue;
			}
			if (ticketItem_bedId.equals(bedId)) {
				return ticketItem;
			}
		}
		return null;
	}

	public String getDonorName() {
		return getProperty("donor.name");
	}

	public void putDonorName(String donorName) {
		addProperty("donor.name", donorName);
	}

	public BloodGroupType getBloodGroup() {
		String property = getProperty("blood.group");
		return BloodGroupType.fromNameString(property);
	}

	public void putBloodGroup(BloodGroupType bloodGroupType) {
		if (bloodGroupType == null) {
			return;
		}

		addProperty("blood.group", bloodGroupType.name());
	}

	public boolean hasAnyLabItem() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				if (ticketItem.isLabTest()) {
					return Boolean.TRUE;
				}
			}
		}
		return Boolean.FALSE;
	}

	public void putGivenByUser(User currentUser) {
		if (currentUser == null) {
			setDiscountGivenByUser(null);
			putDiscountGivenByUserName(null);
			return;
		}

		setDiscountGivenByUser(currentUser.getId());
		putDiscountGivenByUserName(currentUser.getFullName());
	}

	public String getDiscountGivenByUserName() {
		return getProperty("DISCOUNT.GIVEN_BY_USER_NAME");
	}

	private void putDiscountGivenByUserName(String givenByUserName) {
		addProperty("DISCOUNT.GIVEN_BY_USER_NAME", givenByUserName);
	}

	public List<PosTransaction> getAllocatedTransactionAndDirectTransactions() {
		Map<String, PosTransaction> transactionMap = new HashMap<String, PosTransaction>();

		Set<TicketPaymentAllocation> ticketPaymentAllocations = getTicketPaymentAllocations();
		if (ticketPaymentAllocations != null) {
			for (TicketPaymentAllocation allocation : ticketPaymentAllocations) {
				String transactionId = allocation.getTransactionId();
				PosTransaction transaction = PosTransactionDAO.getInstance().get(transactionId);
				if (transaction != null) {
					transactionMap.put(transactionId, transaction);
				}
			}
		}
		// From ticket's direct transactions
		Set<PosTransaction> directTransactions = getTransactions();
		if (directTransactions != null) {
			for (PosTransaction transaction : directTransactions) {
				if (transaction != null) {
					transactionMap.putIfAbsent(transaction.getId(), transaction);
				}
			}
		}

		return new ArrayList<>(transactionMap.values());
	}

	public String getTicketIdDisplay() {
		TicketType ticketType = getTicketType();
		if (ticketType != null && TicketType.OUTDOOR == ticketType) {
			return getOutdoorTicketId();
		}

		return getId();
	}

	public boolean isApplyDiscountOnDoctorFee() {
		TicketType ticketType = getTicketType();
		if (ticketType != null && TicketType.OUTDOOR == ticketType) {
			for (TicketItem ticketItem : getTicketItems()) {
				if (ticketItem.isHasModifiers()) {
					for (TicketItemModifier modifier : ticketItem.getTicketItemModifiers()) {
						if (modifier.isApplyDiscountOnDoctorFee()) {
							return true;
						}
					}
				}
			}
		}
		return false;
	}

	public void putOTPriceIncludeSurgeonFees(Boolean priceIncludeSurgeonFees) {
		addProperty("price_include_surgeon_fees", String.valueOf(priceIncludeSurgeonFees)); //$NON-NLS-1$
	}

	public boolean isOTPriceIncludeSurgeonFees() {
		return getBooleanProperty("price_include_surgeon_fees", Boolean.FALSE); //$NON-NLS-1$
	}

	public String getOTRequestId() {
		return getProperty("ot.surgeryinfo_id");
	}

	public void putOTRequestId(String surgeryID) {
		addProperty("ot.surgeryinfo_id", surgeryID);
	}

	public OTStatus getOTRequestStatus() {
		return OTStatus.fromNameString(getProperty("ot.surgeryinfo_status"));
	}

	public void putOTRequestStatus(String surgeryStatus) {
		addProperty("ot.surgeryinfo_status", surgeryStatus);
	}

	public String getOtName() {
		return getProperty("otName", "");
	}

	public void putOtName(String otName) {
		addProperty("otName", otName);
	}

	public void putBookingDischarged(boolean bookingDischarged) {
		addProperty("bookingDischarged", String.valueOf(bookingDischarged));
	}

	public boolean isBookingDischarged() {
		return getBooleanProperty("bookingDischarged", false);
	}
}