package com.floreantpos.model.dao;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.DoctorVisitMedicineItem;
import com.floreantpos.model.MedicationSchedule;
import com.floreantpos.model.MedicationStatus;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.PatientBookingStatus;
import com.floreantpos.swing.PaginatedListModel;

public class MedicationScheduleDAO extends BaseMedicationScheduleDAO {

	public MedicationScheduleDAO() {
	}

	@Override
	public String save(MedicationSchedule medicationSchedule, Session s) throws HibernateException {
		updateTime(medicationSchedule);
		return super.save(medicationSchedule, s);
	}

	@Override
	public void update(MedicationSchedule medicationSchedule, Session s) throws HibernateException {
		updateTime(medicationSchedule);
		super.update(medicationSchedule, s);
	}

	@Override
	public void saveOrUpdate(MedicationSchedule medicationSchedule, Session s) throws HibernateException {
		updateTime(medicationSchedule);
		super.saveOrUpdate(medicationSchedule, s);
	}

	@Override
	public void delete(MedicationSchedule medicationSchedule, Session s) throws HibernateException {
		medicationSchedule.setDeleted(true);
		super.update(medicationSchedule, s);
	}

	@Override
	public List<MedicationSchedule> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MedicationSchedule.class);
			addDeletedFilter(criteria);

			criteria.addOrder(Order.asc(MedicationSchedule.PROP_MEDICATION_TIME));
			return criteria.list();
		}
	}

	public void saveMedicineItemMedicationSchedules(List<DoctorVisitMedicineItem> dvMedicineItems, List<MedicationSchedule> newMedicationSchedules) {

		Transaction tx = null;

		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			for (DoctorVisitMedicineItem dvMedicineItem : dvMedicineItems) {
				if (dvMedicineItem.isMedicationScheduled()) {
					continue;
				}

				dvMedicineItem.putMedicationScheduled(true);
				DoctorVisitMedicineItemDAO.getInstance().saveOrUpdate(dvMedicineItem, session);
			}

			for (MedicationSchedule medicationSchedule : newMedicationSchedules) {
				saveOrUpdate(medicationSchedule, session);
			}

			tx.commit();
		}
	}

	public List<MedicationSchedule> getOtherPendingMedicationSchedulesBy(String admissionId, String patientId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MedicationSchedule.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(admissionId)) {
				criteria.add(Restrictions.eq(MedicationSchedule.PROP_ADMISSION_ID, admissionId));
			}

			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq(MedicationSchedule.PROP_PATIENT_ID, patientId));
			}

			criteria.add(Restrictions.eq(MedicationSchedule.PROP_STATUS, MedicationStatus.PENDING.name()));

			criteria.addOrder(Order.asc(MedicationSchedule.PROP_MEDICATION_TIME));
			return criteria.list();
		}
	}

	public void loadMedicationSchedules(PaginatedListModel<MedicationSchedule> dataModel, Date fromDate, Date toDate, String patientId, String outletId) {
		loadMedicationSchedules(dataModel, fromDate, toDate, patientId, null, false, null, null, null, null, outletId, null);
	}

	public void loadMedicationSchedules(PaginatedListModel<MedicationSchedule> dataModel, Date fromDate, Date toDate, String patientId, String medicineName,
			boolean isFromPatientDialog, String roomId, String bedId, MedicationStatus medicationStatus, String nurseId, String outletId,
			PatientBookingStatus patientBookingStatus) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MedicationSchedule.class);
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(MedicationSchedule.PROP_OUTLET_ID, outletId));
			}
			DoctorVisitDAO.createRoomAndBedCriteria(roomId, bedId, criteria, MedicationSchedule.PROP_ADMISSION_ID, patientBookingStatus);
			if (StringUtils.isNotBlank(nurseId)) {
				criteria.add(Restrictions.eq(MedicationSchedule.PROP_NURSE_ID, nurseId));
			}
			if (isFromPatientDialog) {
				criteria.add(Restrictions.le(MedicationSchedule.PROP_MEDICATION_TIME, toDate));
			}
			else {
				if (fromDate != null) {
					criteria.add(Restrictions.ge(MedicationSchedule.PROP_MEDICATION_TIME, fromDate));
				}

				if (toDate != null) {
					criteria.add(Restrictions.le(MedicationSchedule.PROP_MEDICATION_TIME, toDate));
				}
			}

			if (medicationStatus != null) {
				criteria.add(Restrictions.eq(MedicationSchedule.PROP_STATUS, medicationStatus.name()));
			}

			if (StringUtils.isNotBlank(medicineName)) {
				DetachedCriteria menuItemNameSubCriteria = DetachedCriteria.forClass(MenuItem.class);
				menuItemNameSubCriteria.setProjection(Property.forName(MenuItem.PROP_ID));

				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.eq(MenuItem.PROP_BARCODE, medicineName));
				disjunction.add(Restrictions.eq(MenuItem.PROP_SKU, medicineName));
				disjunction.add(Restrictions.ilike(MenuItem.PROP_NAME, medicineName, MatchMode.ANYWHERE));
				menuItemNameSubCriteria.add(disjunction);

				criteria.add(Property.forName(MedicationSchedule.PROP_ITEM_ID).in(menuItemNameSubCriteria)); //$NON-NLS-1$
			}

			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq(MedicationSchedule.PROP_PATIENT_ID, patientId));
			}

			dataModel.setNumRows(rowCount(criteria));

			criteria.addOrder(Order.asc(MedicationSchedule.PROP_MEDICATION_TIME));
			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());
			dataModel.setData(criteria.list());
		}
	}

}