package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.OnlineStore;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.Store;
import com.floreantpos.util.POSUtil;

public class OnlineStoreDAO extends BaseOnlineStoreDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public OnlineStoreDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime((OnlineStore) obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime((OnlineStore) obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime((OnlineStore) obj);
		super.saveOrUpdate(obj, s);
	}

	public OnlineStore findBy(String storeId, String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(OnlineStore.PROP_STORE_ID, storeId));
			criteria.add(Restrictions.eq(OnlineStore.PROP_OUTLET_ID, outletId));
			criteria.setMaxResults(1);
			return (OnlineStore) criteria.uniqueResult();
		}
	}

	public List<OnlineStore> findBy(String storeId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(OnlineStore.PROP_STORE_ID, storeId));
			return criteria.list();
		}
	}

	public Boolean isStoreOnLiveMode(String storeId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(OnlineStore.PROP_STORE_ID, storeId));
			criteria.add(Restrictions.eq(OnlineStore.PROP_SEARCHABLE, true));
			Number number = (Number) criteria.uniqueResult();
			return number != null && number.intValue() > 0;
		}
	}

	public String generateFriendlyUrl(OnlineStore onlineStore) {
		try (Session session = createNewSession()) {
			return generateFriendlyUrl(onlineStore, session);
		}
	}

	@SuppressWarnings("rawtypes")
	public String generateFriendlyUrl(OnlineStore onlineStore, Session session) {
		String friendlyUrl = onlineStore.buildFriendlyUrl();
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.setProjection(Projections.property(OnlineStore.PROP_FRIENDLY_URL));
		criteria.add(Restrictions.like(OnlineStore.PROP_FRIENDLY_URL, friendlyUrl, MatchMode.START));
		if (StringUtils.isNotBlank(onlineStore.getId())) {
			criteria.add(Restrictions.ne(OnlineStore.PROP_ID, onlineStore.getId()));
		}
		int randomId = 1;
		List list = criteria.list();
		if (list == null || list.isEmpty()) {
			return friendlyUrl + "-" + randomId; //$NON-NLS-1$
		}
		List<Integer> randomIds = new ArrayList<>();
		for (Iterator iterator = list.iterator(); iterator.hasNext();) {
			String existingFriendlyUrl = (String) iterator.next();
			int lastIndexOf = existingFriendlyUrl.lastIndexOf("-"); //$NON-NLS-1$
			if (lastIndexOf != -1) {
				String lastNumberString = existingFriendlyUrl.substring(lastIndexOf + 1);
				int lastNumberAsInt = POSUtil.parseInteger(lastNumberString);
				if (lastNumberAsInt > 0) {
					randomIds.add(lastNumberAsInt);
				}
			}
		}
		if (!randomIds.isEmpty() && Collections.max(randomIds) > 0) {
			randomId = Collections.max(randomIds) + 1;
		}
		return friendlyUrl + "-" + randomId; //$NON-NLS-1$
	}

	public OnlineStore findByFriendlyUrl(String friendlyUrl) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(OnlineStore.PROP_FRIENDLY_URL, friendlyUrl));
			//criteria.add(Restrictions.eq(OnlineStore.PROP_LIVE, Boolean.TRUE));
			//criteria.add(Restrictions.eq(OnlineStore.PROP_TEMPORARILY_SHUTDOWN, Boolean.FALSE));
			addDeletedFilter(criteria);
			criteria.setMaxResults(1);
			OnlineStore onlineStore = (OnlineStore) criteria.uniqueResult();
			if (onlineStore != null) {
				return onlineStore;
			}

			//if no store found with friend url, try splitting friendly by ___
			//as if friendly url is not set for a store, it's friendly url becomes storeId___outletId
			String[] split = friendlyUrl.split("___"); //$NON-NLS-1$
			if (split.length < 2) {
				return null;
			}
			String storeId = split[0];
			String outletId = split[1];

			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(OnlineStore.PROP_STORE_ID, storeId));
			criteria.add(Restrictions.eq(OnlineStore.PROP_OUTLET_ID, outletId));
			addDeletedFilter(criteria);
			criteria.setMaxResults(1);

			return (OnlineStore) criteria.uniqueResult();
		}
	}

	@SuppressWarnings("unchecked")
	public void loadData(String searchKeyword, String storeName, Pagination<OnlineStore> pagination, Boolean isEnable, Boolean isLive, Boolean isFeatured,
			Boolean isSticky, boolean isShowTestData) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			if (StringUtils.isNotBlank(searchKeyword)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.like(OnlineStore.PROP_FRIENDLY_URL, searchKeyword, MatchMode.ANYWHERE));
				criteria.add(disjunction);
			}
			if (StringUtils.isNotBlank(storeName)) {
				criteria.add(Restrictions.like(OnlineStore.PROP_STORE_NAME, storeName, MatchMode.ANYWHERE).ignoreCase());
			}
			if (isEnable != null) {
				criteria.add(Restrictions.eq(OnlineStore.PROP_ENABLE, isEnable));
			}
			if (isLive != null) {
				criteria.add(Restrictions.eq(OnlineStore.PROP_LIVE, isLive));
			}
			if (isFeatured != null) {
				criteria.add(Restrictions.eq(OnlineStore.PROP_FEATURED, isFeatured));
			}
			if (isSticky != null) {
				criteria.add(Restrictions.eq(OnlineStore.PROP_STICKY, isSticky));
			}
			if (isShowTestData) {
				//				String searchBy = "\"" + Store.JSON_PROP_MARKED_AS_TEST_DATA + "\":\"" + isShowTestData + "\""; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				//				criteria.add(Restrictions.ilike(OnlineStore.PROP_PROPERTIES, searchBy, MatchMode.ANYWHERE));

				criteria.add(Restrictions.eq(OnlineStore.PROP_TEST_STORE, isShowTestData));
			}
			else {
				//				String key = "\"" + Store.JSON_PROP_MARKED_AS_TEST_DATA + "\""; //$NON-NLS-1$ //$NON-NLS-2$
				//				String searchBy = "\"" + Store.JSON_PROP_MARKED_AS_TEST_DATA + "\":\"" + isShowTestData + "\""; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.isNull(OnlineStore.PROP_TEST_STORE));
				//				disjunction.add(Restrictions.not(Restrictions.ilike(OnlineStore.PROP_PROPERTIES, key, MatchMode.ANYWHERE)));
				disjunction.add(Restrictions.eq(OnlineStore.PROP_TEST_STORE, isShowTestData));
				criteria.add(disjunction);
			}

			criteria.setProjection(Projections.rowCount());
			Number uniqueResult = (Number) criteria.uniqueResult();
			int rowCount = uniqueResult == null ? 0 : uniqueResult.intValue();
			pagination.setNumRows(rowCount);
			if (rowCount == 0) {
				pagination.setData(new ArrayList<>());
				return;
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.desc(OnlineStore.PROP_LAST_UPDATE_TIME));
			criteria.setFirstResult(pagination.getCurrentRowIndex());
			criteria.setMaxResults(pagination.getPageSize());
			pagination.setData(criteria.list());
		}
	}

}