package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.Customer;
import com.floreantpos.model.DoctorVisit;
import com.floreantpos.model.DoctorVisitDiagnosisItem;
import com.floreantpos.model.DoctorVisitMedicineItem;
import com.floreantpos.model.NurseVisit;
import com.floreantpos.model.Patient;
import com.floreantpos.model.PatientAllergy;
import com.floreantpos.model.PatientBookingStatus;
import com.floreantpos.model.PatientHistory;
import com.floreantpos.model.PatientHistory.HistoryRefType;
import com.floreantpos.model.Prescription;
import com.floreantpos.model.PrescriptionItem;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.Severity;
import com.floreantpos.model.Symptom;
import com.floreantpos.model.User;

public class PatientHistoryDAO extends BasePatientHistoryDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PatientHistoryDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof PatientHistory) {
			PatientHistory patientHistory = (PatientHistory) obj;
			patientHistory.setDeleted(Boolean.TRUE);
			super.update(patientHistory, s);
		}
	}

	public PatientHistory getPatientHistory(HistoryRefType historyRefType, String refId) {
		try (Session session = createNewSession()) {
			return getPatientHistory(historyRefType, refId, session);
		}
	}

	public PatientHistory getPatientHistory(HistoryRefType historyRefType, String refId, Session session) {
		if (historyRefType == null || StringUtils.isBlank(refId)) {
			return null;
		}

		Criteria criteria = session.createCriteria(PatientHistory.class);
		criteria.add(Restrictions.eq(PatientHistory.PROP_HISTORY_REF_TYPE, historyRefType.name()));
		criteria.add(Restrictions.eq(PatientHistory.PROP_REF_ID, refId));
		criteria.setMaxResults(1);
		return (PatientHistory) criteria.uniqueResult();
	}

	public void saveAdmissionHistory(BookingInfo booking) {
		saveAdmissionHistory(booking, null);
	}

	public void saveAdmissionHistory(BookingInfo booking, Session session) {
		PatientHistory patientHistory = session == null ? getPatientHistory(HistoryRefType.BOOKING_INFO, booking.getId())
				: getPatientHistory(HistoryRefType.BOOKING_INFO, booking.getId(), session);

		AtomicReference<PatientHistory> historyAtomic = new AtomicReference<>(patientHistory);

		if (historyAtomic.get() == null) {
			PatientHistory history = new PatientHistory();
			history.setCreateDate(booking.getCreatedDate());
			history.setRefId(booking.getId());
			history.setHistoryRefType(HistoryRefType.BOOKING_INFO.name());
			history.setSummary("Admission info");
			historyAtomic.set(history);
		}
		historyAtomic.get().setPatientId(booking.getCustomerId());
		historyAtomic.get().setDoctorId(booking.getDoctorId());
		historyAtomic.get().putDoctorName(booking.getDoctorName());

		if (PatientBookingStatus.DISCHARGED.name().equals(booking.getStatus())) {
			historyAtomic.get().putBookingReleaseDate(booking.getToDate());
		}

		Customer patient = booking.getCustomer();
		// @formatter:off
		historyAtomic.get().putPatientVital(historyAtomic.get().new PatientVital(
					patient.getApproxAge(), 
					booking.getPatientHeightInCm(), 
					booking.getPatientWeight(), 
					booking.getPatientPressureUp(), 
					booking.getPatientPressureDown(), 
					booking.getPatientTemperature(), 
					booking.getPatientBpm()
				));
		// @formatter:on

		List<Symptom> symptoms = booking.getSymptoms();
		if (symptoms != null && !symptoms.isEmpty()) {
			historyAtomic.get().putProblems(symptoms.stream().map(t -> t.getName()).collect(Collectors.toList()));
		}

		List<PatientAllergy> patientAllergies = booking.getPatientAllergies();
		// @formatter:off
		if (patientAllergies != null && !patientAllergies.isEmpty()) {
			historyAtomic.get().putAllergies(patientAllergies.stream()
							.map(t -> historyAtomic.get().new PatientAllergy(
									t.getAllergyName(), 
									Severity.fromString(t.getSeverity()).getDisplayString()))
							.collect(Collectors.toList()));
		}
		// @formatter:on

		historyAtomic.get().setNote(booking.getNote());

		if (StringUtils.isNotBlank(historyAtomic.get().getId())) {
			if (session == null) {
				update(historyAtomic.get());
			}
			else {
				session.update(historyAtomic.get());
			}
		}
		else {
			if (session == null) {
				save(historyAtomic.get());
			}
			else {
				session.save(historyAtomic.get());
			}
		}
	}

	public void saveDoctorVisitHistory(DoctorVisit doctorVisit) {
		saveDoctorVisitHistory(doctorVisit, null);
	}

	public void saveDoctorVisitHistory(DoctorVisit doctorVisit, Session session) {
		PatientHistory patientHistory = session == null ? getPatientHistory(HistoryRefType.DOCTOR_VISIT, doctorVisit.getId())
				: getPatientHistory(HistoryRefType.DOCTOR_VISIT, doctorVisit.getId(), session);

		AtomicReference<PatientHistory> historyAtomic = new AtomicReference<>(patientHistory);

		if (historyAtomic.get() == null) {
			PatientHistory history = new PatientHistory();
			history.setCreateDate(doctorVisit.getCreatedDate());
			history.setRefId(doctorVisit.getId());
			history.setHistoryRefType(HistoryRefType.DOCTOR_VISIT.name());
			history.setPatientId(doctorVisit.getPatientId());
			history.setSummary("Doctor visited");
			historyAtomic.set(history);
		}
		historyAtomic.get().setDoctorId(doctorVisit.getDoctorId());
		historyAtomic.get().putDoctorName(doctorVisit.getDoctorName());

		historyAtomic.get().putProblems(doctorVisit.getProblems());

		List<DoctorVisitDiagnosisItem> diagnosisItems = doctorVisit.getDiagnosisItems();
		// @formatter:off
		if (diagnosisItems != null && !diagnosisItems.isEmpty()) {
			historyAtomic.get().putPatientDiagnoses(diagnosisItems.stream()
						.map(t -> historyAtomic.get().new PatientDiagnosis(
							t.getName()))
						.collect(Collectors.toList()));
		}
		// @formatter:on

		List<DoctorVisitMedicineItem> medicineItems = doctorVisit.getMedicineItems();
		// @formatter:off
		if (medicineItems != null && !medicineItems.isEmpty()) {
			historyAtomic.get().putPatientMedications(medicineItems.stream()
						.map(t -> historyAtomic.get().new PatientMedication(
								t.getItemNameAsString(), 
								t.getInstructionDisplay(false, false)))
						.collect(Collectors.toList()));
		}
		// @formatter:on

		List<String> instructions = doctorVisit.getInstructions();
		if (instructions != null && !instructions.isEmpty()) {
			historyAtomic.get().putInstructions(instructions);
		}

		historyAtomic.get().setNote(doctorVisit.getNotes());

		if (StringUtils.isNotBlank(historyAtomic.get().getId())) {
			if (session == null) {
				update(historyAtomic.get());
			}
			else {
				session.update(historyAtomic.get());
			}
		}
		else {
			if (session == null) {
				save(historyAtomic.get());
			}
			else {
				session.save(historyAtomic.get());
			}
		}
	}

	public void saveNurseVisitHistory(NurseVisit nurseVisit) {
		saveNurseVisitHistory(nurseVisit, null);
	}

	public void saveNurseVisitHistory(NurseVisit nurseVisit, Session session) {
		PatientHistory patientHistory = session == null ? getPatientHistory(HistoryRefType.NURSE_VISIT, nurseVisit.getId())
				: getPatientHistory(HistoryRefType.NURSE_VISIT, nurseVisit.getId(), session);

		AtomicReference<PatientHistory> historyAtomic = new AtomicReference<>(patientHistory);

		PatientHistory history = historyAtomic.get();
		if (history == null) {
			history = new PatientHistory();
			history.setCreateDate(nurseVisit.getCreatedDate());
			history.setRefId(nurseVisit.getId());
			history.setHistoryRefType(HistoryRefType.NURSE_VISIT.name());
			history.setPatientId(nurseVisit.getPatientId());
			history.setSummary("Nurse visited");
		}
		User nurse = nurseVisit.getNurse();
		if (nurse != null) {
			history.putNurseName(nurse.getFullName());
		}

		Patient patient = nurseVisit.getPatient();
		// @formatter:off
		history.putPatientVital(history.new PatientVital(
					patient.getApproxAge(), 
					nurseVisit.getPatientHeightInCm(), 
					nurseVisit.getPatientWeight(), 
					nurseVisit.getPressureUp(), 
					nurseVisit.getPressureDown(), 
					nurseVisit.getTemparature(), 
					nurseVisit.getHeartBitPerMinute()
				));
		// @formatter:on

		history.putProblems(nurseVisit.getProblems());
		history.setNote(nurseVisit.getNotes());

		if (StringUtils.isNotBlank(history.getId())) {
			if (session == null) {
				update(history);
			}
			else {
				session.update(history);
			}
		}
		else {
			if (session == null) {
				save(history);
			}
			else {
				session.save(history);
			}
		}
	}

	public void savePrescriptionHistory(Prescription prescription) {
		savePrescriptionHistory(prescription, null);
	}

	public void savePrescriptionHistory(Prescription prescription, Session session) {
		AtomicReference<PatientHistory> historyAtomic = new AtomicReference<>(getPatientHistory(HistoryRefType.PRESCRIPTION, prescription.getId()));

		if (historyAtomic.get() == null) {
			PatientHistory history = new PatientHistory();
			history.setCreateDate(prescription.getCreatedDate());
			history.setRefId(prescription.getId());
			history.setDoctorId(prescription.getDoctorId());
			history.putDoctorName(prescription.getDoctorName());
			history.setHistoryRefType(HistoryRefType.PRESCRIPTION.name());
			history.setSummary("Prescription");

			historyAtomic.set(history);
		}

		historyAtomic.get().setPatientId(prescription.getPatientId());

		Customer patient = prescription.getPatient();
		// @formatter:off
		historyAtomic.get().putPatientVital(historyAtomic.get().new PatientVital(
					patient.getApproxAge(), 
					prescription.getHeightInCM(), 
					prescription.getWeightInKG(), 
					prescription.getBpUp(), 
					prescription.getBpDown(), 
					prescription.getTemparature(), 
					prescription.getBpm()
				));
		// @formatter:on

		historyAtomic.get().putProblems(prescription.getProblems());

		Map<Boolean, List<PrescriptionItem>> partitioned = prescription.getItems().stream()
				.collect(Collectors.partitioningBy(t -> ProductType.PATHOLOGY.name().equals(t.getProductType())));
		List<PrescriptionItem> diagnosisItems = partitioned.get(true);
		List<PrescriptionItem> medicineItems = partitioned.get(false);
		// @formatter:off
		if (diagnosisItems != null && !diagnosisItems.isEmpty()) {
			historyAtomic.get().putPatientDiagnoses(diagnosisItems.stream()
						.map(t -> historyAtomic.get().new PatientDiagnosis(
							t.getName()))
						.collect(Collectors.toList()));
		}

		if (medicineItems != null && !medicineItems.isEmpty()) {
			historyAtomic.get().putPatientMedications(medicineItems.stream()
						.map(t -> historyAtomic.get().new PatientMedication(
								t.getMenuItem().getDisplayName(), 
								t.getInstructionDisplay(false, false)))
						.collect(Collectors.toList()));
		}
		// @formatter:on

		historyAtomic.get().setNote(prescription.getNotes());

		if (StringUtils.isNotBlank(historyAtomic.get().getId())) {
			if (session == null) {
				update(historyAtomic.get());
			}
			else {
				session.update(historyAtomic.get());
			}
		}
		else {
			if (session == null) {
				save(historyAtomic.get());
			}
			else {
				session.save(historyAtomic.get());
			}
		}
	}

	public List<PatientHistory> findHistories(String patientId, Date fromDate, Date toDate) {
		return findHistories(patientId, fromDate, toDate, false);
	}

	public List<PatientHistory> findHistories(String patientId, Date fromDate, Date toDate, boolean skipNurseVisits) {
		if (StringUtils.isBlank(patientId)) {
			throw new PosException("Patient ID is required");
		}
		//if (fromDate == null) {
		//	throw new PosException("Select from date");
		//}
		//if (toDate == null) {
		//	throw new PosException("Select to date");
		//}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PatientHistory.class);
			criteria.add(Restrictions.eq(PatientHistory.PROP_PATIENT_ID, patientId));
			if (fromDate != null && toDate != null) {
				criteria.add(Restrictions.between(PatientHistory.PROP_CREATE_DATE, fromDate, toDate));
			}
			if (skipNurseVisits) {
				criteria.add(Restrictions.ne(PatientHistory.PROP_HISTORY_REF_TYPE, HistoryRefType.NURSE_VISIT.name()));
			}
			criteria.addOrder(Order.desc(PatientHistory.PROP_CREATE_DATE));
			return criteria.list();
		}
	}

	public List<PatientHistory> findLatestHistoriesByPatientId(String patientId) {
		List<PatientHistory> list = new ArrayList<>();
		if (StringUtils.isBlank(patientId)) {
			return list;
		}

		try (Session session = createNewSession()) {
			for (HistoryRefType historyRefType : HistoryRefType.values()) {
				PatientHistory patientHistory = getPatientHistory(session, patientId, historyRefType);
				if (patientHistory != null) {
					list.add(patientHistory);
				}
			}
		}
		if (!list.isEmpty()) {
			list.sort(Comparator.comparing(PatientHistory::getCreateDate).reversed());
		}
		return list;
	}

	private PatientHistory getPatientHistory(Session session, String patientId, HistoryRefType refType) {
		Criteria criteria = session.createCriteria(PatientHistory.class);
		criteria.add(Restrictions.eq(PatientHistory.PROP_PATIENT_ID, patientId));
		criteria.add(Restrictions.eq(PatientHistory.PROP_HISTORY_REF_TYPE, refType.name()));
		criteria.setMaxResults(1);
		criteria.addOrder(Order.desc(PatientHistory.PROP_CREATE_DATE));
		return (PatientHistory) criteria.uniqueResult();
	}

}