package com.floreantpos.report;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.Period;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.imageio.ImageIO;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.WordUtils;
import org.apache.commons.lang3.StringUtils;
import org.castor.core.util.Base64Encoder;

import com.floreantpos.PosLog;
import com.floreantpos.model.Bed;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.ImageResource;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Room;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.ext.PaperSize;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.POSUtil;

import net.sourceforge.barbecue.Barcode;
import net.sourceforge.barbecue.BarcodeFactory;
import net.sourceforge.barbecue.BarcodeImageHandler;
import pl.allegro.finance.tradukisto.MoneyConverters;

public class AdmissionReportUtil {

	public static final String PROP_PRINT_HEADER = "admission_form.print_header"; //$NON-NLS-1$
	public static final String PROP_RECEIPT_REPORT_FOOTER_MARGIN = "admission_form.receipt_report_footer_margin"; //$NON-NLS-1$
	public static final String PROP_RECEIPT_REPORT_HEADER_MARGIN = "admission_form.receipt_report_header_margin"; //$NON-NLS-1$
	public static final String PROP_PRINT_RECEIPT_STORE_LOGO = "admission_form.print_receipt_store_logo"; //$NON-NLS-1$
	public static final String PROP_TERMS_AND_CONDITIONS = "admission_form.terms_and_conditions"; //$NON-NLS-1$

	public static HashMap<String, Object> createAdmissionInfoHtmlPrintMap(Map<String, Object> mapForPreview, Ticket ticket, BookingInfo bookingInfo) {
		HashMap<String, Object> contextMap = new HashMap<String, Object>();
		contextMap.put("StringUtils", StringUtils.class); //$NON-NLS-1$
		Outlet outlet = DataProvider.get().getOutlet();
		ReportUtil.populateMedlogicsProperties(contextMap, outlet);

		contextMap.put("termsAndConditions", outlet.getProperty(PROP_TERMS_AND_CONDITIONS)); //$NON-NLS-1$
		contextMap.put("showStoreLogo", outlet.getBooleanProperty(PROP_PRINT_RECEIPT_STORE_LOGO, false)); //$NON-NLS-1$
		contextMap.put("receiptHeaderMargin", outlet.getProperty(PROP_RECEIPT_REPORT_HEADER_MARGIN)); //$NON-NLS-1$
		contextMap.put("receiptFooterMargin", outlet.getProperty(PROP_RECEIPT_REPORT_FOOTER_MARGIN)); //$NON-NLS-1$
		contextMap.put("visibleReportHeader", outlet.getBooleanProperty(PROP_PRINT_HEADER, false)); //$NON-NLS-1$

		Customer customer = ticket.getCustomer();
		Doctor doctor = StringUtils.isBlank(bookingInfo.getDoctorId()) ? null : DoctorDAO.getInstance().get(bookingInfo.getDoctorId());

		Boolean showReportsInBangla = outlet.isShowReportsInBangla();
		contextMap.put("tab", showReportsInBangla ? "ঃ" : ":"); //$NON-NLS-1$
		contextMap.put("patientNameTxt", showReportsInBangla ? "রোগীর নাম" : "Patient name"); //$NON-NLS-1$
		contextMap.put("bookingIdTxt", showReportsInBangla ? "রেজি নং" : "Reg. No"); //$NON-NLS-1$
		contextMap.put("patientAddressAndPhoneTxt", showReportsInBangla ? "রোগীর ঠিকানা এবং টেলিফোন নম্বর" : "Patient address and phone"); //$NON-NLS-1$
		contextMap.put("patientGurdianTxt", //$NON-NLS-1$
				showReportsInBangla ? "রোগীর দায়িত্ব গ্রহণকারী ব্যক্তি/সংস্থার নাম এবং টেলিফোন নম্বর" : "Patient gurdian and phone");
		contextMap.put("doctorTxt", showReportsInBangla ? "বিশেষজ্ঞ ডাক্তার/অধ্যাপকের নাম" : "Doctor"); //$NON-NLS-1$
		contextMap.put("ageTxt", showReportsInBangla ? "বয়স" : "Age"); //$NON-NLS-1$
		contextMap.put("sexTxt", showReportsInBangla ? "লিঙ্গ:" : "Sex:"); //$NON-NLS-1$
		contextMap.put("admissionDateTxt", showReportsInBangla ? "ভর্তির তারিখ" : "Admission date"); //$NON-NLS-1
		contextMap.put("problemNameTxt", showReportsInBangla ? "রোগের নাম" : "Problem"); //$NON-NLS-1$
		contextMap.put("termsAndConditionsTxt", showReportsInBangla ? "ভর্তি সংক্রান্ত নিয়মাবলী" : "Terms And Conditions"); //$NON-NLS-1$ 
		contextMap.put("patientDetailsTxt", showReportsInBangla ? "ভর্তির ফরম এবং রোগীর তথ্য বিবরণী" : "ADMISSION FORM AND PATIENT DETAILS"); //$NON-NLS-1$

		contextMap.put("patientName", getValue(customer.getName())); //$NON-NLS-1$
		contextMap.put("bookingId", getValue(bookingInfo.getBookingId())); //$NON-NLS-1$
		contextMap.put("patientAddress", getValue(customer.getAddress())); //$NON-NLS-1$
		contextMap.put("patientPhoneNo", getValue(customer.getMobileNo())); //$NON-NLS-1$

		StringBuilder guardianInfo = new StringBuilder();
		if (StringUtils.isNotBlank(bookingInfo.getAdmittedBy())) {
			guardianInfo.append(bookingInfo.getAdmittedBy());
		}
		if (StringUtils.isNotBlank(bookingInfo.getAdmittedByPhoneNumber())) {
			guardianInfo.append(", "); //$NON-NLS-1$ 
			guardianInfo.append(bookingInfo.getAdmittedByPhoneNumber());
		}
		contextMap.put("patientGurdian", guardianInfo.toString()); //$NON-NLS-1$ 
		contextMap.put("doctor", doctor == null ? "" : doctor.getName()); //$NON-NLS-1$

		contextMap.put("bedTxt", showReportsInBangla ? "বেড" : "Bed"); //$NON-NLS-1$
		List<Bed> beds = bookingInfo.getBeds();
		if (beds != null && !beds.isEmpty()) {
			Room room = beds.get(0).getRoom();
			contextMap.put("ward", room != null ? room.getName() : ""); //$NON-NLS-1$ //$NON-NLS-2$
		}
		else {
			contextMap.put("ward", ""); //$NON-NLS-1$
		}
		contextMap.put("bed", getValue(bookingInfo.getBedDisplay())); //$NON-NLS-1$

		Date createDate = ticket.getCreateDate();
		Calendar c1 = Calendar.getInstance();
		c1.setTime(createDate);

		Calendar c2 = Calendar.getInstance();
		c2.setTime(bookingInfo.getFromDate());
		c2.set(Calendar.HOUR_OF_DAY, c1.get(Calendar.HOUR_OF_DAY));
		c2.set(Calendar.MINUTE, c1.get(Calendar.MINUTE));
		c2.set(Calendar.SECOND, c1.get(Calendar.SECOND));

		if (bookingInfo.getFromDate() == null) {
			contextMap.put("admissionDate", ""); //$NON-NLS-1$ //$NON-NLS-2$
		}
		else {
			contextMap.put("admissionDate", //$NON-NLS-1$
					showReportsInBangla ? formatAdmissionDate(c2.getTime())
							: DateUtil.formatDateWithBrowserTimeOffset(c2.getTime(), new SimpleDateFormat("dd-MM-yyyy"))); //$NON-NLS-1$
		}
		String sex = getGenderDisplayString(bookingInfo, showReportsInBangla);
		contextMap.put("sex", sex); //$NON-NLS-1$

		if (StringUtils.isBlank(sex)) {
			contextMap.put("sexTxt", ""); //$NON-NLS-1$ //$NON-NLS-2$
		}
		//		StringBuilder problemBuilder = new StringBuilder();
		//		List<Symptom> symptoms = bookingInfo.getSymptoms();
		//		if (symptoms != null && symptoms.size() > 0) {
		//			for (Iterator<Symptom> iterator = symptoms.iterator(); iterator.hasNext();) {
		//				Symptom symptom = (Symptom) iterator.next();
		//				problemBuilder.append(symptom.getName());
		//				if (iterator.hasNext()) {
		//					problemBuilder.append(",");
		//				}
		//			}
		//		}
		contextMap.put("problemName", bookingInfo.getAdmissionReason() == null ? "" : bookingInfo.getAdmissionReason()); //$NON-NLS-1$

		String SPACE = " "; //$NON-NLS-1$
		if (customer.getDateOfBirth() != null) {
			LocalDate dob = DateUtil.convertDateToLocalDate(customer.getDateOfBirth());
			LocalDate currentDate = LocalDate.now();
			Period agePeriod = Period.between(dob, currentDate);
			int years = agePeriod.getYears();
			int months = agePeriod.getMonths();
			int days = agePeriod.getDays();
			String yearString = String.valueOf(years);
			String monthString = String.valueOf(months);
			String dayString = String.valueOf(days);
			if (showReportsInBangla) {
				contextMap.put("age", //$NON-NLS-1$
						convertToBanglaFormat(yearString) + format("Y", showReportsInBangla) + SPACE //$NON-NLS-1$
								+ (months > 0 ? (convertToBanglaFormat(monthString) + format("M", showReportsInBangla) + SPACE) : "") //$NON-NLS-1$
								+ (days > 0 ? (convertToBanglaFormat(dayString) + format("D", showReportsInBangla)) : "")); //$NON-NLS-1$
			}
			else {
				contextMap.put("age", yearString + "Y " + (months > 0 ? (monthString + "M ") : "") + (days > 0 ? (dayString + "D") : "")); //$NON-NLS-1$
			}
		}
		else {
			contextMap.put("age", "");
		}
		if (showReportsInBangla) {
			contextMap.put("fontSize", "10px"); //$NON-NLS-1$
			contextMap.put("footerStyle", //$NON-NLS-1$
					"position:absolute;bottom:0;width:100%;top:0;font-size: 11px;font-family: SolaimanLipi, -apple-system, Arial, BlinkMacSystemFont, sans, Helvetica, sans-serif"); //$NON-NLS-1$
		}
		contextMap.put("showBottomBorder", showReportsInBangla); //$NON-NLS-1$

		int receiptReportHeaderMargin = outlet.getReceiptReportHeaderMargin();
		int receiptReportFooterMargin = outlet.getReceiptReportFooterMargin();
		if (mapForPreview != null) {
			contextMap.putAll(mapForPreview);
			receiptReportHeaderMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptHeaderMarginValue")); //$NON-NLS-1$
			receiptReportFooterMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptFooterMarginValue")); //$NON-NLS-1$
		}
		if (receiptReportFooterMargin < 20) {
			receiptReportFooterMargin = 20;
		}
		contextMap.put("reportHeaderMargin", receiptReportHeaderMargin + "px"); //$NON-NLS-1$ //$NON-NLS-2$
		contextMap.put("reportFooterMargin", receiptReportFooterMargin + "px"); //$NON-NLS-1$ //$NON-NLS-2$

		ImageResource logoImage = outlet.getLogoImageResource();
		if (logoImage != null) {
			contextMap.put("logo", new String(Base64Encoder.encode(logoImage.getImageBytes()))); //$NON-NLS-1$
		}

		User owner = ticket.getOwner();
		String serverName = "";//$NON-NLS-1$
		if (owner != null) {
			serverName = owner.getFullName();
		}
		else {
			serverName += ticket.getProperty("owner.firstname", "") + SPACE + ticket.getProperty("owner.lastname", ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		}
		contextMap.put("colPrintedBy", //$NON-NLS-1$
				"Printed: " + DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd/MM/yy hh:mm a") + SPACE //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
						+ "by" + SPACE + serverName);

		if (StringUtils.isNotBlank(bookingInfo.getBookingId())) {
			try {
				Barcode barcode = BarcodeFactory.createCode128(bookingInfo.getBookingId());
				BufferedImage bufferedImage = BarcodeImageHandler.getImage(barcode);
				ByteArrayOutputStream bos = new ByteArrayOutputStream();
				ImageIO.write(bufferedImage, "PNG", bos); //$NON-NLS-1$
				byte[] data = bos.toByteArray();
				contextMap.put("barcode", new String(Base64Encoder.encode(data))); //$NON-NLS-1$
			} catch (Exception e) {
				PosLog.error(AdmissionReportUtil.class, e);
			}
		}

		return contextMap;
	}

	private static String getValue(String value) {
		if (value == null) {
			return "";
		}
		return value;
	}

	public static String formatAdmissionDate(Date admissionDate) {
		if (admissionDate == null) {
			return ""; //$NON-NLS-1$
		}
		return convertToBanglaFormat(DateUtil.formatDateWithBrowserTimeOffset(admissionDate, new SimpleDateFormat("dd-MM-yyyy"))); //$NON-NLS-1$
	}

	private static String convertToBanglaFormat(String admissionDate) {
		if (admissionDate == null) {
			return "";
		}
		return admissionDate.replaceAll("0", "০").replaceAll("1", "১").replaceAll("2", "২").replaceAll("3", "৩").replaceAll("4", "৪").replaceAll("5", "৫")
				.replaceAll("6", "৬").replaceAll("7", "৭").replaceAll("8", "৮").replaceAll("9", "৯").replaceAll("AM", "সকাল").replaceAll("PM", "বিকাল");
	}

	private static String format(String ageFormat, Boolean showReportsInBangla) {
		if (showReportsInBangla) {
			switch (ageFormat) {
				case "Y":
					return " বছর";
				case "M":
					return " মাস";
				case "D":
					return " দিন";
				default:
					break;
			}
		}
		return ageFormat;
	}

	public static String getGenderDisplayString(BookingInfo bookingInfo, boolean showReportsInBangla) {
		String sex = bookingInfo.getPatientSex();
		if (sex == null) {
			return "";
		}
		if (showReportsInBangla) {
			if (sex.equalsIgnoreCase("Male")) { //$NON-NLS-1$
				return "পুরুষ"; //$NON-NLS-1$
			}
			if (sex.equalsIgnoreCase("Female")) { //$NON-NLS-1$
				return "নারী"; //$NON-NLS-1$
			}
		}
		return sex;
	}

	public static PaperSize getPageSize(Map<String, Object> mapForPreview, boolean isPharma, Outlet outlet) {
		PaperSize paperSize;
		if (mapForPreview == null) {
			PaperSize receiptPaperSize;
			if (isPharma) {
				receiptPaperSize = outlet.getPharmacyReceiptPaperSize();
			}
			else {
				receiptPaperSize = outlet.getReceiptPaperSize();
			}
			paperSize = receiptPaperSize;
		}
		else {
			paperSize = (PaperSize) mapForPreview.get("paperSize"); //$NON-NLS-1$
		}
		return paperSize;
	}

	public static String getMoneyIntoWords(String input) {
		MoneyConverters converter = MoneyConverters.ENGLISH_BANKING_MONEY_VALUE;
		return WordUtils.capitalize(converter.asWords(new BigDecimal(input)));
	}

	public static String getDefaultTamplete() {
		String fileName = "AdmissionTemplate.txt";

		try (InputStream inputStream = AdmissionReportUtil.class.getResourceAsStream("/" + fileName);) {
			return IOUtils.toString(inputStream, "utf-8");
		} catch (IOException e) {
			PosLog.error(AdmissionReportUtil.class, e);
		}
		return null;
	}
}
