package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Specimen;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class SpecimenDAO extends BaseSpecimenDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Specimen bean = (Specimen) obj;
		if (bean == null) {
			throw new PosException("Specimen not found!");
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<Specimen> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(Order.asc(Specimen.PROP_NAME));
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public SpecimenDAO() {
	}

	public Specimen findByName(String specimenName) {
		try (Session session = createNewSession()) {
			return findByName(specimenName, session);
		}
	}

	public Specimen findByName(String specimenName, Session session) {
		Criteria criteria = session.createCriteria(Specimen.class);
		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(Specimen.PROP_NAME, specimenName).ignoreCase());
		return (Specimen) criteria.uniqueResult();
	}

	public void findByName(String specimenName, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Specimen.class);
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(specimenName)) {
				criteria.add(Restrictions.ilike(Specimen.PROP_NAME, specimenName, MatchMode.ANYWHERE));
			}
			tableModel.setNumRows(rowCount(criteria));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Specimen.PROP_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public void saveOrUpdateSpecimenList(List<Specimen> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		saveOrUpdateSpecimenList(dataList, updateLastUpdateTime, updateSyncTime, false, false);
	}

	public void saveOrUpdateSpecimenList(List<Specimen> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, boolean saveNewDataOnly,
			boolean forceUpdate) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<Specimen> iterator = dataList.iterator(); iterator.hasNext();) {
				Specimen item = (Specimen) iterator.next();
				Specimen existingItem = get(item.getId());
				if (existingItem != null) {
					if (!forceUpdate && (saveNewDataOnly || !BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime()))) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}
}