package com.floreantpos.mailservices;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;

import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.util.DataProvider;
import com.orocube.common.util.mail.MailAttachment;
import com.sendgrid.Attachments;
import com.sendgrid.Content;
import com.sendgrid.Email;
import com.sendgrid.Mail;
import com.sendgrid.Method;
import com.sendgrid.Request;
import com.sendgrid.Response;
import com.sendgrid.SendGrid;

public class SendGridMailService {

	private static String apiKey;
	private static String senderEmail;

	public static void setApiKey(String apiKey) {
		SendGridMailService.apiKey = apiKey;
	}

	public static void setSenderEmail(String senderEmail) {
		SendGridMailService.senderEmail = senderEmail;
	}

	public static boolean sendMail(String to, String subject, String contentBody) {
		return sendMail(to, subject, contentBody, null, null, null, null);
	}

	public static boolean sendMail(String to, String subject, String contentBody, String contentId, String fileName, byte[] fileData) {
		return sendMail(to, subject, contentBody, contentId, fileName, fileData, "application/pdf");
	}

	public static boolean sendMail(String to, String subject, String contentBody, String contentId, String fileName, byte[] fileData, String contentType) {
		try {
			checkEmailConfig();

			Mail mail = buildMail(to, subject, contentBody);
			if (fileData != null) {
				mail.addAttachments(buildAttachment(fileName, contentId, contentType, fileData));
			}

			return executeSend(mail);
		} catch (Exception e) {
			PosLog.error(SendGridMailService.class, e);
			throw new RuntimeException("Failed to send email", e);
		}
	}

	public static boolean sendMailWithAttachments(String to, String subject, String contentBody, List<MailAttachment> attachments) {
		try {
			checkEmailConfig();

			Mail mail = buildMail(to, subject, contentBody);
			if (attachments != null) {
				for (MailAttachment att : attachments) {
					if (att.getAttachment() != null) {
						mail.addAttachments(buildAttachment(att.getFileName(), att.getFileName(), "application/pdf", att.getAttachment()));
					}
				}
			}

			return executeSend(mail);
		} catch (Exception e) {
			PosLog.error(SendGridMailService.class, e);
			throw new RuntimeException("Failed to send email with attachments", e);
		}
	}

	private static void checkEmailConfig() {
		String senderHost = DataProvider.get().getSenderEmailHost();
		if (StringUtils.isEmpty(senderHost) || StringUtils.isEmpty(apiKey)) {
			throw new PosException("Sendgrid is not configured");
		}
		if (StringUtils.isEmpty(senderEmail)) {
			senderEmail = "noreply@" + senderHost.trim();
		}
	}

	private static Mail buildMail(String to, String subject, String contentBody) {
		Email from = new Email(senderEmail);
		Email toEmail = new Email(to);
		Content content = new Content("text/html", contentBody);
		return new Mail(from, subject, toEmail, content);
	}

	private static Attachments buildAttachment(String fileName, String contentType, String type, byte[] fileData) {
		Attachments attachments = new Attachments();
		Base64 x = new Base64();
		String encodedData = x.encodeAsString(fileData);
		attachments.setContent(encodedData);
		attachments.setType(contentType);
		attachments.setFilename(fileName);
		attachments.setDisposition("attachment");//$NON-NLS-1$ 
		attachments.setContentId(fileName);
		return attachments;
	}

	private static boolean executeSend(Mail mail) throws Exception {

		SendGrid sg = new SendGrid(apiKey);

		Request request = new Request();
		request.setMethod(Method.POST);
		request.setEndpoint("mail/send");
		request.setBody(mail.build());
		Response response = sg.api(request);

		PosLog.debug(SendGridMailService.class, "SendGrid response code: " + response.getStatusCode());
		PosLog.debug(SendGridMailService.class, "SendGrid response headers: " + response.getHeaders());

		PosLog.info(SendGridMailService.class, "mail sent");

		return response.getStatusCode() >= 200 && response.getStatusCode() < 300;
	}

	public static boolean isValidEmail(String email) {
		Pattern validEmailRegExp = Pattern.compile(POSConstants.EMAIL_PATTERN, Pattern.CASE_INSENSITIVE);
		Matcher matcher = validEmailRegExp.matcher(email);
		return matcher.find();
	}
}
