package com.floreantpos.model;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.model.base.BaseCryoStorage;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

public class CryoStorage extends BaseCryoStorage implements TimedModel, PropertyContainer2 {
	private static final long serialVersionUID = 1L;

	private transient JsonObject propertiesContainer;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public CryoStorage () {
    }

	/**
	 * Constructor for primary key
	 */
	public CryoStorage (java.lang.String id) {
		super(id);
	}

	/**
	 * Constructor for required fields
	 */
	public CryoStorage (
		java.lang.String id,
		StorageType storageType,
		java.lang.String name) {
		super(id, storageType, name);
	}

	/*[CONSTRUCTOR MARKER END]*/

	/**
	 * Check if this container is at capacity
	 */
	public boolean isFull() {
		Integer capacity = getCapacity();
		Integer currentCount = getCurrentCount();
		return capacity != null && currentCount != null && currentCount >= capacity;
	}

	/**
	 * Check if this container has available space
	 */
	public boolean hasSpace() {
		return !isFull();
	}

	/**
	 * Get available space count
	 */
	public int getAvailableSpace() {
		Integer capacity = getCapacity();
		Integer currentCount = getCurrentCount();
		if (capacity == null || currentCount == null) return 0;
		return Math.max(0, capacity - currentCount);
	}

	/**
	 * Get utilization percentage
	 */
	public double getUtilizationPercentage() {
		Integer capacity = getCapacity();
		Integer currentCount = getCurrentCount();
		if (capacity == null || capacity == 0 || currentCount == null) return 0.0;
		return (currentCount.doubleValue() / capacity.doubleValue()) * 100.0;
	}

	/**
	 * Check if content type is allowed
	 */
	public boolean isContentTypeAllowed(String contentType) {
		String allowed = getAllowedContentTypes();
		if (StringUtils.isEmpty(allowed)) return true; // No restriction
		return allowed.contains(contentType);
	}

	/**
	 * Check if this is a root container (no parent)
	 */
	public boolean isRoot() {
		return StringUtils.isEmpty(getParentId());
	}

	/**
	 * Check if this is a root level type (LOCATION)
	 */
	public boolean isRootLevel() {
		StorageType type = getStorageType();
		return type != null && type.isRootLevel();
	}

	/**
	 * Check if this is a leaf level type (POSITION - cannot have children)
	 */
	public boolean isLeafLevel() {
		StorageType type = getStorageType();
		return type != null && type.isLeafLevel();
	}

	/**
	 * Get the hierarchical level (1=LOCATION, 2=TANK, 3=CANE, 4=GOBLET, 5=POSITION)
	 */
	public int getLevel() {
		StorageType type = getStorageType();
		return type != null ? type.getLevel() : 0;
	}

	/**
	 * Check if this container can accept a child of the given type
	 */
	public boolean canAcceptChildType(StorageType childType) {
		StorageType type = getStorageType();
		return type != null && type.canBeParentOf(childType);
	}

	/**
	 * Check if this container's type can be a child of the given parent type
	 */
	public boolean canBeChildOfType(StorageType parentType) {
		StorageType type = getStorageType();
		return type != null && type.canBeChildOf(parentType);
	}

	/**
	 * Get what parent type is expected for this container's type
	 */
	public StorageType getExpectedParentType() {
		StorageType type = getStorageType();
		return type != null ? type.getExpectedParentType() : null;
	}

	/**
	 * Check if maintenance is due
	 */
	public boolean isMaintenanceDue() {
		java.util.Date nextMaintenance = getNextMaintenanceDate();
		if (nextMaintenance == null) return false;
		return nextMaintenance.before(new java.util.Date());
	}

	/**
	 * Get full storage path (to be implemented with DAO)
	 * Returns comma-separated path: "Main Lab,TANK-02,CANE-18,GOBLET-3"
	 */
	public String getFullPath() {
		// Implementation would require DAO to traverse parent hierarchy
		// For now, return just the name
		return getName();
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	@Override
	public void setProperties(String jsonProperties) {
		super.setProperties(jsonProperties);
		this.propertiesContainer = null;
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("CryoStorage[");
		sb.append("id=").append(getId());
		sb.append(", type=").append(getStorageType());
		sb.append(", name=").append(getName());
		sb.append(", status=").append(getStatus());
		sb.append(", capacity=").append(getCurrentCount()).append("/").append(getCapacity());
		sb.append("]");
		return sb.toString();
	}
}
