package com.floreantpos.model;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.floreantpos.model.base.BaseCyclePlan;
import com.floreantpos.model.consultation.ProtocolPhaseTemplete;
import com.floreantpos.model.util.DataProvider;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

public class CyclePlan extends BaseCyclePlan implements TimedModel, PropertyContainer2 {
	private static final long serialVersionUID = 1L;

	@JsonIgnore
	private transient JsonObject propertiesContainer;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	private Couple couple;
	@JsonIgnore
	private transient ProtocolTemplate protocolTemplate;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public CyclePlan () {
    }

	/**
	 * Constructor for primary key
	 */
	public CyclePlan (java.lang.String id) {
		super(id);
	}

	/**
	 * Constructor for required fields
	 */
	public CyclePlan (
		java.lang.String id,
		java.lang.String coupleId) {

		super (
			id,
			coupleId);
	}

	/*[CONSTRUCTOR MARKER END]*/

	@JsonIgnore
	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	@Override
	public void setProperties(String jsonProperties) {
		super.setProperties(jsonProperties);
		this.propertiesContainer = null;
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/**
	 * Get the Couple associated with this cycle plan
	 */
	public Couple getCouple() {
		if (this.couple == null) {
			String coupleId = getCoupleId();
			if (StringUtils.isNotBlank(coupleId)) {
				this.couple = (Couple) DataProvider.get().getObjectOf(Couple.class, coupleId);
			}
		}
		return this.couple;
	}

	public void setCouple(Couple couple) {
		this.couple = couple;
		setCoupleId(couple != null ? couple.getId() : null);
	}

	/**
	 * Get the ProtocolTemplate that was used to create this cycle plan (for reference)
	 */
	@JsonIgnore
	public ProtocolTemplate getProtocolTemplate() {
		if (this.protocolTemplate == null) {
			String templateId = getProtocolTemplateId();
			if (StringUtils.isNotBlank(templateId)) {
				this.protocolTemplate = (ProtocolTemplate) DataProvider.get().getObjectOf(ProtocolTemplate.class, templateId);
			}
		}
		return this.protocolTemplate;
	}

	public void setProtocolTemplate(ProtocolTemplate template) {
		this.protocolTemplate = template;
		setProtocolTemplateId(template != null ? template.getId() : null);
		setProtocolName(template != null ? template.getName() : null);
	}

	/**
	 * Get the list of protocol phases as a sorted list
	 */
	@JsonIgnore
	public List<ProtocolPhase> getProtocolPhasesList() {
		java.util.Set<ProtocolPhase> phases = getProtocolPhases();
		if (phases == null || phases.isEmpty()) {
			return new ArrayList<>();
		}
		List<ProtocolPhase> list = new ArrayList<>(phases);
		list.sort((p1, p2) -> {
			Integer order1 = p1.getSortOrder() != null ? p1.getSortOrder() : Integer.MAX_VALUE;
			Integer order2 = p2.getSortOrder() != null ? p2.getSortOrder() : Integer.MAX_VALUE;
			return order1.compareTo(order2);
		});
		return list;
	}

	/**
	 * Initialize cycle plan from a protocol template (deep copy)
	 */
	public void initializeFromTemplate(ProtocolTemplate template) {
		if (template == null) {
			return;
		}

		setProtocolTemplate(template);

		// Deep copy phases from template
		List<ProtocolPhaseTemplete> templatePhases = template.getProtocolPhases();
		if (templatePhases != null && !templatePhases.isEmpty()) {
			java.util.Set<ProtocolPhase> phases = getProtocolPhases();
			if (phases == null) {
				phases = new java.util.HashSet<>();
				setProtocolPhases(phases);
			}
			for (ProtocolPhaseTemplete templatePhase : templatePhases) {
				ProtocolPhase phase = ProtocolPhase.fromTemplate(templatePhase);
				phase.setCyclePlan(this);
				phases.add(phase);
			}
		}

		// Set initial status
		if (StringUtils.isBlank(getCycleStatus())) {
			setCycleStatus(determineInitialStatus());
		}
	}

	/**
	 * Determine initial status based on phases or use default
	 */
	private String determineInitialStatus() {
		List<ProtocolPhase> phases = getProtocolPhasesList();
		if (phases != null && !phases.isEmpty()) {
			// Get first phase name (already sorted by sortOrder)
			return phases.get(0).getPhaseName();
		}
		return "PLANNED"; // Default status
	}

	/**
	 * Get current cycle status, dynamically determined from phases if available
	 */
	public String getCurrentCycleStatus() {
		String status = getCycleStatus();
		if (StringUtils.isNotBlank(status)) {
			return status;
		}

		// Try to determine from phases
		List<ProtocolPhase> phases = getProtocolPhasesList();
		if (phases != null && !phases.isEmpty()) {
			// Find the first in-progress phase, or the last completed phase
			ProtocolPhase inProgressPhase = null;
			ProtocolPhase lastCompletedPhase = null;

			for (ProtocolPhase phase : phases) {
				if ("IN_PROGRESS".equals(phase.getStatus())) {
					inProgressPhase = phase;
					break;
				}
				if ("COMPLETED".equals(phase.getStatus())) {
					lastCompletedPhase = phase;
				}
			}

			if (inProgressPhase != null) {
				return inProgressPhase.getPhaseName();
			}
			if (lastCompletedPhase != null) {
				return lastCompletedPhase.getPhaseName();
			}
		}

		return "PLANNED"; // Default
	}

}
