package com.floreantpos.model;

import java.util.Date;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.floreantpos.model.base.BaseEmbryoTransfer;
import com.floreantpos.model.consultation.TransferOutcome;
import com.floreantpos.model.util.DataProvider;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

public class EmbryoTransfer extends BaseEmbryoTransfer implements TimedModel, PropertyContainer2 {
	private static final long serialVersionUID = 1L;

	@JsonIgnore
	private transient JsonObject propertiesContainer;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	private Couple couple;
	private CyclePlan cyclePlan;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public EmbryoTransfer () {
    }

	/**
	 * Constructor for primary key
	 */
	public EmbryoTransfer (java.lang.String id) {
		super(id);
	}

	/**
	 * Constructor for required fields
	 */
	public EmbryoTransfer (
		java.lang.String id,
		java.lang.String coupleId,
		java.lang.String cyclePlanId) {

		super (
			id,
			coupleId,
			cyclePlanId);
	}

	/*[CONSTRUCTOR MARKER END]*/

	@JsonIgnore
	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	@Override
	public void setProperties(String jsonProperties) {
		super.setProperties(jsonProperties);
		this.propertiesContainer = null;
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/**
	 * Get the Couple associated with this transfer
	 */
	public Couple getCouple() {
		if (this.couple == null) {
			String coupleId = getCoupleId();
			if (StringUtils.isNotBlank(coupleId)) {
				this.couple = (Couple) DataProvider.get().getObjectOf(Couple.class, coupleId);
			}
		}
		return this.couple;
	}

	public void setCouple(Couple couple) {
		this.couple = couple;
		setCoupleId(couple != null ? couple.getId() : null);
	}

	/**
	 * Get the CyclePlan associated with this transfer
	 */
	public CyclePlan getCyclePlan() {
		if (this.cyclePlan == null) {
			String cyclePlanId = getCyclePlanId();
			if (StringUtils.isNotBlank(cyclePlanId)) {
				this.cyclePlan = (CyclePlan) DataProvider.get().getObjectOf(CyclePlan.class, cyclePlanId);
			}
		}
		return this.cyclePlan;
	}

	public void setCyclePlan(CyclePlan cyclePlan) {
		this.cyclePlan = cyclePlan;
		setCyclePlanId(cyclePlan != null ? cyclePlan.getId() : null);
	}

	/**
	 * Get outcome as enum
	 */
	public TransferOutcome getOutcomeEnum() {
		String outcome = getOutcome();
		if (StringUtils.isBlank(outcome)) {
			return TransferOutcome.PENDING;
		}
		try {
			return TransferOutcome.valueOf(outcome);
		} catch (IllegalArgumentException e) {
			return TransferOutcome.PENDING;
		}
	}

	/**
	 * Set outcome from enum
	 */
	public void setOutcomeEnum(TransferOutcome outcome) {
		setOutcome(outcome != null ? outcome.name() : TransferOutcome.PENDING.name());
	}

	/**
	 * Check if transfer is fresh (not frozen)
	 */
	public boolean isFreshTransfer() {
		return "FRESH".equalsIgnoreCase(getTransferType());
	}

	/**
	 * Check if transfer is frozen embryo transfer
	 */
	public boolean isFrozenTransfer() {
		return "FET".equalsIgnoreCase(getTransferType());
	}

	/**
	 * Check if outcome is successful
	 */
	public boolean isSuccessful() {
		return getOutcomeEnum().isSuccessful();
	}

	/**
	 * Check if pregnancy is confirmed
	 */
	public boolean isPregnancyConfirmed() {
		return getOutcomeEnum().isPregnancyConfirmed();
	}

	/**
	 * Check if outcome is final
	 */
	public boolean isFinalOutcome() {
		return getOutcomeEnum().isFinal();
	}

	/**
	 * Calculate beta HCG doubling time (should be ~48 hours for healthy pregnancy)
	 */
	public Double getBetaHcgDoublingTime() {
		Date firstDate = getFirstBetaHcgDate();
		Date secondDate = getSecondBetaHcgDate();
		Double firstValue = getFirstBetaHcgValue();
		Double secondValue = getSecondBetaHcgValue();

		if (firstDate == null || secondDate == null || firstValue == null || secondValue == null) {
			return null;
		}

		if (firstValue <= 0 || secondValue <= 0) {
			return null;
		}

		long timeDiff = secondDate.getTime() - firstDate.getTime();
		double hoursElapsed = timeDiff / (1000.0 * 60 * 60);

		double ratio = secondValue / firstValue;
		if (ratio <= 1) {
			return null; // Not doubling
		}

		// Doubling time formula: T * log(2) / log(ratio)
		return hoursElapsed * Math.log(2) / Math.log(ratio);
	}

	/**
	 * Check if beta HCG is doubling appropriately (< 72 hours is good)
	 */
	public boolean isBetaHcgDoublingNormal() {
		Double doublingTime = getBetaHcgDoublingTime();
		if (doublingTime == null) {
			return false;
		}
		// Normal doubling time is 48-72 hours in early pregnancy
		return doublingTime >= 24 && doublingTime <= 72;
	}

	/**
	 * Get number of days post-transfer
	 */
	public Integer getDaysPostTransfer() {
		if (getTransferDate() == null) {
			return null;
		}
		Date now = new Date();
		long timeDiff = now.getTime() - getTransferDate().getTime();
		return (int) (timeDiff / (1000 * 60 * 60 * 24));
	}


	/**
	 * Get the list of embryos as a list
	 */
	@JsonIgnore
	public java.util.List<Embryo> getEmbryosList() {
		java.util.Set<Embryo> embryos = getEmbryos();
		if (embryos == null || embryos.isEmpty()) {
			return new java.util.ArrayList<>();
		}
		return new java.util.ArrayList<>(embryos);
	}
}
