package com.floreantpos.model;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.floreantpos.model.util.DateUtil;
import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonNull;
import com.google.gson.JsonObject;

public interface PropertyContainer2 {

	@JsonIgnore
	public abstract JsonObject getPropertyStore();

	public abstract String getProperties();

	public abstract void setProperties(String jsonProperties);

	public default boolean hasProperty(String key) {
		JsonObject propertyStore = this.getPropertyStore();
		if (propertyStore == null) {
			return false;
		}
		return propertyStore.has(key);
	}

	public default String getProperty(String key) {
		JsonObject propertyStore = this.getPropertyStore();
		if (propertyStore.has(key)) {
			JsonElement jsonElement = propertyStore.get(key);
			if (jsonElement instanceof JsonNull) {
				return ""; //$NON-NLS-1$
			}
			else if (jsonElement instanceof JsonArray) {
				Gson gson = new Gson();
				return gson.toJson(jsonElement);
			}

			return jsonElement.getAsString();
		}
		return null;
	}

	public default String getProperty(String key, String defaultValue) {
		if (this.getPropertyStore().has(key)) {
			JsonElement jsonElement = this.getPropertyStore().get(key);
			if (jsonElement instanceof JsonNull) {
				return ""; //$NON-NLS-1$
			}
			return jsonElement.getAsString();
		}
		return defaultValue;
	}

	public default Boolean getBooleanProperty(String key, Boolean defaultValue) {
		JsonObject propertyStore = getPropertyStore();
		JsonElement string = propertyStore.get(key);
		if (string == null || string.isJsonNull()) {
			return defaultValue;
		}
		return string.getAsBoolean();
	}

	public default double getDoubleProperty(String key) {
		return parseDouble(getProperty(key));
	}

	public default void addIntProperty(String key, Integer intValue) {
		addProperty(key, String.valueOf(intValue));
	}

	public default int getIntProperty(String key) {
		return (int) parseDouble(getProperty(key));
	}

	public default Date getDateProperty(String key) {
		String dateString = getProperty(key);
		if (StringUtils.isNotBlank(dateString)) {
			try {
				return DateUtil.parseUTCString(dateString);
			} catch (Exception ignored) {
			}
		}
		return null;
	}

	public default void addDateProperty(String key, Date date) {
		if (date == null) {
			addProperty(key, null);
			return;
		}
		addProperty(key, DateUtil.formatDateAsUTCString(date));
	}

	public default void addProperty(String key, String value) {
		this.getPropertyStore().addProperty(key, value);
		setProperties(this.getPropertyStore().toString());
	}

	public default void addJsonObject(String key, JsonObject value) {
		this.getPropertyStore().add(key, value);
		setProperties(this.getPropertyStore().toString());
	}

	public default void addJsonArray(String key, JsonArray value) {
		this.getPropertyStore().add(key, value);
		setProperties(this.getPropertyStore().toString());
	}

	public default JsonObject getJsonObject(String key) {
		return this.getPropertyStore().getAsJsonObject(key);
	}

	public default void removeProperty(String propertyName) {
		this.getPropertyStore().remove(propertyName);
		setProperties(this.getPropertyStore().toString());
	}

	public static double parseDouble(String s) {
		if (StringUtils.isBlank(s)) {
			return 0;
		}

		try {
			return Double.parseDouble(s);
		} catch (Exception x) {
			return 0;
		}
	}

	/**
	 * A generic helper to deserialize a POJO from the property store.
	 * @param key The key in the JSON.
	 * @param clazz The class of the POJO to create.
	 * @param gson 
	 * @return A deserialized object, or a new instance if not found.
	 */
	public default <T> T getObjectProperty(String key, Class<T> clazz, Gson gson) {
		JsonElement element = getPropertyStore().get(key);
		if (element != null && !element.isJsonNull()) {
			return gson.fromJson(element, clazz);
		}
		try {
			// Return a new empty instance to prevent NullPointerExceptions
			return clazz.getDeclaredConstructor().newInstance();
		} catch (Exception e) {
			// This should not happen for simple POJOs
			return null;
		}
	}

	/**
	 * A generic helper to serialize a POJO into the property store.
	 * @param key The key to store the object under in the JSON.
	 * @param object The POJO to serialize.
	 * @param gson 
	 */
	public default void setObjectProperty(String key, Object object, Gson gson) {
		JsonElement jsonElement = gson.toJsonTree(object);
		getPropertyStore().add(key, jsonElement);
		// Persist the entire updated JSON object back to the string property
		setProperties(getPropertyStore().toString());
	}

	/**
	 * A new generic helper to deserialize a List of POJOs from the property store.
	 * @param key The key in the JSON.
	 * @param listType The Type of the list, obtained via TypeToken.
	 * @param gson The Gson instance to use for deserialization.
	 * @return A deserialized List, or a new empty List if not found.
	 */
	public default <T> List<T> getObjectListProperty(String key, java.lang.reflect.Type listType, Gson gson) {
		JsonElement element = getPropertyStore().get(key);
		if (element != null && element.isJsonArray()) {
			return gson.fromJson(element, listType);
		}
		return new java.util.ArrayList<>();
	}

}
