package com.floreantpos.model;

/**
 * Enum representing the types of cryogenic storage containers
 * in the hierarchical storage system.
 *
 * Hierarchy (top to bottom):
 * LOCATION → TANK → CANE → GOBLET → POSITION
 */
public enum StorageType {
	/**
	 * Physical location (e.g., "Main Lab", "Satellite Clinic")
	 * Top level - has no parent
	 */
	LOCATION("Location", 1, null),

	/**
	 * Liquid nitrogen storage tank (Dewar)
	 * Parent: LOCATION
	 */
	TANK("Tank", 2, LOCATION),

	/**
	 * Cane/Rod that hangs inside a tank
	 * Parent: TANK
	 */
	CANE("Cane", 3, TANK),

	/**
	 * Goblet/Cup that holds straws on a cane
	 * Parent: CANE
	 */
	GOBLET("Goblet", 4, CANE),

	/**
	 * Individual position/slot within a goblet
	 * Parent: GOBLET
	 */
	POSITION("Position", 5, GOBLET);

	private final String displayName;
	private final int level;
	private final StorageType expectedParentType;

	StorageType(String displayName, int level, StorageType expectedParentType) {
		this.displayName = displayName;
		this.level = level;
		this.expectedParentType = expectedParentType;
	}

	public String getDisplayName() {
		return displayName;
	}

	/**
	 * Get the hierarchical level (1 = top, 5 = bottom)
	 */
	public int getLevel() {
		return level;
	}

	/**
	 * Get the expected parent type for this storage type
	 * @return parent type, or null if this is a root level (LOCATION)
	 */
	public StorageType getExpectedParentType() {
		return expectedParentType;
	}

	/**
	 * Check if this type can be a parent of the given child type
	 */
	public boolean canBeParentOf(StorageType childType) {
		return childType != null && this.equals(childType.getExpectedParentType());
	}

	/**
	 * Check if this type can be a child of the given parent type
	 */
	public boolean canBeChildOf(StorageType parentType) {
		return this.expectedParentType != null && this.expectedParentType.equals(parentType);
	}

	/**
	 * Check if this is a root level type (has no parent)
	 */
	public boolean isRootLevel() {
		return expectedParentType == null;
	}

	/**
	 * Check if this is a leaf level type (cannot have children)
	 */
	public boolean isLeafLevel() {
		return this == POSITION;
	}

	/**
	 * Convert string value to enum (case-insensitive)
	 */
	public static StorageType fromString(String value) {
		if (value == null) {
			return null;
		}
		for (StorageType type : StorageType.values()) {
			if (type.name().equalsIgnoreCase(value)) {
				return type;
			}
		}
		throw new IllegalArgumentException("Unknown StorageType: " + value);
	}

	@Override
	public String toString() {
		return name();
	}
}
