package com.floreantpos.model.consultation;

/**
 * Enum representing the biopsy/PGT (Preimplantation Genetic Testing) status of an embryo.
 */
public enum BiopsyStatus {
	//@formatter:off
	NOT_BIOPSIED("Not Biopsied", "Embryo has not undergone biopsy"),
	SCHEDULED("Scheduled", "Biopsy is scheduled but not yet performed"),
	BIOPSIED("Biopsied", "Biopsy has been performed, awaiting results"),
	RESULTS_PENDING("Results Pending", "Biopsy completed, results being processed by lab"),
	NORMAL("Normal", "PGT results show normal/euploid embryo"),
	ABNORMAL("Abnormal", "PGT results show chromosomal abnormalities/aneuploid"),
	MOSAIC("Mosaic", "PGT results show mosaic pattern (some normal, some abnormal cells)"),
	INCONCLUSIVE("Inconclusive", "PGT results are inconclusive or no result obtained"),
	FAILED("Failed", "Biopsy or testing failed");
	//@formatter:on

	private final String displayName;
	private final String description;

	BiopsyStatus(String displayName, String description) {
		this.displayName = displayName;
		this.description = description;
	}

	public String getDisplayName() {
		return displayName;
	}

	public String getDescription() {
		return description;
	}

	/**
	 * Check if biopsy has been performed
	 */
	public boolean isBiopsied() {
		return this != NOT_BIOPSIED && this != SCHEDULED;
	}

	/**
	 * Check if results are available
	 */
	public boolean hasResults() {
		return this == NORMAL || this == ABNORMAL ||
		       this == MOSAIC || this == INCONCLUSIVE || this == FAILED;
	}

	/**
	 * Check if embryo is recommended for transfer (normal or mosaic)
	 */
	public boolean isRecommendedForTransfer() {
		return this == NORMAL || this == MOSAIC || this == NOT_BIOPSIED;
	}

	/**
	 * Check if embryo has abnormal results
	 */
	public boolean isAbnormal() {
		return this == ABNORMAL;
	}

	/**
	 * Check if still waiting for results
	 */
	public boolean isPending() {
		return this == BIOPSIED || this == RESULTS_PENDING;
	}

	@Override
	public String toString() {
		return displayName;
	}
}
