package com.floreantpos.model.consultation;

/**
 * Enum representing the consultation status of a couple in the IVF clinic.
 */
public enum ConsultationStatus {
	//@formatter:off
	INITIAL_CONSULTATION("Initial Consultation", "Couple has scheduled or completed initial consultation"),
	WORKUP_IN_PROGRESS("Workup in Progress", "Medical workup and testing underway"),
	WORKUP_COMPLETE("Workup Complete", "All initial testing and evaluation completed"),
	TREATMENT_PLANNING("Treatment Planning", "Reviewing results and planning treatment protocol"),
	READY_FOR_TREATMENT("Ready for Treatment", "Couple is ready to start IVF cycle"),
	IN_TREATMENT("In Treatment", "Currently undergoing active IVF treatment"),
	TREATMENT_PAUSED("Treatment Paused", "Treatment temporarily paused (patient request, medical reasons)"),
	TREATMENT_COMPLETED("Treatment Completed", "Treatment cycle completed, awaiting results"),
	SUCCESSFUL("Successful", "Treatment successful, pregnancy achieved"),
	UNSUCCESSFUL("Unsuccessful", "Treatment unsuccessful, discussing next steps"),
	FOLLOW_UP("Follow-Up", "Post-treatment follow-up care"),
	DISCHARGED("Discharged", "Patient discharged from IVF clinic care"),
	INACTIVE("Inactive", "Couple no longer actively pursuing treatment");
	//@formatter:on

	private final String displayName;
	private final String description;

	ConsultationStatus(String displayName, String description) {
		this.displayName = displayName;
		this.description = description;
	}

	public String getDisplayName() {
		return displayName;
	}

	public String getDescription() {
		return description;
	}

	/**
	 * Check if couple is actively in treatment
	 */
	public boolean isActive() {
		return this == WORKUP_IN_PROGRESS || this == WORKUP_COMPLETE ||
		       this == TREATMENT_PLANNING || this == READY_FOR_TREATMENT ||
		       this == IN_TREATMENT || this == TREATMENT_COMPLETED ||
		       this == FOLLOW_UP;
	}

	/**
	 * Check if initial consultation phase
	 */
	public boolean isInitialPhase() {
		return this == INITIAL_CONSULTATION || this == WORKUP_IN_PROGRESS;
	}

	/**
	 * Check if treatment is ongoing
	 */
	public boolean isTreatmentOngoing() {
		return this == IN_TREATMENT;
	}

	/**
	 * Check if treatment has concluded (success or not)
	 */
	public boolean isConcluded() {
		return this == SUCCESSFUL || this == UNSUCCESSFUL ||
		       this == DISCHARGED || this == INACTIVE;
	}

	@Override
	public String toString() {
		return displayName;
	}
}
