package com.floreantpos.model.consultation;

/**
 * Enum representing the status of an embryo throughout its lifecycle.
 */
public enum EmbryoStatus {
	//@formatter:off
	CREATED("Created", "Embryo just created after fertilization"),
	DEVELOPING("Developing", "Embryo is developing (Day 1-6)"),
	VIABLE("Viable", "Embryo is viable and can be used for transfer or freezing"),
	READY_FOR_TRANSFER("Ready for Transfer", "Embryo is ready for fresh transfer"),
	TRANSFERRED("Transferred", "Embryo has been transferred to uterus"),
	FROZEN("Frozen", "Embryo is cryopreserved in storage"),
	THAWED("Thawed", "Embryo has been thawed for transfer"),
	ARRESTED("Arrested", "Embryo development has stopped"),
	FRAGMENTED("Fragmented", "Embryo is heavily fragmented, poor quality"),
	DISCARDED("Discarded", "Embryo has been discarded (poor quality or patient decision)"),
	BIOPSIED("Biopsied", "Embryo has undergone biopsy for genetic testing"),
	ABNORMAL("Abnormal", "Embryo has abnormal genetic results");
	//@formatter:on

	private final String displayName;
	private final String description;

	EmbryoStatus(String displayName, String description) {
		this.displayName = displayName;
		this.description = description;
	}

	public String getDisplayName() {
		return displayName;
	}

	public String getDescription() {
		return description;
	}

	/**
	 * Check if embryo is viable for use
	 */
	public boolean isViable() {
		return this == VIABLE || this == READY_FOR_TRANSFER ||
		       this == FROZEN || this == THAWED || this == BIOPSIED;
	}

	/**
	 * Check if embryo is currently in storage
	 */
	public boolean isInStorage() {
		return this == FROZEN;
	}

	/**
	 * Check if embryo has been used
	 */
	public boolean isUsed() {
		return this == TRANSFERRED;
	}

	/**
	 * Check if embryo is no longer viable
	 */
	public boolean isNonViable() {
		return this == ARRESTED || this == FRAGMENTED ||
		       this == DISCARDED || this == ABNORMAL;
	}

	/**
	 * Check if embryo is still developing
	 */
	public boolean isDeveloping() {
		return this == CREATED || this == DEVELOPING;
	}

	@Override
	public String toString() {
		return displayName;
	}
}
