package com.floreantpos.model.consultation;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * Represents an actual phase instance in a CyclePlan, deep-copied from ProtocolPhaseTemplete.
 * This is a runtime, trackable version that includes actual dates and completion status.
 *
 * This is a non-persistent Plain Old Java Object (POJO), designed to be
 * part of a list that is serialized into JSON and stored in CyclePlan.
 */
public class ProtocolPhase implements Serializable {
	private static final long serialVersionUID = 1L;

	// Template fields (copied from ProtocolPhaseTemplete)
	private String phaseName;
	private String description;
	private Integer estimatedStartDay;
	private Integer estimatedDurationDays;
	private Integer sortOrder;

	// Runtime tracking fields
	private Date actualStartDate;
	private Date actualEndDate;
	private String status; // "PENDING", "IN_PROGRESS", "COMPLETED", "SKIPPED"

	// Deep copies with tracking
	private List<CycleMedication> medications = new ArrayList<>();
	private List<CycleMonitoringTask> monitoringTasks = new ArrayList<>();

	public ProtocolPhase() {
		this.status = "PENDING";
	}

	/**
	 * Factory method to create a ProtocolPhase from a ProtocolPhaseTemplete (deep copy)
	 */
	public static ProtocolPhase fromTemplate(ProtocolPhaseTemplete template) {
		if (template == null) {
			return null;
		}

		ProtocolPhase phase = new ProtocolPhase();
		phase.setPhaseName(template.getPhaseName());
		phase.setDescription(template.getDescription());
		phase.setEstimatedStartDay(template.getEstimatedStartDay());
		phase.setEstimatedDurationDays(template.getEstimatedDurationDays());
		phase.setSortOrder(template.getSortOrder());

		// Deep copy medications
		if (template.getMedications() != null) {
			for (ProtocolMedication medTemplate : template.getMedications()) {
				phase.getMedications().add(CycleMedication.fromTemplate(medTemplate));
			}
		}

		// Deep copy monitoring tasks
		if (template.getMonitoringTasks() != null) {
			for (ProtocolMonitoringTask taskTemplate : template.getMonitoringTasks()) {
				phase.getMonitoringTasks().add(CycleMonitoringTask.fromTemplate(taskTemplate));
			}
		}

		return phase;
	}

	// Getters and Setters

	public String getPhaseName() {
		return phaseName;
	}

	public void setPhaseName(String phaseName) {
		this.phaseName = phaseName;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public Integer getEstimatedStartDay() {
		return estimatedStartDay;
	}

	public void setEstimatedStartDay(Integer estimatedStartDay) {
		this.estimatedStartDay = estimatedStartDay;
	}

	public Integer getEstimatedDurationDays() {
		return estimatedDurationDays;
	}

	public void setEstimatedDurationDays(Integer estimatedDurationDays) {
		this.estimatedDurationDays = estimatedDurationDays;
	}

	public Integer getSortOrder() {
		return sortOrder;
	}

	public void setSortOrder(Integer sortOrder) {
		this.sortOrder = sortOrder;
	}

	public Date getActualStartDate() {
		return actualStartDate;
	}

	public void setActualStartDate(Date actualStartDate) {
		this.actualStartDate = actualStartDate;
	}

	public Date getActualEndDate() {
		return actualEndDate;
	}

	public void setActualEndDate(Date actualEndDate) {
		this.actualEndDate = actualEndDate;
	}

	public String getStatus() {
		return status;
	}

	public void setStatus(String status) {
		this.status = status;
	}

	public List<CycleMedication> getMedications() {
		return medications;
	}

	public void setMedications(List<CycleMedication> medications) {
		this.medications = medications;
	}

	public List<CycleMonitoringTask> getMonitoringTasks() {
		return monitoringTasks;
	}

	public void setMonitoringTasks(List<CycleMonitoringTask> monitoringTasks) {
		this.monitoringTasks = monitoringTasks;
	}
}
