package com.floreantpos.model.consultation;

/**
 * Enum representing the outcome of an embryo transfer attempt.
 */
public enum TransferOutcome {
	//@formatter:off
	PENDING("Pending", "Waiting for pregnancy test results"),
	POSITIVE("Positive", "Pregnancy test positive, awaiting confirmation"),
	NEGATIVE("Negative", "Pregnancy test negative, transfer failed"),
	BIOCHEMICAL_PREGNANCY("Biochemical Pregnancy", "Positive test but no clinical pregnancy, early miscarriage"),
	CLINICAL_PREGNANCY("Clinical Pregnancy", "Gestational sac and fetal heartbeat confirmed by ultrasound"),
	CLINICAL_PREGNANCY_TWINS("Clinical Pregnancy - Twins", "Two gestational sacs with fetal heartbeats confirmed"),
	CLINICAL_PREGNANCY_TRIPLETS("Clinical Pregnancy - Triplets", "Three gestational sacs with fetal heartbeats confirmed"),
	ONGOING_PREGNANCY("Ongoing Pregnancy", "Pregnancy continuing beyond 12 weeks"),
	LIVE_BIRTH("Live Birth", "Successful delivery of healthy baby"),
	MISCARRIAGE("Miscarriage", "Pregnancy loss after clinical confirmation"),
	ECTOPIC_PREGNANCY("Ectopic Pregnancy", "Pregnancy outside the uterus"),
	CANCELLED("Cancelled", "Transfer attempt was cancelled before procedure");
	//@formatter:on

	private final String displayName;
	private final String description;

	TransferOutcome(String displayName, String description) {
		this.displayName = displayName;
		this.description = description;
	}

	public String getDisplayName() {
		return displayName;
	}

	public String getDescription() {
		return description;
	}

	/**
	 * Check if the outcome is successful (pregnancy achieved)
	 */
	public boolean isSuccessful() {
		return this == POSITIVE || this == CLINICAL_PREGNANCY ||
		       this == CLINICAL_PREGNANCY_TWINS || this == CLINICAL_PREGNANCY_TRIPLETS ||
		       this == ONGOING_PREGNANCY || this == LIVE_BIRTH;
	}

	/**
	 * Check if the outcome is final (no longer pending)
	 */
	public boolean isFinal() {
		return this != PENDING;
	}

	/**
	 * Check if the outcome failed
	 */
	public boolean isFailed() {
		return this == NEGATIVE || this == BIOCHEMICAL_PREGNANCY ||
		       this == MISCARRIAGE || this == ECTOPIC_PREGNANCY ||
		       this == CANCELLED;
	}

	/**
	 * Check if pregnancy is confirmed
	 */
	public boolean isPregnancyConfirmed() {
		return this == CLINICAL_PREGNANCY || this == CLINICAL_PREGNANCY_TWINS ||
		       this == CLINICAL_PREGNANCY_TRIPLETS || this == ONGOING_PREGNANCY ||
		       this == LIVE_BIRTH;
	}

	/**
	 * Check if this is the ultimate success
	 */
	public boolean isLiveBirth() {
		return this == LIVE_BIRTH;
	}

	@Override
	public String toString() {
		return displayName;
	}
}
