package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.BalanceForwardTransaction;
import com.floreantpos.model.BalanceSubType;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.BankAccount;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.CustomPayment;
import com.floreantpos.model.Customer;
import com.floreantpos.model.EndBalanceTransaction;
import com.floreantpos.model.ExpenseTransaction;
import com.floreantpos.model.GiftCard;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PayoutReason;
import com.floreantpos.model.PayoutRecepient;
import com.floreantpos.model.PayoutSubReason;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Project;
import com.floreantpos.model.RecipientType;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.SalaryAdvanceTransaction;
import com.floreantpos.model.SalaryTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TransactionSubType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.util.AccountManagerTransactionSearchCriteria;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.DefaultDataInserter;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

public class BalanceUpdateTransactionDAO extends BaseBalanceUpdateTransactionDAO {

	public BalanceUpdateTransactionDAO() {
	}

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	private void adjustBalanceUpdateTransaction(BalanceUpdateTransaction balanceTransaction, Session session) {
		save(balanceTransaction, session);

		BalanceType balanceType = balanceTransaction.getBalanceType();
		if (balanceType == BalanceType.GIFT_CARD) {
			GiftCard giftCard = GiftCardDAO.getInstance().findByCardNumber(session, balanceTransaction.getAccountNumber());
			if (giftCard == null) {
				//throw new PosException("No gift card found with account number " + balanceTransaction.getAccountNumber()); //$NON-NLS-1$
				return;
			}
			giftCard.setBalance(giftCard.getBalance() + balanceTransaction.getAmount());
			GiftCardDAO.getInstance().saveOrUpdate(giftCard, session);
			return;
		}

		Customer customer = CustomerDAO.getInstance().get(balanceTransaction.getAccountNumber(), session);
		if (customer == null) {
			//throw new PosException("No customer found with id " + balanceTransaction.getAccountNumber()); //$NON-NLS-1$
			return;
		}

		if (balanceType == BalanceType.CUSTOMER) {
			customer.setBalance(customer.getBalance() + balanceTransaction.getAmount());
		}
		else if (balanceType == BalanceType.LOYALTY) {
			customer.setLoyaltyPoint((int) (customer.getLoyaltyPoint() + balanceTransaction.getAmount()));
		}
		customer.setUpdateLastUpdateTime(false);
		customer.setUpdateSyncTime(false);
		CustomerDAO.getInstance().saveOrUpdate(customer, session);
	}

	public void saveBalanceUpdateTrans(BalanceType balanceType, Ticket ticket, PosTransaction transaction, TransactionType transactionType,
			String accountNumber, Double amount, Double balanceBefore, TransactionSubType subType, Session session) {
		saveBalanceUpdateTrans(balanceType, ticket, transaction, transactionType, accountNumber, amount, balanceBefore, subType, null, null, session);

	}

	public void saveBalanceUpdateTrans(BalanceType balanceType, Ticket ticket, PosTransaction transaction, TransactionType transactionType,
			String accountNumber, Double amount, Double balanceBefore, TransactionSubType subType, CustomPayment customPayment, String customPaymnetRef,
			Session session) {

		BalanceUpdateTransaction balanceUpdateTransaction = balanceType.createTransaction(transaction == null ? PaymentType.CASH : transaction.getPaymentType(),
				transactionType, amount);
		if (customPayment != null) {
			balanceUpdateTransaction.setPaymentType(PaymentType.CUSTOM_PAYMENT);
			balanceUpdateTransaction.setCustomPaymnetId(customPayment.getId());
			balanceUpdateTransaction.setCustomPaymnetRef(customPaymnetRef);
		}
		balanceUpdateTransaction.setTransactionId(transaction == null ? null : transaction.getId());
		balanceUpdateTransaction.setTicketId(ticket == null ? null : ticket.getId());
		balanceUpdateTransaction.setAccountNumber(accountNumber);

		if (subType != null) {
			balanceUpdateTransaction.setTransactionSubType(subType.name());
			balanceUpdateTransaction.setDescription(subType.name());
		}
		if (!NumberUtil.isZero(balanceBefore)) {
			balanceBefore = NumberUtil.round(balanceBefore);
			balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		}

		BalanceUpdateTransactionDAO.getInstance().save(balanceUpdateTransaction, session);
	}

	public void saveOrUpdateBalanceUpdateTransactions(List<BalanceUpdateTransaction> customerBalanceTransactions, boolean updateLastUpdateTime,
			boolean updateSyncTime) throws Exception {
		if (customerBalanceTransactions == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (BalanceUpdateTransaction transaction : customerBalanceTransactions) {
				BalanceUpdateTransaction existingTransaction = get(transaction.getId());
				if (existingTransaction != null) {
					continue;
				}
				transaction.setUpdateLastUpdateTime(updateLastUpdateTime);
				transaction.setUpdateSyncTime(updateSyncTime);
				adjustBalanceUpdateTransaction(transaction, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public double getTicketCommissionAmount(Session session, String ticketId, BalanceType balanceType) {
		return getTicketCommissionAmount(session, ticketId, "", balanceType); //$NON-NLS-1$
	}

	public double getTicketCommissionAmount(Session session, String ticketId, String accountNo, BalanceType balanceType) {
		Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
		criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));

		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TICKET_ID, ticketId));
		if (StringUtils.isNotBlank(accountNo)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
		}
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, balanceType.name()));
		double scFromItems = POSUtil.getDoubleAmount(criteria.uniqueResult());

		return scFromItems;
	}

	//	public void findBy(String accountNo, Date beginDate, Date currentDate, PaginationSupport tableModel) {
	//		try (Session session = createNewSession()) {
	//			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
	//			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));
	//			tableModel.setNumRows(rowCount(criteria));
	//
	//			criteria.setFirstResult(tableModel.getCurrentRowIndex());
	//			criteria.setMaxResults(tableModel.getPageSize());
	//			criteria.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
	//			tableModel.setRows(criteria.list());
	//		}
	//	}

	public Map<String, Double> findSummaryBy(String accountNo, Date beginDate, Date currentDate) {
		Map<String, Double> map = new HashMap<String, Double>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
			//criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.groupProperty(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER), "accountNo"); //$NON-NLS-1$
			projectionList.add(Projections.groupProperty(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING), "balanceType"); //$NON-NLS-1$
			projectionList.add(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT), "total"); //$NON-NLS-1$
			criteria.setProjection(projectionList);

			List<Object[]> list = criteria.list();
			for (Object[] obj : list) {
				String balanceTypeString = (String) obj[1];
				double amount = (double) obj[2];
				map.put(balanceTypeString, amount);
			}
		}
		return map;
	}

	public void loadStoreBalancePayment(PaginationSupport model, AccountManagerTransactionSearchCriteria acctMgrTxnCriteria) {
		loadAccountsManagerTransactions(model, acctMgrTxnCriteria);
	}

	public void loadAccountsManagerTransactions(PaginationSupport model, AccountManagerTransactionSearchCriteria acctMgrTxnCriteria) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

			initAccountsManagerPaymentCriteria(acctMgrTxnCriteria, criteria);

			if (StringUtils.isNotBlank(acctMgrTxnCriteria.getOutletId())) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, acctMgrTxnCriteria.getOutletId()));
			}

			if (StringUtils.isNotBlank(acctMgrTxnCriteria.getMemoNo())) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_MEMO_NO, acctMgrTxnCriteria.getMemoNo()));
			}

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			model.setNumRows(rowCount.intValue());

			if (!(acctMgrTxnCriteria.getBalanceType() == BalanceType.STORE_BALANCE)) {
				criteria.addOrder(Order.asc(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER));
			}

			if (StringUtils.isNotBlank(acctMgrTxnCriteria.getMemoNo())) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_MEMO_NO, acctMgrTxnCriteria.getMemoNo()));
			}

			criteria.addOrder(Order.asc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));

			criteria.setProjection(null);
			criteria.setFirstResult(model.getCurrentRowIndex());
			
			if (model.getPageSize() == -1) {
				criteria.setMaxResults(rowCount.intValue());
			}
			else {
				criteria.setMaxResults(model.getPageSize());
			}

			List<BalanceUpdateTransaction> transactions = criteria.list();
			
			//calculate balance forward
			double initialBalanceForStore = 0.0;
			if (acctMgrTxnCriteria.getBalanceType() == BalanceType.STORE_BALANCE) {
				initialBalanceForStore = calculateStoreBalanceBeforeDate(acctMgrTxnCriteria.getFromDate(), acctMgrTxnCriteria, session);
			}
			else {
				initialBalanceForStore = calculateAccountsManagerBalanceBeforeDate(acctMgrTxnCriteria.getFromDate(), acctMgrTxnCriteria, session);
			}
			
			BalanceForwardTransaction forwardTransaction = new BalanceForwardTransaction();
			forwardTransaction.setAmount(initialBalanceForStore);
			forwardTransaction.setTicketId("Balance forward");

			List<BalanceUpdateTransaction> allTransactionList = new ArrayList<BalanceUpdateTransaction>();
			//allTransactionList.add(forwardTransaction);
			
			if (transactions != null) {
				String accountsManagerNumber = null;
				double totalAmount = initialBalanceForStore;
				for (BalanceUpdateTransaction posTransaction : transactions) {
					accountsManagerNumber = posTransaction.getAccountNumber();
					allTransactionList.add(posTransaction);
					totalAmount += posTransaction.getAmount();
				}
				
				if (!acctMgrTxnCriteria.isExpenseOnly()) {
					allTransactionList.add(createEndBalanceByTransction(accountsManagerNumber, totalAmount));
				}
			}
			
			// Calculate cumulative balance for each transaction
			if (allTransactionList != null && !allTransactionList.isEmpty()) {
				double runningBalance = 0.0;
				// For STORE_BALANCE, start with the initial balance calculated before fromDate
				if (acctMgrTxnCriteria.getBalanceType() == BalanceType.STORE_BALANCE) {
					runningBalance = initialBalanceForStore;
				}
				for (BalanceUpdateTransaction transaction : allTransactionList) {
					if (transaction instanceof BalanceForwardTransaction) {
						// BalanceForwardTransaction represents the starting balance (for non-STORE_BALANCE)
						runningBalance = transaction.getAmount() != null ? transaction.getAmount() : 0.0;
					} else if (transaction instanceof EndBalanceTransaction) {
						// EndBalanceTransaction should not have cumulative balance calculated
						// It's a summary row, so skip it
						continue;
					} else {
						// For regular transactions, add the amount
						double amount = transaction.getAmount() != null ? transaction.getAmount() : 0.0;
						runningBalance = runningBalance + amount;
					}
					transaction.setCumulativeBalance(runningBalance);
				}
			}
			
			// Sort transactions by transaction time in descending order (newest first)
			allTransactionList.sort(Comparator.comparing(
				BalanceUpdateTransaction::getTransactionTime,
				Comparator.nullsLast(Comparator.reverseOrder())
			));
			
			model.setRows(allTransactionList);
		}
	}

	public double getTransferToStoreAmount(AccountManagerTransactionSearchCriteria acctMgrTxnCriteria) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

			acctMgrTxnCriteria.setExpenseOnly(false);

			initAccountsManagerPaymentCriteria(acctMgrTxnCriteria, criteria);

			if (acctMgrTxnCriteria.getBalanceSubType() != null) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, acctMgrTxnCriteria.getBalanceSubType().name()));
			}

			criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));
			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null && (uniqueResult instanceof Number)) {
				return ((Number) uniqueResult).doubleValue();
			}

		}
		return 0;
	}

	/**
	 * Calculate the store balance before a given date by summing all STORE_BALANCE transactions
	 * that occurred before the specified date.
	 * 
	 * @param beforeDate The date before which to calculate the balance (exclusive)
	 * @param acctMgrTxnCriteria Search criteria for filtering (used for outlet context)
	 * @param session Hibernate session
	 * @return The calculated balance before the date, or 0.0 if no transactions exist
	 */
	private double calculateStoreBalanceBeforeDate(Date beforeDate, AccountManagerTransactionSearchCriteria acctMgrTxnCriteria, Session session) {
		if (beforeDate == null) {
			return 0.0;
		}

		Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

		// Filter by STORE_BALANCE and ACCOUNTS_MANAGER_BALANCE types to include complete income and expenses
		Disjunction disjunction = Restrictions.disjunction();
		disjunction.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.STORE_BALANCE.name()));
		disjunction.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.ACCOUNTS_MANAGER_BALANCE.name()));
		criteria.add(disjunction);

		// Filter out voided transactions
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));

		// Filter by outletId from current outlet context
		Outlet outlet = DataProvider.get().getOutlet();
		if (outlet != null) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, outlet.getId()));
		}

		// Filter transactions before the specified date (strict less-than to exclude fromDate)
		criteria.add(Restrictions.lt(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, beforeDate));

		// Sum all transaction amounts (amounts are already signed: CREDIT = positive, DEBIT = negative)
		criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));

		Object uniqueResult = criteria.uniqueResult();
		if (uniqueResult != null && (uniqueResult instanceof Number)) {
			return POSUtil.getDoubleAmount(((Number) uniqueResult).doubleValue());
		}

		return 0.0;
	}
	
	/**
	 * Calculate the store balance before a given date by summing all STORE_BALANCE transactions
	 * that occurred before the specified date.
	 * 
	 * @param beforeDate The date before which to calculate the balance (exclusive)
	 * @param acctMgrTxnCriteria Search criteria for filtering (used for outlet context)
	 * @param session Hibernate session
	 * @return The calculated balance before the date, or 0.0 if no transactions exist
	 */
	private double calculateAccountsManagerBalanceBeforeDate(Date beforeDate, AccountManagerTransactionSearchCriteria acctMgrTxnCriteria, Session session) {
		if (beforeDate == null) {
			return 0.0;
		}

		Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

		// Filter by STORE_BALANCE and ACCOUNTS_MANAGER_BALANCE types to include complete income and expenses
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.ACCOUNTS_MANAGER_BALANCE.name()));
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, acctMgrTxnCriteria.getAccountNo()));

		// Filter out voided transactions
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));

		// Filter by outletId from current outlet context
		Outlet outlet = DataProvider.get().getOutlet();
		if (outlet != null) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, outlet.getId()));
		}

		// Filter transactions before the specified date (strict less-than to exclude fromDate)
		criteria.add(Restrictions.lt(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, beforeDate));

		// Sum all transaction amounts (amounts are already signed: CREDIT = positive, DEBIT = negative)
		criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));

		Object uniqueResult = criteria.uniqueResult();
		if (uniqueResult != null && (uniqueResult instanceof Number)) {
			return POSUtil.getDoubleAmount(((Number) uniqueResult).doubleValue());
		}

		return 0.0;
	}

	private void initAccountsManagerPaymentCriteria(AccountManagerTransactionSearchCriteria acctMgrTxnCriteria, Criteria criteria) {

		if (acctMgrTxnCriteria.getBalanceType() != null) {
			// When querying for STORE_BALANCE, also include ACCOUNTS_MANAGER_BALANCE transactions
			// to show complete income and expenses
			if (acctMgrTxnCriteria.getBalanceType() == BalanceType.STORE_BALANCE) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.STORE_BALANCE.name()));
				disjunction.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.ACCOUNTS_MANAGER_BALANCE.name()));
				criteria.add(disjunction);
				//criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, acctMgrTxnCriteria.getBalanceType().name()));

			}
			else {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, acctMgrTxnCriteria.getBalanceType().name()));
			}
		}
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));

		if (acctMgrTxnCriteria.getFromDate() != null && acctMgrTxnCriteria.getToDate() != null) {
			criteria.add(
					Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, acctMgrTxnCriteria.getFromDate(), acctMgrTxnCriteria.getToDate()));
		}
		else {
			if (acctMgrTxnCriteria.getFromDate() != null) {
				//criteria.add(Restrictions.ge(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, acctMgrTxnCriteria.getFromDate()));
				Criterion eventTimeCriterion = Restrictions.ge(PosTransaction.PROP_EVENT_TIME, acctMgrTxnCriteria.getFromDate());
				Criterion transactionTimeCriterion = Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, acctMgrTxnCriteria.getFromDate());

				criteria.add(Restrictions.or(eventTimeCriterion, transactionTimeCriterion));
			}
			if (acctMgrTxnCriteria.getToDate() != null) {
				//criteria.add(Restrictions.lt(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, acctMgrTxnCriteria.getToDate()));

				Criterion eventTimeCriterion = Restrictions.lt(PosTransaction.PROP_EVENT_TIME, acctMgrTxnCriteria.getToDate());
				Criterion transactionTimeCriterion = Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, acctMgrTxnCriteria.getToDate());

				criteria.add(Restrictions.or(eventTimeCriterion, transactionTimeCriterion));
			}
		}

		if (StringUtils.isNotBlank(acctMgrTxnCriteria.getAccountNo())) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, acctMgrTxnCriteria.getAccountNo()));
		}

		if (acctMgrTxnCriteria.getTransactionType() != null) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_TYPE, acctMgrTxnCriteria.getTransactionType().name()));
		}

		if (acctMgrTxnCriteria.isExpenseOnly()) {
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, BalanceSubType.EXPENSE.name()));
			disjunction.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, BalanceSubType.EXPENSE_REFUND.name()));
			disjunction.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, BalanceSubType.EXPENSE_REVERT.name()));
			criteria.add(disjunction);
		}

		if (StringUtils.isNotBlank(acctMgrTxnCriteria.getRecipientId())) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_RECEPIENT_ID, acctMgrTxnCriteria.getRecipientId()));
		}

		if (StringUtils.isNotBlank(acctMgrTxnCriteria.getReasonId())) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_REASON_ID, acctMgrTxnCriteria.getReasonId()));
		}

		if (StringUtils.isNotBlank(acctMgrTxnCriteria.getProjectId())) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_PROJECT_ID, acctMgrTxnCriteria.getProjectId()));
		}

		if (StringUtils.isNotBlank(acctMgrTxnCriteria.getSubCategoryId())) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_SUB_REASON_ID, acctMgrTxnCriteria.getSubCategoryId()));
		}

		if (StringUtils.isNotBlank(acctMgrTxnCriteria.getPerformById())) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_PERFORMER_ID, acctMgrTxnCriteria.getPerformById()));
		}
	}

	public double findTotalExpanses(AccountManagerTransactionSearchCriteria acctMgrTxnCriteria) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			acctMgrTxnCriteria.setExpenseOnly(true);
			initAccountsManagerPaymentCriteria(acctMgrTxnCriteria, criteria);

			criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));
			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null && (uniqueResult instanceof Number)) {
				return ((Number) uniqueResult).doubleValue();
			}
		}
		return 0;
	}

	private BalanceUpdateTransaction createForwardBalanceByTransction(String accountsManagerNumber, BalanceUpdateTransaction balanceUpdateTransaction) {
		return createForwardBalanceByTransction(accountsManagerNumber, balanceUpdateTransaction, null);
	}

	private BalanceUpdateTransaction createForwardBalanceByTransction(String accountsManagerNumber, BalanceUpdateTransaction balanceUpdateTransaction,
			Double calculatedBalance) {
		BalanceForwardTransaction forwardTransaction = new BalanceForwardTransaction();
		if (balanceUpdateTransaction != null) {
			// Use calculated balance if provided, otherwise fall back to transaction's balanceBefore
			double balanceAmount = (calculatedBalance != null) ? calculatedBalance : balanceUpdateTransaction.getBalanceBefore();
			forwardTransaction.setAmount(balanceAmount);
			forwardTransaction.setPaymentTypeString(balanceUpdateTransaction.getPaymentTypeString());
			forwardTransaction.setTransactionType(balanceUpdateTransaction.getTransactionType());
			forwardTransaction.setAccountNumber(accountsManagerNumber);
			forwardTransaction.setOutletId(balanceUpdateTransaction.getOutletId());
			//forwardTransaction.setFromCashdrawerId(balanceUpdateTransaction.getFromCashdrawerId());
		}
		forwardTransaction.setTicketId("Balance forward");
		return forwardTransaction;
	}

	private BalanceUpdateTransaction createEndBalanceByTransction(String accountsManagerNumber, double totalAmount) {
		EndBalanceTransaction endBalanceTransaction = new EndBalanceTransaction();
		endBalanceTransaction.setAccountNumber(accountsManagerNumber);
		endBalanceTransaction.setAmount(totalAmount);
		endBalanceTransaction.setTicketId("End Balance");
		return endBalanceTransaction;
	}

	//	public List<BalanceUpdateTransaction> findPaymentsBy(Date startTime, Date endTime, Customer customer, boolean showAgentPayment) {
	//		try (Session session = createNewSession()) {
	//			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
	//
	//			if (customer != null) {
	//				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, customer.getId()));
	//			}
	//
	//			if (showAgentPayment) {
	//				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
	//			}
	//			else {
	//				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.LAB_DOCTOR_FEE_PAYMENT.name()));
	//			}
	//			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(startTime), DateUtil.endOfDay(endTime)));
	//			criteria.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
	//			return criteria.list();
	//		}
	//	}

	public BalanceUpdateTransaction saveAccountsManagerAddBalanceTransaction(User accountsManager, Double balanceToAdd, String note) {
		return saveAccountsManagerAddBalanceTransaction(accountsManager, balanceToAdd, note, null, "", null);
	}

	public BalanceUpdateTransaction saveAccountsManagerAddBalanceTransaction(User accountsManager, Double balanceToAdd, String note, Object paymentMethod,
			String customPaymentInfo, BankAccount linkedWithBankAccount) {
		return saveAccountsManagerAddBalanceTransaction(accountsManager, balanceToAdd, note, paymentMethod, customPaymentInfo, linkedWithBankAccount, false);
	}

	public BalanceUpdateTransaction saveAccountsManagerAddBalanceTransaction(User accountsManager, Double balanceToAdd, String note, Object paymentMethod,
			String customPaymentInfo, BankAccount linkedWithBankAccount, boolean isMedlogics) {
		Transaction tx = null;
		BalanceUpdateTransaction transaction;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			PaymentType paymentType = getPaymentType(paymentMethod);
			BalanceUpdateTransaction balanceUpdateTransaction = null;
			BalanceSubType balanceSubType = BalanceSubType.ADD_BALANCE;
			if (paymentMethod instanceof BankAccount || linkedWithBankAccount != null) {
				BankAccount bankAccount = linkedWithBankAccount;
				if (bankAccount == null) {
					bankAccount = (BankAccount) paymentMethod;
				}
				balanceSubType = BalanceSubType.TRANSFER_IN;
				balanceUpdateTransaction = saveBankAccountToCashInTransaction(session, paymentType, bankAccount, note, balanceToAdd, accountsManager);
			}
			transaction = populateAccountsManagerTransaction(null, accountsManager, balanceToAdd, note, null, session, balanceSubType, null, paymentType,
					customPaymentInfo);
			transaction.putBalanceTransferInfo(null, accountsManager, balanceToAdd, linkedWithBankAccount);

			if (balanceUpdateTransaction != null) {
				if (isMedlogics) {
					LedgerEntryDAO.getInstance().saveBankAccountToCashInLedgerEntryForMed(balanceUpdateTransaction, session);
				}
				else {
					LedgerEntryDAO.getInstance().saveBankAccountToCashInLedgerEntryForByz(balanceUpdateTransaction, session);
				}
			}
			else {
				if (isMedlogics) {
					LedgerEntryDAO.getInstance().saveCashDepositLedgerEntryForMed(transaction, session);
				}
				else {
					LedgerEntryDAO.getInstance().saveCashDepositLedgerEntryForByz(transaction, session);
				}
			}

			tx.commit();
		}

		return transaction;
	}

	public BalanceUpdateTransaction saveBankAccountToCashInTransaction(Session session, PaymentType paymentType, BankAccount bankAccount, String note,
			double cashInAmmount, User toAccountsManager) {

		String bankAccountId = bankAccount.getId();

		double balanceBefore = bankAccount.getBalance();

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.BANK_ACCOUNT_BALANCE.createTransaction(paymentType, TransactionType.DEBIT,
				cashInAmmount);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.TRANSFER_OUT.name());

		balanceUpdateTransaction.setAccountNumber(bankAccountId);
		balanceUpdateTransaction.setDescription("Transferred to account owner");
		balanceUpdateTransaction.putTransactionReason("Transferred to account owner");
		balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		balanceUpdateTransaction.putBalanceTransferInfo(null, toAccountsManager, cashInAmmount, bankAccount);
		balanceUpdateTransaction.putNote(note);
		bankAccount.setBalance(NumberUtil.round(balanceBefore - cashInAmmount));

		save(balanceUpdateTransaction, session);
		BankAccountDAO.getInstance().saveOrUpdate(bankAccount, session);

		return balanceUpdateTransaction;
	}

	public BalanceUpdateTransaction populateAccountsManagerTransaction(CashDrawer cashDrawer, User accountsManager, Session session) {
		return populateAccountsManagerTransaction(cashDrawer, accountsManager, cashDrawer.getDrawerAccountable(), null, session);
	}

	public BalanceUpdateTransaction populateAccountsManagerTransaction(CashDrawer cashDrawer, User accountsManager, Double balanceToAdd, String note,
			Session session) {
		return populateAccountsManagerTransaction(cashDrawer, accountsManager, balanceToAdd, note, null, session);
	}

	public BalanceUpdateTransaction populateAccountsManagerTransaction(CashDrawer cashDrawer, User accountsManager, Double balanceToAdd, String note,
			String projectId, Session session) {
		return populateAccountsManagerTransaction(cashDrawer, accountsManager, balanceToAdd, note, projectId, session, BalanceSubType.ADD_BALANCE, null);
	}

	public BalanceUpdateTransaction populateAccountsManagerTransaction(CashDrawer cashDrawer, User accountsManager, Double balanceToAdd, String note,
			String projectId, Session session, BalanceSubType balanceSubType, PosTransaction sourceTransaction) {
		return populateAccountsManagerTransaction(cashDrawer, accountsManager, balanceToAdd, note, projectId, session, balanceSubType, sourceTransaction, null,
				"");
	}

	public BalanceUpdateTransaction populateAccountsManagerTransaction(CashDrawer cashDrawer, User accountsManager, Double balanceToAdd, String note,
			String projectId, Session session, BalanceSubType balanceSubType, PosTransaction sourceTransaction, PaymentType paymentType,
			String customPaymentInfo) {
		if (accountsManager == null) {
			return null;
		}
		UserDAO userDao = UserDAO.getInstance();
		userDao.refresh(accountsManager, session);

		PaymentType selectedPaymentType = PaymentType.CASH;
		if (paymentType != null) {
			selectedPaymentType = paymentType;
		}
		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.ACCOUNTS_MANAGER_BALANCE.createTransaction(selectedPaymentType, TransactionType.CREDIT,
				balanceToAdd);
		if (balanceSubType != null) {
			balanceUpdateTransaction.setTransactionSubType(balanceSubType.name());
		}

		balanceUpdateTransaction.setAccountNumber(accountsManager.getId());
		balanceUpdateTransaction.setOutletId(accountsManager.getOutletId());
		if (cashDrawer != null) {
			balanceUpdateTransaction.setFromCashdrawerId(cashDrawer.getId());
		}
		if (StringUtils.isNotBlank(note)) {
			balanceUpdateTransaction.setDescription(note);
		}
		if (StringUtils.isNotBlank(projectId)) {
			balanceUpdateTransaction.setProjectId(projectId);
		}

		if (sourceTransaction != null) {
			balanceUpdateTransaction.setReasonId(sourceTransaction.getReasonId());

			balanceUpdateTransaction.putSourceId(sourceTransaction.getId());
			balanceUpdateTransaction.putSourceType(PosTransaction.REF);
		}
		if (StringUtils.isNotBlank(customPaymentInfo)) {
			balanceUpdateTransaction.putCustomePaymentInfo(customPaymentInfo);
		}

		Double balanceBefore = accountsManager.getAccountsManagerAmount();
		accountsManager.setAccountsManagerAmount(NumberUtil.round(balanceBefore + balanceToAdd));

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		balanceUpdateTransaction
				.setDescription("transferred " + balanceToAdd + " to ACM account " + accountsManager.getFullName() + "from drawer " + cashDrawer.getId());

		userDao.saveOrUpdate(accountsManager, session);
		save(balanceUpdateTransaction, session);

		return balanceUpdateTransaction;
	}

	public void saveAccountsManagerExpenses(String batchNo, Date transactionTime, User accountsManager, User currentUser, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note) throws Exception {
		saveAccountsManagerExpenses(batchNo, transactionTime, accountsManager, currentUser, project, recepient, reason, subReason, expensesAmount, null, null,
				note);
	}

	public void saveAccountsManagerExpenses(String batchNo, Date transactionTime, User accountsManager, User currentUser, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String memoNo, String memoImageId, String note)
			throws Exception {
		saveAccountsManagerExpenses(batchNo, transactionTime, accountsManager, currentUser, project, recepient, reason, subReason, expensesAmount, memoNo,
				memoImageId, note, null);
	}

	public void saveAccountsManagerExpenses(String batchNo, Date transactionTime, User accountsManager, User currentUser, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String memoNo, String memoImageId, String note,
			String requisitionId) throws Exception {
		saveAccountsManagerExpenses(batchNo, transactionTime, accountsManager, currentUser, project, recepient, reason, subReason, expensesAmount, memoNo,
				memoImageId, note, requisitionId, null);
	}

	public void saveAccountsManagerExpenses(String batchNo, Date transactionTime, User accountsManager, User currentUser, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String memoNo, String memoImageId, String note,
			String requisitionId, ChartOfAccounts chartOfAccounts) throws Exception {
		Transaction tx = null;
		try (Session session = createNewSession()) {

			tx = session.beginTransaction();
			String projectId = project == null ? null : project.getId();
			String coaCode = chartOfAccounts == null ? "" : chartOfAccounts.getAccountCode();

			BalanceUpdateTransaction balanceUpdateTransaction = saveAccountsManagerExpensesTransaction(transactionTime, accountsManager, batchNo, projectId,
					recepient, reason, subReason, expensesAmount, note, session);
			balanceUpdateTransaction.setMemoNo(memoNo);
			balanceUpdateTransaction.putMemoImageId(memoImageId);
			balanceUpdateTransaction.putRequisitionId(requisitionId);
			
			balanceUpdateTransaction.addProperty(BalanceUpdateTransaction.EXPENSE_BY_ID, accountsManager.getId());
			balanceUpdateTransaction.addProperty(BalanceUpdateTransaction.EXPENSE_RECIPIENT_ID, recepient.getId());
			balanceUpdateTransaction.addProperty(BalanceUpdateTransaction.EXPENSE_BY_NAME, accountsManager.getFullName());
			balanceUpdateTransaction.addProperty(BalanceUpdateTransaction.EXPENSE_RECIPIENT_NAME, recepient.getName());

			balanceUpdateTransaction.setDescription("Expense by: " + accountsManager.getFullName());

			ExpenseTransaction expenseTransaction = ExpenseTransactionDAO.getInstance().saveExpensesTransaction(currentUser, transactionTime, accountsManager,
					batchNo, project, recepient, reason, subReason, expensesAmount, note, balanceUpdateTransaction, coaCode, session);

			expenseTransaction.setRequisitionId(requisitionId);
			balanceUpdateTransaction.putReferenceId(expenseTransaction.getId());
			balanceUpdateTransaction.setAccountProcessed(true);
			balanceUpdateTransaction.setRecipientType(RecipientType.RECIPIENT.name());

			update(balanceUpdateTransaction, session);

			LedgerEntryDAO.getInstance().saveExpensesLedgerEntry(balanceUpdateTransaction, true, coaCode, session);

			tx.commit();
		}
	}

	public BalanceUpdateTransaction saveAccountsManagerExpensesTransaction(Date eventTime, User accountsManager, String batchNo, String projectId,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note, Session session) {
		return saveAccountsManagerExpensesTransaction(eventTime, accountsManager, batchNo, projectId, recepient, reason, subReason, expensesAmount, note, null,
				session);
	}

	public BalanceUpdateTransaction saveAccountsManagerExpensesTransaction(Date eventTime, User accountsManager, String batchNo, String projectId,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note, InventoryVendor vendor,
			Session session) {
		return saveAccountsManagerExpensesTransaction(eventTime, accountsManager, batchNo, projectId, recepient, reason, subReason, expensesAmount, note,
				vendor, session, false);
	}

	public BalanceUpdateTransaction saveAccountsManagerExpensesTransaction(Date eventTime, User accountsManager, String batchNo, String projectId,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note, InventoryVendor vendor,
			Session session, boolean isRevertTransaction) {
		return saveAccountsManagerExpensesTransaction(eventTime, accountsManager, batchNo, projectId, recepient, reason, subReason, expensesAmount, note,
				vendor, session, isRevertTransaction, null);
	}

	public BalanceUpdateTransaction saveAccountsManagerExpensesTransaction(Date eventTime, User accountsManager, String batchNo, String projectId,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note, InventoryVendor vendor,
			Session session, boolean isRevertTransaction, PosTransaction sourceTransaction) {
		BalanceUpdateTransaction balanceUpdateTransaction;
		UserDAO userInstance = UserDAO.getInstance();
		if (accountsManager != null) {
			userInstance.refresh(accountsManager, session);
		}

		double userCashBalance = UserDAO.getInstance().getUserCashBalance(accountsManager, session);
		if (!isRevertTransaction && userCashBalance < expensesAmount) {
			throw new PosException("There is not enough balance in accounts manager.");
		}

		BalanceType balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
		balanceUpdateTransaction = balanceType.createTransaction(PaymentType.CASH, isRevertTransaction ? TransactionType.CREDIT : TransactionType.DEBIT,
				expensesAmount);
		balanceUpdateTransaction.setTransactionSubType(isRevertTransaction ? BalanceSubType.EXPENSE_REVERT.name() : BalanceSubType.EXPENSE.name());
		if (eventTime != null) {
			balanceUpdateTransaction.setEventTime(eventTime);
		}

		balanceUpdateTransaction.setBatchNo(batchNo);
		if (StringUtils.isNotBlank(projectId)) {
			balanceUpdateTransaction.setProjectId(projectId);
		}
		if (vendor != null) {
			balanceUpdateTransaction.putVendorId(vendor.getId());
			balanceUpdateTransaction.putVendorName(vendor.getName());
			balanceUpdateTransaction.setRecepientId(vendor.getId());
			balanceUpdateTransaction.setRecipientType(RecipientType.VENDOR.name());
		}
		balanceUpdateTransaction.setRecepientId(recepient == null ? "" : recepient.getId());
		balanceUpdateTransaction.setReasonId(reason == null ? "" : reason.getId());
		if (subReason != null) {
			balanceUpdateTransaction.setSubReasonId(subReason.getId());
		}
		balanceUpdateTransaction.setDescription(note);
		Double balanceBefore = 0d;

		if (accountsManager != null) {
			balanceUpdateTransaction.setAccountNumber(accountsManager.getId());
			balanceBefore = userCashBalance;

			double userNewBalance = isRevertTransaction ? balanceBefore + expensesAmount : balanceBefore - expensesAmount;
			accountsManager.setAccountsManagerAmount(userNewBalance);

			UserDAO.getInstance().updateUserCashBalance(session, accountsManager, userNewBalance);
		}

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);

		if (sourceTransaction != null) {
			balanceUpdateTransaction.putSourceId(sourceTransaction.getId());
			balanceUpdateTransaction.putSourceType(PosTransaction.REF);
		}

		save(balanceUpdateTransaction, session);
		return balanceUpdateTransaction;
	}

	public void saveStoreExpenses(String batchNo, Date transactionTime, Outlet outlet, User currentUser, Project project, PayoutRecepient recepient,
			PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note) throws Exception {
		saveStoreExpenses(batchNo, transactionTime, outlet, currentUser, project, recepient, reason, subReason, expensesAmount, note, null, null);
	}

	public void saveStoreExpenses(String batchNo, Date transactionTime, Outlet outlet, User currentUser, Project project, PayoutRecepient recepient,
			PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note, String memoNo, ChartOfAccounts chartOfAccounts)
			throws Exception {

		try (Session session = createNewSession()) {
			Transaction tx = session.beginTransaction();

			String coaCode = chartOfAccounts == null ? "" : chartOfAccounts.getAccountCode();

			BalanceUpdateTransaction balanceUpdateTransaction = saveStoreExpensesTransaction(transactionTime, outlet, batchNo, project, recepient, reason,
					subReason, expensesAmount, note, session, false);
			ExpenseTransaction expenseTransaction = ExpenseTransactionDAO.getInstance().saveStoreExpensesTransaction(currentUser, transactionTime, outlet,
					batchNo, project, recepient, reason, subReason, expensesAmount, note, balanceUpdateTransaction, memoNo, coaCode, session);

			balanceUpdateTransaction.putReferenceId(expenseTransaction.getId());
			balanceUpdateTransaction.setRecipientType(RecipientType.RECIPIENT.name());
			balanceUpdateTransaction.setAccountProcessed(true);
			if (StringUtils.isNotBlank(memoNo)) {
				balanceUpdateTransaction.setMemoNo(memoNo);
			}

			update(balanceUpdateTransaction, session);

			LedgerEntryDAO.getInstance().saveExpensesLedgerEntry(balanceUpdateTransaction, false, coaCode, session);

			tx.commit();
		}
	}

	public BalanceUpdateTransaction saveStoreExpensesTransaction(Date eventTime, Outlet outlet, String batchNo, Project project, PayoutRecepient recepient,
			PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note, Session session, boolean isRevertTransaction) {

		Double outletCashBeforeBalance = StoreBalanceDAO.getInstance().getOutletCashBalance(outlet, session);
		if (outletCashBeforeBalance < expensesAmount) {
			throw new PosException("There is not enough balance in store.");
		}

		BalanceUpdateTransaction balanceUpdateTransaction;
		BalanceType balanceType = BalanceType.STORE_BALANCE;
		balanceUpdateTransaction = balanceType.createTransaction(PaymentType.CASH, isRevertTransaction ? TransactionType.CREDIT : TransactionType.DEBIT,
				expensesAmount);
		balanceUpdateTransaction.setTransactionSubType(isRevertTransaction ? BalanceSubType.EXPENSE_REVERT.name() : BalanceSubType.EXPENSE.name());
		if (eventTime != null) {
			balanceUpdateTransaction.setEventTime(eventTime);
		}

		balanceUpdateTransaction.setAccountNumber(outlet.getId());
		balanceUpdateTransaction.setBatchNo(batchNo);
		if (project != null) {
			balanceUpdateTransaction.setProjectId(project.getId());
		}
		balanceUpdateTransaction.setRecepientId(recepient == null ? "" : recepient.getId());
		balanceUpdateTransaction.setReasonId(reason == null ? "" : reason.getId());
		if (subReason != null) {
			balanceUpdateTransaction.setSubReasonId(subReason.getId());
		}
		balanceUpdateTransaction.setDescription(note);

		double outletCashBalance = NumberUtil.round(isRevertTransaction ? outletCashBeforeBalance + expensesAmount : outletCashBeforeBalance - expensesAmount);
		StoreBalanceDAO.getInstance().updateOutletCashBalance(session, outlet, outletCashBalance);

		balanceUpdateTransaction.setBalanceBefore(outletCashBeforeBalance);
		save(balanceUpdateTransaction, session);
		return balanceUpdateTransaction;
	}

	public BalanceUpdateTransaction saveStoreToCashDrawerTransaction(CashDrawer cashDrawer, Double balanceToAdd, Session session) throws Exception {

		//Store store = DataProvider.get().getStore();
		//Double storePreviousBalance = StoreDAO.getInstance().getStoreCashBalance(store, session);
		Outlet outlet = DataProvider.get().getOutlet();
		Double outletPreviousBalance = StoreBalanceDAO.getInstance().getOutletCashBalance(outlet, session);

		double storeCurrentBalance = NumberUtil.round(outletPreviousBalance - balanceToAdd);
		StoreBalanceDAO.getInstance().updateOutletCashBalance(session, outlet, storeCurrentBalance);

		BalanceUpdateTransaction depositToStore = BalanceType.STORE_BALANCE.createTransaction(PaymentType.CASH, TransactionType.DEBIT, balanceToAdd);
		depositToStore.setTransactionSubType(BalanceSubType.TRANSFER_OUT.name());
		depositToStore.setRecepientId(cashDrawer.getId());
		depositToStore.setReasonId(DefaultDataInserter.addOrGetStoreReason(true).getId());
		depositToStore.setBalanceBefore(outletPreviousBalance);
		depositToStore.setAccountNumber(outlet.getId());
		depositToStore.setDescription("Amount transferred from store to drawer");
		save(depositToStore, session);

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.CASH_DRAWER_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT,
				balanceToAdd);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.BEGIN_CASH_DEPOSIT.name());
		balanceUpdateTransaction.setAccountNumber(cashDrawer.getId());
		balanceUpdateTransaction.setDescription("Add balance to drawer from store");
		balanceUpdateTransaction.setAccountProcessed(true);
		save(balanceUpdateTransaction, session);

		return depositToStore;
	}

	public BalanceUpdateTransaction saveCashDrawerToStoreTransaction(CashDrawer cashDrawer, Double balanceToAdd, Session session) throws Exception {

		//Store store = DataProvider.get().getStore();
		//Double storeCashBeforeBalance = StoreDAO.getInstance().getStoreCashBalance(store, session);
		Outlet outlet = DataProvider.get().getOutlet();
		Double outletCashBeforeBalance = StoreBalanceDAO.getInstance().getOutletCashBalance(outlet, session);

		double outletCashBalance = NumberUtil.round(outletCashBeforeBalance + balanceToAdd);
		StoreBalanceDAO.getInstance().updateOutletCashBalance(session, outlet, outletCashBalance);

		BalanceUpdateTransaction depositToStore = BalanceType.STORE_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT, balanceToAdd);
		depositToStore.setTransactionSubType(BalanceSubType.ADD_BALANCE.name());
		depositToStore.setRecepientId(DefaultDataInserter.addOrGetStoreRecepient().getId());
		depositToStore.setReasonId(DefaultDataInserter.addOrGetStoreReason(true).getId());
		depositToStore.setBalanceBefore(outletCashBeforeBalance);
		depositToStore.setAccountNumber(cashDrawer.getId());
		if (cashDrawer != null) {
			depositToStore.setFromCashdrawerId(cashDrawer.getId());
		}
		depositToStore.setDescription("transferred " + balanceToAdd + " to store account from drawer " + cashDrawer.getId());
		save(depositToStore, session);

		return depositToStore;
	}

	/**
	 * Transfer balance from store to accounts manager
	 */
	public void saveStoreToAccManagerTransaction(Outlet outlet, User accountsManager, Double balanceToTransfer, String note) throws Exception {

		if (accountsManager == null) {
			throw new PosException("Please select a accounts manager");
		}

		BalanceUpdateTransaction storeTransaction;
		BalanceUpdateTransaction acmTransaction;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			GenericDAO genericDAO = GenericDAO.getInstance();
			genericDAO.refresh(accountsManager, session);

			Double storeCashBeforeBalance = StoreBalanceDAO.getInstance().getOutletCashBalance(outlet, session);
			Double accountsManagerBalanceBefore = accountsManager.getAccountsManagerAmount();

			// Check if store has enough balance
			if (storeCashBeforeBalance < balanceToTransfer) {
				throw new PosException("There is not enough balance in store.");
			}

			// Create store debit transaction (transfer out from store)
			storeTransaction = BalanceType.STORE_BALANCE.createTransaction(PaymentType.CASH, TransactionType.DEBIT, balanceToTransfer);
			storeTransaction.setTransactionSubType(BalanceSubType.TRANSFER_OUT.name());
			storeTransaction.setAccountNumber(outlet.getId());
			storeTransaction.setReasonId("TRANSFER_OUT"); //$NON-NLS-1$
			storeTransaction.setBalanceBefore(storeCashBeforeBalance);
			//storeTransaction.setDescription("Transfer to accounts manager: " + accountsManager.getFullName() + ". " + note);

			storeTransaction.setDescription("Transferred " + balanceToTransfer + " from store account to ACM " + accountsManager.getFullName());
			storeTransaction.putNote(note);

			storeTransaction.setAccountProcessed(true);
			storeTransaction.setRecipientType(RecipientType.USER.name());
			storeTransaction.setRecepientId(accountsManager.getId());
			storeTransaction.putBalanceTransferInfo(null, accountsManager, balanceToTransfer);

			// Update store balance
			double storeCashBalance = NumberUtil.round(storeCashBeforeBalance - balanceToTransfer);
			StoreBalanceDAO.getInstance().updateOutletCashBalance(session, outlet, storeCashBalance);

			save(storeTransaction, session);

			// Create accounts manager credit transaction (transfer in to ACM)
			acmTransaction = BalanceType.ACCOUNTS_MANAGER_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT, balanceToTransfer);
			acmTransaction.setTransactionSubType(BalanceSubType.TRANSFER_IN.name());
			acmTransaction.setAccountNumber(accountsManager.getId());
			acmTransaction.setBalanceBefore(accountsManagerBalanceBefore);
			acmTransaction.setDescription("Received " + balanceToTransfer + " from store account to ACM " + accountsManager.getFullName());
			acmTransaction.putNote(note);

			acmTransaction.setAccountProcessed(true);

			acmTransaction.setFromAccountId(outlet.getId());
			acmTransaction.setFromAccountType(RecipientType.STORE.name());

			// Update accounts manager balance
			accountsManager.setAccountsManagerAmount(NumberUtil.round(accountsManagerBalanceBefore + balanceToTransfer));
			genericDAO.saveOrUpdate(accountsManager, session);

			save(acmTransaction, session);

			tx.commit();
		}
	}

	public BalanceUpdateTransaction saveAccManagerToCashDrawerTransaction(User accountsManager, CashDrawer cashDrawer, Double balanceToTransfer, User grantedBy,
			Session session) throws Exception {

		if (accountsManager == null) {
			throw new PosException("Please select a accounts manager");
		}

		GenericDAO genericDAO = GenericDAO.getInstance();
		genericDAO.refresh(accountsManager, session);
		Double balanceBefore = accountsManager.getAccountsManagerAmount();

		BalanceType balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
		BalanceUpdateTransaction balanceUpdateTransaction = balanceType.createTransaction(PaymentType.CASH, TransactionType.DEBIT, balanceToTransfer);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.ADD_BALANCE.name());

		balanceUpdateTransaction.setAccountNumber(accountsManager.getId());
		balanceUpdateTransaction.setDescription("Transfer");
		balanceUpdateTransaction.putNote(
				String.format("Balance transfer from %s (%s) to cash drawer %s", accountsManager.getFullName(), accountsManager.getId(), cashDrawer.getId()));
		balanceUpdateTransaction.setRecepientId(DefaultDataInserter.addOrGetDrawerRecepient().getId());
		balanceUpdateTransaction.setReasonId(DefaultDataInserter.addOrGetDrawerReason(true).getId());
		balanceUpdateTransaction.setPerformerId(DataProvider.get().getCurrentUser().getId());
		balanceUpdateTransaction.putGrantedByUserId(grantedBy.getId());
		balanceUpdateTransaction.putGrantedByUserName(grantedBy.getFullName());

		accountsManager.setAccountsManagerAmount(NumberUtil.round(balanceBefore - balanceToTransfer));
		genericDAO.saveOrUpdate(accountsManager, session);

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		save(balanceUpdateTransaction, session);

		return balanceUpdateTransaction;
	}

	public void saveAccManagerToStoreTransaction(User accountsManager, Double balanceToTransfer, String note) throws Exception {

		if (accountsManager == null) {
			throw new PosException("Please select a accounts manager");
		}

		BalanceUpdateTransaction acmTransaction;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			GenericDAO genericDAO = GenericDAO.getInstance();
			genericDAO.refresh(accountsManager, session);
			Double balanceBefore = accountsManager.getAccountsManagerAmount();

			//Store store = DataProvider.get().getStore();
			//Double storeCashBeforeBalance = StoreDAO.getInstance().getStoreCashBalance(store, session);
			Outlet outlet = DataProvider.get().getOutlet();
			Double outletCashBeforeBalance = StoreBalanceDAO.getInstance().getOutletCashBalance(outlet, session);

			BalanceType balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
			acmTransaction = balanceType.createTransaction(PaymentType.CASH, TransactionType.DEBIT, balanceToTransfer);
			acmTransaction.setTransactionSubType(BalanceSubType.TRANSFER_OUT.name());
			acmTransaction.setRecepientId(DefaultDataInserter.addOrGetStoreRecepient().getId());
			acmTransaction.setRecipientType(RecipientType.STORE.name());
			acmTransaction.setReasonId(DefaultDataInserter.addOrGetStoreReason(true).getId());
			acmTransaction.setAccountNumber(accountsManager.getId());
			acmTransaction.setDescription(note);
			acmTransaction.setFromAccountId(accountsManager.getId());
			acmTransaction.setFromAccountType(RecipientType.USER.name());

			accountsManager.setAccountsManagerAmount(NumberUtil.round(balanceBefore - balanceToTransfer));
			genericDAO.saveOrUpdate(accountsManager, session);

			double outletCashBalance = NumberUtil.round(outletCashBeforeBalance + balanceToTransfer);
			//StoreDAO.getInstance().updateStoreCashBalance(session, store, storeCashBalance);
			StoreBalanceDAO.getInstance().updateOutletCashBalance(session, outlet, outletCashBalance);

			acmTransaction.setBalanceBefore(balanceBefore);
			save(acmTransaction, session);

			BalanceUpdateTransaction depositToStore = BalanceType.STORE_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT, balanceToTransfer);
			depositToStore.setTransactionSubType(BalanceSubType.TRANSFER_IN.name());
			depositToStore.setRecepientId(DefaultDataInserter.addOrGetStoreRecepient().getId());
			depositToStore.setRecipientType(RecipientType.STORE.name());
			depositToStore.setReasonId(DefaultDataInserter.addOrGetStoreReason(true).getId());
			depositToStore.setBalanceBefore(outletCashBeforeBalance);
			depositToStore.setAccountNumber(outlet.getId());
			depositToStore.setDescription("Transfer to store.");
			depositToStore.putNote(note);
			depositToStore.setAccountProcessed(true);
			depositToStore.setFromAccountId(accountsManager.getId());
			depositToStore.setFromAccountType(RecipientType.USER.name());
			save(depositToStore, session);

			tx.commit();
		}

		LedgerEntryDAO.getInstance().saveAccManagerToStoreLedgerEntry(acmTransaction);
	}

	public void saveWithdrawBalanceFromStoreTransaction(Double balanceToWithdraw, String note) throws Exception {
		BalanceUpdateTransaction balanceUpdateTransaction;
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			balanceUpdateTransaction = saveWithdrawBalanceFromStoreTransaction(balanceToWithdraw, note, null, session);

			tx.commit();
		}
		LedgerEntryDAO.getInstance().saveWithdrawBalanceFromStoreLedgerEntry(balanceUpdateTransaction);
	}

	public BalanceUpdateTransaction saveWithdrawBalanceFromStoreTransaction(Double balanceToWithdraw, String note, User grantedBy, Session session)
			throws Exception {

		//Store store = DataProvider.get().getStore();
		//Double storeCashBeforeBalance = StoreDAO.getInstance().getStoreCashBalance(store, session);
		Outlet store = DataProvider.get().getOutlet();
		Double storeCashBeforeBalance = StoreBalanceDAO.getInstance().getOutletCashBalance(store, session);

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.STORE_BALANCE.createTransaction(PaymentType.CASH, TransactionType.DEBIT,
				balanceToWithdraw);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.WITHDRAW.name());
		balanceUpdateTransaction.setRecepientId(DefaultDataInserter.addOrGetStoreRecepient().getId());
		balanceUpdateTransaction.setReasonId(DefaultDataInserter.addOrGetStoreReason(false).getId());
		balanceUpdateTransaction.setBalanceBefore(storeCashBeforeBalance);
		balanceUpdateTransaction.setDescription(note);

		balanceUpdateTransaction.setPerformerId(DataProvider.get().getCurrentUser().getId());
		if (grantedBy != null) {
			balanceUpdateTransaction.putGrantedByUserId(grantedBy.getId());
			balanceUpdateTransaction.putGrantedByUserName(grantedBy.getFullName());
		}
		save(balanceUpdateTransaction, session);

		double storeCashBalance = NumberUtil.round(storeCashBeforeBalance - balanceToWithdraw);
		//StoreDAO.getInstance().updateStoreCashBalance(session, store, storeCashBalance);
		StoreBalanceDAO.getInstance().updateOutletCashBalance(session, store, storeCashBalance);

		return balanceUpdateTransaction;
	}

	public void saveBeginCashTransaction(CashDrawer cashDrawer, Session session) throws Exception {
		Double beginCashAmount = cashDrawer.getBeginCash();
		if (beginCashAmount <= 0) {
			return;
		}

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.CASH_DRAWER_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT,
				beginCashAmount);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.BEGIN_CASH_DEPOSIT.name());
		balanceUpdateTransaction.setAccountNumber(cashDrawer.getId());
		balanceUpdateTransaction.setDescription("Cash drawer begin cash");
		save(balanceUpdateTransaction, session);

		LedgerEntryDAO.getInstance().saveBeginCashLedgerEntry(balanceUpdateTransaction, session);
	}

	public boolean isPresent(String batchNo) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BATCH_NO, batchNo));
			criteria.setMaxResults(1);
			List list = criteria.list();
			return list != null && !list.isEmpty();
		}
	}

	public BalanceUpdateTransaction saveSalaryExpenseTransaction(Session session, Date eventTime, String batchNo, Project project, Object paymentMethod,
			String paymentRef, User accountsManager, User paidToUser, double expensesAmount, String note, Boolean isAMDepositMethod,
			PosTransaction salaryTransaction) throws Exception {
		return saveSalaryExpenseTransaction(session, eventTime, batchNo, project, paymentMethod, paymentRef, accountsManager, paidToUser, expensesAmount, note,
				isAMDepositMethod, salaryTransaction, false);
	}

	public BalanceUpdateTransaction saveSalaryExpenseTransaction(Session session, Date eventTime, String batchNo, Project project, Object paymentMethod,
			String paymentRef, User accountsManager, User paidToUser, double expensesAmount, String note, Boolean isAMDepositMethod,
			PosTransaction salaryTransaction, boolean isRevert) throws Exception {
		String bankOrCustomPaymentId = getPaymentMethodId(paymentMethod);
		PaymentType paymentType = getPaymentType(paymentMethod);

		if (paymentMethod instanceof BankAccount) {
			return saveBankAccountExpenseTransaction(session, eventTime, paymentType, bankOrCustomPaymentId, note, batchNo, paidToUser, expensesAmount,
					salaryTransaction, isRevert);
		}

		//Store store = DataProvider.get().getStore();
		Outlet outlet = DataProvider.get().getOutlet();
		BalanceType balanceType;
		Double balanceBefore;

		if (isAMDepositMethod) {
			balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
			balanceBefore = accountsManager.getAccountsManagerAmount();
		}
		else {
			balanceType = BalanceType.STORE_BALANCE;
			//balanceBefore = StoreDAO.getInstance().getStoreCashBalance(store, session);
			balanceBefore = StoreBalanceDAO.getInstance().getOutletCashBalance(outlet, session);
		}

		BalanceUpdateTransaction balanceUpdateTransaction = balanceType.createTransaction(paymentType,
				isRevert ? TransactionType.CREDIT : TransactionType.DEBIT, expensesAmount);
		balanceUpdateTransaction.setTransactionSubType(isRevert ? BalanceSubType.EXPENSE_REVERT.name() : BalanceSubType.EXPENSE.name());
		balanceUpdateTransaction.putSourceType(salaryTransaction.REF);
		balanceUpdateTransaction.putSourceId(salaryTransaction.getId());
		if (salaryTransaction instanceof SalaryTransaction) {
			balanceUpdateTransaction.putTransactionReason("Salary payment");
		}
		else if (salaryTransaction instanceof SalaryAdvanceTransaction) {
			balanceUpdateTransaction.putTransactionReason("Salary advance payment");
		}

		if (eventTime != null) {
			balanceUpdateTransaction.setEventTime(eventTime);
		}

		balanceUpdateTransaction.setBatchNo(batchNo);
		balanceUpdateTransaction.setProject(project);

		balanceUpdateTransaction.setDescription(note);

		double cashBalance = NumberUtil.round(isRevert ? balanceBefore + expensesAmount : balanceBefore - expensesAmount);

		if (isAMDepositMethod) {
			balanceUpdateTransaction.setAccountNumber(accountsManager.getId());

			accountsManager.setAccountsManagerAmount(cashBalance);
			UserDAO.getInstance().saveOrUpdate(accountsManager, session);
		}
		else {
			balanceUpdateTransaction.setAccountNumber(outlet.getId());

			//StoreDAO.getInstance().updateStoreCashBalance(session, store, cashBalance);
			StoreBalanceDAO.getInstance().updateOutletCashBalance(session, outlet, cashBalance);
		}

		if (paidToUser != null) {
			balanceUpdateTransaction.putSalaryPaidTo(paidToUser.getId());
		}

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		save(balanceUpdateTransaction, session);

		return balanceUpdateTransaction;
	}

	public BalanceUpdateTransaction saveBankAccountExpenseTransaction(Session session, Date eventTime, PaymentType paymentType, String bankAccountId,
			String note, String batchNo, User paidToUser, double expenseAmount, PosTransaction salaryTransaction) throws Exception {
		return saveBankAccountExpenseTransaction(session, eventTime, paymentType, bankAccountId, note, batchNo, paidToUser, expenseAmount, salaryTransaction,
				false);
	}

	public BalanceUpdateTransaction saveBankAccountExpenseTransaction(Session session, Date eventTime, PaymentType paymentType, String bankAccountId,
			String note, String batchNo, User paidToUser, double expenseAmount, PosTransaction salaryTransaction, boolean isRevert) throws Exception {
		BankAccount bankAccount = loadBankAccount(bankAccountId, session);

		Double balanceBefore = bankAccount.getBalance();

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.BANK_ACCOUNT_BALANCE.createTransaction(paymentType,
				isRevert ? TransactionType.CREDIT : TransactionType.DEBIT, expenseAmount);
		balanceUpdateTransaction.setTransactionSubType(isRevert ? BalanceSubType.EXPENSE_REVERT.name() : BalanceSubType.EXPENSE.name());

		balanceUpdateTransaction.setAccountNumber(bankAccountId);
		balanceUpdateTransaction.setBankAccountId(bankAccountId);
		balanceUpdateTransaction.setDescription(note);
		balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		balanceUpdateTransaction.setAccountProcessed(true);
		balanceUpdateTransaction.setBatchNo(batchNo);
		balanceUpdateTransaction.putSourceType(salaryTransaction.REF);
		balanceUpdateTransaction.putSourceId(salaryTransaction.getId());

		if (eventTime != null) {
			balanceUpdateTransaction.setEventTime(eventTime);
		}

		if (paidToUser != null) {
			balanceUpdateTransaction.putSalaryPaidTo(paidToUser.getId());
		}

		bankAccount.setBalance(NumberUtil.round(isRevert ? balanceBefore + expenseAmount : balanceBefore - expenseAmount));

		save(balanceUpdateTransaction, session);
		BankAccountDAO.getInstance().saveOrUpdate(bankAccount, session);

		return balanceUpdateTransaction;
	}

	private BankAccount loadBankAccount(String bankAccountId, Session session) {
		if (StringUtils.isBlank(bankAccountId)) {
			throw new PosException("Please select a bank account");
		}

		BankAccount bankAccount = BankAccountDAO.getInstance().get(bankAccountId, session);
		if (bankAccount == null) {
			throw new PosException("Could not find bank account");
		}
		return bankAccount;
	}

	public BalanceUpdateTransaction saveBankAccountAddBalanceTransaction(BankAccount bankAccount, PaymentType paymentType, Double balanceToAdd, String note,
			String customPaymentInfo) {
		Transaction tx = null;
		BalanceUpdateTransaction transaction;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			transaction = saveBankAccountAddBalanceTransaction(bankAccount, paymentType, balanceToAdd, note, session, customPaymentInfo);
			tx.commit();
		}
		return transaction;
	}

	public BalanceUpdateTransaction saveBankAccountAddBalanceTransaction(BankAccount bankAccount, Double balanceToAdd, String note, Session session) {
		return saveBankAccountAddBalanceTransaction(bankAccount, PaymentType.CASH, balanceToAdd, note, session);
	}

	public BalanceUpdateTransaction saveBankAccountAddBalanceTransaction(BankAccount bankAccount, PaymentType paymentType, Double balanceToAdd, String note,
			Session session) {
		return saveBankAccountAddBalanceTransaction(bankAccount, paymentType, balanceToAdd, note, session, null);
	}

	public BalanceUpdateTransaction saveBankAccountAddBalanceTransaction(BankAccount bankAccount, PaymentType paymentType, Double balanceToAdd, String note,
			Session session, String customPaymentInfo) {
		return saveBankAccountTransaction(bankAccount, paymentType, null, balanceToAdd, note, session, customPaymentInfo);
	}

	public BalanceUpdateTransaction saveBankAccountTransaction(BankAccount bankAccount, PaymentType paymentType, PosTransaction posTransaction,
			Double balanceToAdd, String note, Session session, String customPaymentInfo) {
		if (bankAccount == null) {
			return null;
		}
		BankAccountDAO bankAccountDAO = BankAccountDAO.getInstance();
		bankAccountDAO.refresh(bankAccount, session);

		TransactionType transactionType = TransactionType.CREDIT;
		if (posTransaction != null) {
			TransactionType type = TransactionType.valueOf(posTransaction.getTransactionType());
			if (posTransaction.isVoided()) {
				if (type == TransactionType.CREDIT) {
					type = TransactionType.DEBIT;
				}
				else if (type == TransactionType.DEBIT) {
					type = TransactionType.CREDIT;
				}
				else if (type == TransactionType.IN) {
					type = TransactionType.OUT;
				}
				else if (type == TransactionType.OUT) {
					type = TransactionType.IN;
				}
			}
			transactionType = type;
		}

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.BANK_ACCOUNT_BALANCE.createTransaction(paymentType, transactionType, balanceToAdd);

		balanceUpdateTransaction.setAccountNumber(bankAccount.getId());

		if (posTransaction != null) {
			balanceUpdateTransaction.setTicketId(posTransaction.getTicketId());
			balanceUpdateTransaction.setTransactionId(posTransaction.getId());
			balanceUpdateTransaction.setProjectId(posTransaction.getProjectId());
			balanceUpdateTransaction.putEntityNo(posTransaction.getEntityNo());
			balanceUpdateTransaction.putEntityId(posTransaction.getEntityId());
			balanceUpdateTransaction.putEntityType(posTransaction.getEntityType());

			//TransactionType transactionType = TransactionType.valueOf(posTransaction.getTransactionType());
			if (posTransaction.isVoided()) {
				balanceUpdateTransaction.setTransactionSubType(BalanceSubType.WITHDRAW.name());
				balanceUpdateTransaction.putTransactionReason("Payment void");
			}
			else if (transactionType == TransactionType.CREDIT || transactionType == TransactionType.IN) {
				balanceUpdateTransaction.setTransactionSubType(BalanceSubType.ADD_BALANCE.name());
				balanceUpdateTransaction.putTransactionReason("Payment receive");
				balanceUpdateTransaction.setDescription("Payment receive");
			}
			else if (transactionType == TransactionType.DEBIT || transactionType == TransactionType.OUT) {
				balanceUpdateTransaction.setTransactionSubType(BalanceSubType.EXPENSE.name());
				balanceUpdateTransaction.putTransactionReason("Payment deposit");
			}
		}
		else {
			balanceUpdateTransaction.setTransactionSubType(BalanceSubType.ADD_BALANCE.name());
			balanceUpdateTransaction.setDescription("Add balance");
			balanceUpdateTransaction.putTransactionReason("Add balance");
		}

		if (StringUtils.isNotBlank(note)) {
			balanceUpdateTransaction.putNote(note);
		}

		Double balanceBefore = bankAccount.getBalance();
		bankAccount.setBalance(NumberUtil.round(balanceBefore + balanceUpdateTransaction.getAmount()));

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);

		if (StringUtils.isNotBlank(customPaymentInfo)) {
			balanceUpdateTransaction.putCustomePaymentInfo(customPaymentInfo);

			if (posTransaction == null
					|| Arrays.asList(TransactionType.CREDIT.name(), TransactionType.IN.name()).contains(posTransaction.getTransactionType())) {
				balanceUpdateTransaction.setDescription("Payment receive. " + balanceUpdateTransaction.getCustomePaymentInfoDisplay());
			}
			else {
				balanceUpdateTransaction.setDescription("Payment deposit. " + balanceUpdateTransaction.getCustomePaymentInfoDisplay());
			}

		}

		bankAccountDAO.saveOrUpdate(bankAccount, session);
		balanceUpdateTransaction.setAccountProcessed(true);
		save(balanceUpdateTransaction, session);

		return balanceUpdateTransaction;
	}

	private PaymentType getPaymentType(Object paymentMethod) {
		if (paymentMethod instanceof String) {
			return PaymentType.CASH;
		}
		if (paymentMethod instanceof CustomPayment) {
			return PaymentType.CUSTOM_PAYMENT;
		}
		if (paymentMethod instanceof BankAccount) {
			return PaymentType.BANK_ACCOUNT;
		}
		return PaymentType.CASH;
	}

	public String getPaymentMethodId(Object paymentMethod) {
		if (paymentMethod instanceof CustomPayment) {
			return ((CustomPayment) paymentMethod).getId();
		}
		if (paymentMethod instanceof BankAccount) {
			return ((BankAccount) paymentMethod).getId();
		}

		return null;
	}

	public List<BalanceUpdateTransaction> findBankAccountTransaction(Outlet outlet, Date from, Date to, String bankAccountId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

			if (from != null && to != null) {
				criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_EVENT_TIME, from, to));

			}

			if (outlet != null) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, outlet.getId()));
			}
			criteria.add(Restrictions.eqOrIsNull(BalanceUpdateTransaction.PROP_VOIDED, false));

			if (StringUtils.isNotBlank(bankAccountId)) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, bankAccountId));
			}

			criteria.addOrder(Order.asc(BalanceUpdateTransaction.PROP_EVENT_TIME));

			List<BalanceUpdateTransaction> list = criteria.list();

			List<BalanceUpdateTransaction> allTransactionList = new ArrayList<BalanceUpdateTransaction>();
			if (list.isEmpty()) {
				BankAccount bankAccount = BankAccountDAO.getInstance().get(bankAccountId);
				if (bankAccount != null) {
					BalanceForwardTransaction forwardTransaction = new BalanceForwardTransaction();
					forwardTransaction.setTicketId("Balance forward");
					forwardTransaction.setAccountNumber(bankAccountId);
					forwardTransaction.setAmount(bankAccount.getBalance());
					allTransactionList.add(forwardTransaction);
				}
			}
			else {
				for (int i = 0; i < list.size(); i++) {
					BalanceUpdateTransaction balanceUpdateTransaction = list.get(i);
					if (i == 0) {
						allTransactionList.add(createForwardBalanceByTransction(bankAccountId, balanceUpdateTransaction));
					}
					allTransactionList.add(balanceUpdateTransaction);
				}
			}

			return allTransactionList;
		}
	}

	public BalanceUpdateTransaction createAccountsManagerTransaction(CashDrawer cashDrawer, User fromAccountsManager, User toAccountsManager,
			BalanceSubType balanceSubType, TransactionType transactionType, Double balanceToAdd, String note, Session session) {
		if (fromAccountsManager == null) {
			return null;
		}
		UserDAO userDao = UserDAO.getInstance();
		userDao.refresh(fromAccountsManager, session);

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.ACCOUNTS_MANAGER_BALANCE.createTransaction(PaymentType.CASH, transactionType,
				balanceToAdd);
		balanceUpdateTransaction.setTransactionSubType(balanceSubType.name());

		balanceUpdateTransaction.setAccountNumber(fromAccountsManager.getId());
		if (cashDrawer != null) {
			balanceUpdateTransaction.setFromCashdrawerId(cashDrawer.getId());
		}
		if (fromAccountsManager != null) {
			//balanceUpdateTransaction.set
		}
		if (StringUtils.isNotBlank(note)) {
			balanceUpdateTransaction.setDescription(note);
		}

		Double balanceBefore = fromAccountsManager.getAccountsManagerAmount();
		fromAccountsManager.setAccountsManagerAmount(NumberUtil.round(balanceBefore + balanceToAdd));

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);

		userDao.saveOrUpdate(fromAccountsManager, session);
		save(balanceUpdateTransaction, session);

		return balanceUpdateTransaction;
	}

	public void transferBalanceToAccountsManager(User fromAccountsManager, User toAccountsManager, Double balanceToTransfer, String note) throws Exception {
		transferBalanceToAccountsManager(fromAccountsManager, toAccountsManager, balanceToTransfer, note, null);
	}

	public void transferBalanceToAccountsManager(User fromAccountsManager, User toAccountsManager, Double balanceToTransfer, String note, String requisitionId)
			throws Exception {
		transferBalanceToAccountsManager(fromAccountsManager, toAccountsManager, balanceToTransfer, null, note, requisitionId);
	}

	public void transferBalanceToAccountsManager(User fromAccountsManager, User toAccountsManager, Double balanceToTransfer, String prefixNote, String note,
			String requisitionId) throws Exception {

		if (fromAccountsManager == null) {
			throw new PosException("Please select accounts manager from whom balance to transfer");
		}
		if (toAccountsManager == null) {
			throw new PosException("Please select accounts manager to whom balance to transfer");
		}

		BalanceUpdateTransaction fromAccountTransaction = null;
		BalanceUpdateTransaction toAccountTransaction = null;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			GenericDAO genericDAO = GenericDAO.getInstance();
			genericDAO.refresh(fromAccountsManager, session);
			genericDAO.refresh(toAccountsManager, session);

			Double fromAccountBalanceBefore = fromAccountsManager.getAccountsManagerAmount();
			Double toAccountBalanceBefore = toAccountsManager.getAccountsManagerAmount();

			BalanceType balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
			fromAccountTransaction = balanceType.createTransaction(PaymentType.CASH, TransactionType.DEBIT, balanceToTransfer);
			fromAccountTransaction.setTransactionSubType(BalanceSubType.TRANSFER_OUT.name());
			fromAccountTransaction.setAccountNumber(fromAccountsManager.getId());
			prefixNote = StringUtils.isBlank(prefixNote) ? StringUtils.EMPTY : prefixNote.trim() + " ";
			//fromAccountTransaction.setDescription(prefixNote + "Transfer to acm " + toAccountsManager.getFullName() + "\n" + note);
			fromAccountTransaction.setDescription("Transferred " + balanceToTransfer + " from ACM " + fromAccountsManager.getFullName() + " to ACM "
					+ toAccountsManager.getFullName());
			fromAccountTransaction.putNote(note);

			genericDAO.saveOrUpdate(fromAccountsManager, session);

			double toAccountBalance = NumberUtil.round(toAccountBalanceBefore + balanceToTransfer);
			//updateStoreCashBalance(session, storeCashBalance);

			fromAccountTransaction.setBalanceBefore(fromAccountBalanceBefore);
			fromAccountTransaction.putBalanceTransferInfo(fromAccountsManager, toAccountsManager, balanceToTransfer);
			fromAccountTransaction.setAccountProcessed(true);
			fromAccountTransaction.putRequisitionId(requisitionId);
			fromAccountTransaction.setRecepientId(toAccountsManager.getId());
			fromAccountTransaction.setRecipientType(RecipientType.USER.name());
			save(fromAccountTransaction, session);

			toAccountTransaction = BalanceType.ACCOUNTS_MANAGER_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT, balanceToTransfer);
			toAccountTransaction.setTransactionSubType(BalanceSubType.TRANSFER_IN.name());
			toAccountTransaction.setBalanceBefore(toAccountBalanceBefore);
			toAccountTransaction.setAccountNumber(toAccountsManager.getId());
			//toAccountTransaction.setDescription(prefixNote + "Transfer from acm " + fromAccountsManager.getFullName());
			toAccountTransaction.setDescription(
					"Received " + balanceToTransfer + " from ACM " + fromAccountsManager.getFullName() + " to ACM " + toAccountsManager.getFullName());
			toAccountTransaction.putBalanceTransferInfo(fromAccountsManager, toAccountsManager, balanceToTransfer);
			toAccountTransaction.setAccountProcessed(true);
			toAccountTransaction.setRecepientId(requisitionId);
			toAccountTransaction.setRecepientId(toAccountsManager.getId());
			toAccountTransaction.setRecipientType(RecipientType.USER.name());
			toAccountTransaction.setFromAccountId(fromAccountsManager.getId());
			toAccountTransaction.setFromAccountType(RecipientType.USER.name());
			toAccountTransaction.putNote(note);
			
			save(toAccountTransaction, session);

			fromAccountsManager.setAccountsManagerAmount(NumberUtil.round(fromAccountBalanceBefore - balanceToTransfer));
			UserDAO.getInstance().update(fromAccountsManager, session);

			toAccountsManager.setAccountsManagerAmount(toAccountBalance);
			UserDAO.getInstance().update(toAccountsManager, session);

			LedgerEntryDAO.getInstance().saveTransferToAcmLedgerEntry(fromAccountTransaction, toAccountTransaction, session);

			tx.commit();
		}
	}

	public List<BalanceUpdateTransaction> getBalanceUpdateTransactionsByIds(List<String> ids) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.add(Restrictions.in(BalanceUpdateTransaction.PROP_ID, ids));
			return criteria.list();
		}
	}

	public void saveAccountsManagerRevertExpenses(ExpenseTransaction sourceExpenseTransaction, String revertReason, User revertedByUser) throws Exception {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			String projectId = sourceExpenseTransaction.getProjectId();
			Date now = StoreDAO.getServerTimestamp();

			String managerId = sourceExpenseTransaction.getAccountManagerId();
			User accountManager = DataProvider.get().getUserById(managerId, sourceExpenseTransaction.getOutletId());

			BalanceUpdateTransaction balanceUpdateTransaction = saveAccountsManagerExpensesTransaction(now, accountManager,
					sourceExpenseTransaction.getBatchNo(), projectId, null, null, null, sourceExpenseTransaction.getAmount(), "", null, session, true); //$NON-NLS-1$

			String memoNo = sourceExpenseTransaction.getMemoNo();
			if (StringUtils.isNotBlank(memoNo)) {
				balanceUpdateTransaction.setMemoNo(memoNo);
			}
			balanceUpdateTransaction.putReferenceId(sourceExpenseTransaction.getId());
			balanceUpdateTransaction.setDescription(revertReason);
			balanceUpdateTransaction.setAccountProcessed(true);
			update(balanceUpdateTransaction, session);

			sourceExpenseTransaction.setVoidedByUser(revertedByUser);
			sourceExpenseTransaction.putRevertReason(revertReason);
			sourceExpenseTransaction.setVoided(true);
			sourceExpenseTransaction.setVoidDate(now);
			sourceExpenseTransaction.setAccountProcessed(true);

			ExpenseTransactionDAO.getInstance().saveOrUpdate(sourceExpenseTransaction, session);

			LedgerEntryDAO.getInstance().saveExpensesLedgerEntry(balanceUpdateTransaction, true, sourceExpenseTransaction.getChartOfAccountsCode(), session);

			tx.commit();
		}
	}

	public void saveStoreRevertExpenses(ExpenseTransaction sourceExpenseTransaction, Outlet outlet, String revertReason, User revertedByUser) throws Exception {

		Date now = StoreDAO.getServerTimestamp();

		try (Session session = createNewSession()) {
			Transaction tx = session.beginTransaction();

			String projectId = sourceExpenseTransaction.getProjectId();
			Project project = (Project) DataProvider.get().getObjectOf(Project.class, projectId);
			BalanceUpdateTransaction balanceUpdateTransaction = saveStoreExpensesTransaction(now, outlet, sourceExpenseTransaction.getBatchNo(), project, null,
					null, null, sourceExpenseTransaction.getAmount(), "", session, true);
			balanceUpdateTransaction.putReferenceId(sourceExpenseTransaction.getId());
			balanceUpdateTransaction.setDescription(revertReason);
			balanceUpdateTransaction.setAccountProcessed(true);
			String memoNo = sourceExpenseTransaction.getMemoNo();
			if (StringUtils.isNotBlank(memoNo)) {
				balanceUpdateTransaction.setMemoNo(memoNo);
			}
			update(balanceUpdateTransaction, session);

			sourceExpenseTransaction.setVoidedByUser(revertedByUser);
			sourceExpenseTransaction.putRevertReason(revertReason);
			sourceExpenseTransaction.setVoided(true);
			sourceExpenseTransaction.setVoidDate(now);
			sourceExpenseTransaction.setAccountProcessed(true);

			ExpenseTransactionDAO.getInstance().saveOrUpdate(sourceExpenseTransaction, session);

			LedgerEntryDAO.getInstance().saveExpensesLedgerEntry(balanceUpdateTransaction, false, sourceExpenseTransaction.getChartOfAccountsCode(), session);

			tx.commit();
		}
	}

	public void saveAdvanceBalanceUpdateTransaction(PosTransaction transaction, Session session) {
		TransactionType balanceAdjustTransactionType = !(transaction instanceof RefundTransaction) && !transaction.isVoided()
				&& transaction.getPaymentType() == PaymentType.ADVANCE_ADJUSTMENT ? TransactionType.DEBIT : TransactionType.CREDIT;
		boolean creditTransaction = balanceAdjustTransactionType == TransactionType.CREDIT;
		TransactionSubType transactionSubType = creditTransaction ? TransactionSubType.BALANCE_ADDED : TransactionSubType.BALANCE_ADJUSTED;

		Customer customer = CustomerDAO.getInstance().get(transaction.getCustomerId(), session);
		Double balanceBefore = customer.getBalance();
		Double adjustAmount = creditTransaction ? transaction.getAmount() : -transaction.getAmount();
		customer.setBalance(NumberUtil.round(balanceBefore + adjustAmount));

		saveBalanceUpdateTrans(BalanceType.CUSTOMER, transaction.getTicket(), transaction, balanceAdjustTransactionType, transaction.getCustomerId(),
				transaction.getAmount(), balanceBefore, transactionSubType, session);
		CustomerDAO.getInstance().update(customer, session);
	}

	public void customerBalanceRefundAmountBalanceUpdateTxn(PaymentType paymentType, Customer customer, PosTransaction refundTransaction, double tenderAmount,
			CustomPayment customPayment, String customPaymnetRef, Session session) {
		Double balanceBefore = customer.getBalance();
		customer.setBalance(NumberUtil.round(balanceBefore - tenderAmount));

		saveBalanceUpdateTrans(BalanceType.CUSTOMER, null, refundTransaction, TransactionType.DEBIT, customer.getId(), tenderAmount, balanceBefore,
				TransactionSubType.REFUNDED, customPayment, customPaymnetRef, session);
		CustomerDAO.getInstance().update(customer, session);
	}
}