package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.CryoStorage;
import com.floreantpos.model.StorageType;
import com.floreantpos.model.TimedModel;

public class CryoStorageDAO extends BaseCryoStorageDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public CryoStorageDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		CryoStorage storage = (CryoStorage) obj;
		storage.setDeleted(Boolean.TRUE);
		saveOrUpdate(storage, session);
	}

	/**
	 * Find all storage containers by type
	 * @param storageType LOCATION, TANK, CANE, GOBLET, POSITION
	 * @return list of storage containers
	 */
	public List<CryoStorage> findByType(StorageType storageType) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(CryoStorage.PROP_STORAGE_TYPE, storageType));
			criteria.add(Restrictions.eq(CryoStorage.PROP_DELETED, false));
			criteria.addOrder(Order.asc(CryoStorage.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(CryoStorage.PROP_NAME));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Find all storage containers by type and status
	 */
	public List<CryoStorage> findByTypeAndStatus(StorageType storageType, String status) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(CryoStorage.PROP_STORAGE_TYPE, storageType));
			criteria.add(Restrictions.eq(CryoStorage.PROP_STATUS, status));
			criteria.add(Restrictions.eq(CryoStorage.PROP_DELETED, false));
			criteria.addOrder(Order.asc(CryoStorage.PROP_SORT_ORDER));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Find all children of a parent container
	 * @param parentId parent container ID
	 * @return list of child containers
	 */
	public List<CryoStorage> findByParentId(String parentId) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(CryoStorage.PROP_PARENT_ID, parentId));
			criteria.add(Restrictions.eq(CryoStorage.PROP_DELETED, false));
			criteria.addOrder(Order.asc(CryoStorage.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(CryoStorage.PROP_NAME));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Find all children of a specific type under a parent
	 */
	public List<CryoStorage> findByParentIdAndType(String parentId, StorageType storageType) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(CryoStorage.PROP_PARENT_ID, parentId));
			criteria.add(Restrictions.eq(CryoStorage.PROP_STORAGE_TYPE, storageType));
			criteria.add(Restrictions.eq(CryoStorage.PROP_DELETED, false));
			criteria.addOrder(Order.asc(CryoStorage.PROP_SORT_ORDER));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Find all root containers (no parent)
	 */
	public List<CryoStorage> findRoots() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(
				Restrictions.isNull(CryoStorage.PROP_PARENT_ID),
				Restrictions.eq(CryoStorage.PROP_PARENT_ID, "")
			));
			criteria.add(Restrictions.eq(CryoStorage.PROP_DELETED, false));
			criteria.addOrder(Order.asc(CryoStorage.PROP_NAME));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Find containers with available space
	 */
	public List<CryoStorage> findAvailable(StorageType storageType) {
		Session session = null;
		try {
			session = createNewSession();
			String hql = "FROM CryoStorage WHERE storageType = :type " +
			             "AND status = 'ACTIVE' " +
			             "AND (currentCount < capacity OR capacity IS NULL) " +
			             "AND deleted = false " +
			             "ORDER BY currentCount, sortOrder";
			Query query = session.createQuery(hql);
			query.setParameter("type", storageType);
			return query.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Find first available container with space
	 */
	public CryoStorage findFirstAvailable(StorageType storageType, String parentId) {
		Session session = null;
		try {
			session = createNewSession();
			String hql = "FROM CryoStorage WHERE storageType = :type " +
			             "AND parentId = :parentId " +
			             "AND status = 'ACTIVE' " +
			             "AND currentCount < capacity " +
			             "AND deleted = false " +
			             "ORDER BY currentCount, sortOrder";
			Query query = session.createQuery(hql);
			query.setParameter("type", storageType);
			query.setParameter("parentId", parentId);
			query.setMaxResults(1);
			List<CryoStorage> results = query.list();
			return results.isEmpty() ? null : results.get(0);
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Find containers needing maintenance
	 */
	public List<CryoStorage> findMaintenanceDue() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.le(CryoStorage.PROP_NEXT_MAINTENANCE_DATE, new Date()));
			criteria.add(Restrictions.eq(CryoStorage.PROP_STATUS, "ACTIVE"));
			criteria.add(Restrictions.eq(CryoStorage.PROP_DELETED, false));
			criteria.addOrder(Order.asc(CryoStorage.PROP_NEXT_MAINTENANCE_DATE));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Get full hierarchical path for a container
	 * Returns: ["Main Lab", "TANK-02", "CANE-18", "GOBLET-3"]
	 */
	public List<CryoStorage> getPathToRoot(String containerId) {
		List<CryoStorage> path = new ArrayList<>();
		Session session = null;
		try {
			session = createNewSession();
			CryoStorage current = get(containerId, session);
			while (current != null) {
				path.add(0, current); // Add to beginning
				String parentId = current.getParentId();
				if (StringUtils.isEmpty(parentId)) {
					break;
				}
				current = get(parentId, session);
			}
			return path;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Get full path as string
	 * Returns: "Main Lab → TANK-02 → CANE-18 → GOBLET-3"
	 */
	public String getPathString(String containerId) {
		List<CryoStorage> path = getPathToRoot(containerId);
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < path.size(); i++) {
			if (i > 0) sb.append(" → ");
			sb.append(path.get(i).getName());
		}
		return sb.toString();
	}

	/**
	 * Update parent count when adding/removing samples
	 */
	public void updateParentCounts(String containerId, int delta) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			CryoStorage current = get(containerId, session);
			while (current != null) {
				int newCount = current.getCurrentCount() + delta;
				current.setCurrentCount(Math.max(0, newCount));

				// Update status based on capacity
				if (current.getCapacity() != null) {
					if (newCount >= current.getCapacity()) {
						current.setStatus("FULL");
					} else if ("FULL".equals(current.getStatus())) {
						current.setStatus("ACTIVE");
					}
				}

				update(current, session);

				// Move to parent
				String parentId = current.getParentId();
				if (StringUtils.isEmpty(parentId)) {
					break;
				}
				current = get(parentId, session);
			}

			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Get storage statistics by type
	 */
	public Map<String, Object> getStatsByType(StorageType storageType) {
		Session session = null;
		try {
			session = createNewSession();
			String hql = "SELECT " +
			             "COUNT(*), " +
			             "SUM(capacity), " +
			             "SUM(currentCount), " +
			             "COUNT(CASE WHEN status = 'ACTIVE' THEN 1 END), " +
			             "COUNT(CASE WHEN status = 'FULL' THEN 1 END) " +
			             "FROM CryoStorage " +
			             "WHERE storageType = :type AND deleted = false";
			Query query = session.createQuery(hql);
			query.setParameter("type", storageType);
			Object[] result = (Object[]) query.uniqueResult();

			Map<String, Object> stats = new HashMap<>();
			stats.put("totalCount", result[0]);
			stats.put("totalCapacity", result[1]);
			stats.put("totalUsed", result[2]);
			stats.put("activeCount", result[3]);
			stats.put("fullCount", result[4]);

			// Calculate utilization percentage
			Long capacity = (Long) result[1];
			Long used = (Long) result[2];
			if (capacity != null && capacity > 0) {
				double utilization = (used.doubleValue() / capacity.doubleValue()) * 100.0;
				stats.put("utilizationPercentage", utilization);
			} else {
				stats.put("utilizationPercentage", 0.0);
			}

			return stats;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Find by name (exact match)
	 */
	public CryoStorage findByName(String name) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(CryoStorage.PROP_NAME, name));
			criteria.add(Restrictions.eq(CryoStorage.PROP_DELETED, false));
			List<CryoStorage> results = criteria.list();
			return results.isEmpty() ? null : results.get(0);
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Search by name (partial match)
	 */
	public List<CryoStorage> searchByName(String namePattern) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ilike(CryoStorage.PROP_NAME, "%" + namePattern + "%"));
			criteria.add(Restrictions.eq(CryoStorage.PROP_DELETED, false));
			criteria.addOrder(Order.asc(CryoStorage.PROP_NAME));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

}
