package com.floreantpos.model.dao;

import java.io.Serializable;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Couple;
import com.floreantpos.model.Customer;
import com.floreantpos.model.CyclePlan;
import com.floreantpos.model.Patient;
import com.floreantpos.swing.PaginationSupport;

public class CyclePlanDAO extends BaseCyclePlanDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public CyclePlanDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		Serializable serializable = super.save(obj, s);
		return serializable;
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public void delete(Object obj, Session session) throws HibernateException {
		CyclePlan cyclePlan = (CyclePlan) obj;
		if (cyclePlan == null) {
			throw new PosException(Messages.getString("MenuItemDAO.0")); //$NON-NLS-1$
		}

		cyclePlan.setDeleted(true);

		session.update(cyclePlan);
	}

	private void doCheckValidation(Object obj) {
		CyclePlan cyclePlan = (CyclePlan) obj;
		// Add validation logic here if needed
	}

	/**
	 * Load full cycle plan with all lazy collections initialized
	 */
	public CyclePlan loadFullCyclePlan(String cyclePlanId) {
		Session session = null;
		try {
			session = createNewSession();
			CyclePlan cyclePlan = (CyclePlan) get(CyclePlan.class, cyclePlanId, session);
			if (cyclePlan != null) {
				initializeCyclePlan(cyclePlan, session);
			}
			return cyclePlan;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Load full cycle plan with all lazy collections initialized
	 */
	public CyclePlan loadFullCyclePlan(CyclePlan cyclePlan) {
		Session session = null;
		try {
			session = createNewSession();
			session.refresh(cyclePlan);
			initializeCyclePlan(cyclePlan, session);
			return cyclePlan;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Initialize lazy collections for a cycle plan within an active session
	 */
	public void initializeCyclePlan(CyclePlan cyclePlan, Session session) {
		if (cyclePlan == null) {
			return;
		}

		try {
			// Initialize cycle stage logs collection - force load by calling size()
			if (cyclePlan.getCycleStageLogs() != null) {
				cyclePlan.getCycleStageLogs().size();
			}

			// Initialize embryos collection - force load by calling size()
			if (cyclePlan.getEmbryos() != null) {
				cyclePlan.getEmbryos().size();
			}

			// Initialize protocol phases collection - force load by calling size()
			if (cyclePlan.getProtocolPhases() != null) {
				cyclePlan.getProtocolPhases().size();

				// Initialize nested cycleStageLogs for each protocol phase
				for (com.floreantpos.model.ProtocolPhase phase : cyclePlan.getProtocolPhases()) {
					if (phase.getCycleStageLogs() != null) {
						phase.getCycleStageLogs().size();
					}
				}
			}

			// Initialize couple and related patient data if couple exists
			if (cyclePlan.getCoupleId() != null) {
				Couple couple = (Couple) session.get(Couple.class, cyclePlan.getCoupleId());
				if (couple != null) {
					// Initialize female partner
					if (couple.getFemalePartnerId() != null) {
						Patient femalePartner = (Patient) session.get(Patient.class, couple.getFemalePartnerId());
						if (femalePartner != null) {
							if (femalePartner.getPatientAllergies() != null) {
								femalePartner.getPatientAllergies().size();
							}
							if (femalePartner.getDeliveryAddresses() != null) {
								femalePartner.getDeliveryAddresses().size();
							}
						}
					}

					// Initialize male partner
					if (couple.getMalePartnerId() != null) {
						Patient malePartner = (Patient) session.get(Patient.class, couple.getMalePartnerId());
						if (malePartner != null) {
							if (malePartner.getPatientAllergies() != null) {
								malePartner.getPatientAllergies().size();
							}
							if (malePartner.getDeliveryAddresses() != null) {
								malePartner.getDeliveryAddresses().size();
							}
						}
					}
				}
			}
		} catch (Exception e) {
			// Log but don't fail - some collections might be null
			PosLog.error(getReferenceClass(), e);
		}
	}

	/**
	 * Search CyclePlan by searchKeyword (patient name, doctor name, plan name)
	 * with pagination support
	 */
	public void findByPatientOrDoctorOrPlanName(String searchKeyword, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(CyclePlan.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			if (StringUtils.isNotEmpty(searchKeyword)) {
				// Create a disjunction (OR condition) for searching across multiple fields
				Disjunction disjunction = Restrictions.disjunction();

				// Search by doctor name (direct field in CyclePlan)
				disjunction.add(Restrictions.ilike(CyclePlan.PROP_DOCTOR_NAME, searchKeyword, MatchMode.ANYWHERE));

				// Search by protocol name (plan name - direct field in CyclePlan)
				disjunction.add(Restrictions.ilike(CyclePlan.PROP_PROTOCOL_NAME, searchKeyword, MatchMode.ANYWHERE));

				// Search by patient name - need to use subquery
				// Find all Couple IDs where female or male partner name matches
				DetachedCriteria coupleSubquery = DetachedCriteria.forClass(Couple.class);
				coupleSubquery.setProjection(Projections.property(Couple.PROP_ID));

				// Create criteria for Patient search
				DetachedCriteria femalePatientSubquery = DetachedCriteria.forClass(Patient.class);
				femalePatientSubquery.add(Restrictions.ilike(Customer.PROP_NAME, searchKeyword, MatchMode.ANYWHERE));
				femalePatientSubquery.setProjection(Projections.property(Patient.PROP_ID));

				DetachedCriteria malePatientSubquery = DetachedCriteria.forClass(Patient.class);
				malePatientSubquery.add(Restrictions.ilike(Customer.PROP_NAME, searchKeyword, MatchMode.ANYWHERE));
				malePatientSubquery.setProjection(Projections.property(Patient.PROP_ID));

				// Couple IDs where female or male partner matches the search
				Disjunction coupleDisjunction = Restrictions.disjunction();
				coupleDisjunction.add(Property.forName(Couple.PROP_FEMALE_PARTNER_ID).in(femalePatientSubquery));
				coupleDisjunction.add(Property.forName(Couple.PROP_MALE_PARTNER_ID).in(malePatientSubquery));
				coupleSubquery.add(coupleDisjunction);

				// Add to main disjunction - CyclePlan.coupleId matches the found couple IDs
				disjunction.add(Property.forName(CyclePlan.PROP_COUPLE_ID).in(coupleSubquery));

				criteria.add(disjunction);
			}

			// Set pagination
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			// Order by last update time (most recent first)
			criteria.addOrder(Order.desc(CyclePlan.PROP_LAST_UPDATE_TIME));

			tableModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

}
