package com.floreantpos.model.dao;

import java.io.Serializable;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.Conjunction;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.CyclePlan;
import com.floreantpos.model.CycleStageLog;
import com.floreantpos.swing.PaginationSupport;


public class CycleStageLogDAO extends BaseCycleStageLogDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public CycleStageLogDAO () {}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		Serializable serializable = super.save(obj, s);
		return serializable;
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public void delete(Object obj, Session session) throws HibernateException {
		CycleStageLog cycleStageLog = (CycleStageLog) obj;
		if (cycleStageLog == null) {
			throw new PosException(Messages.getString("MenuItemDAO.0")); //$NON-NLS-1$
		}

		cycleStageLog.setDeleted(true);

		session.update(cycleStageLog);
	}

	private void doCheckValidation(Object obj) {
		CycleStageLog cycleStageLog = (CycleStageLog) obj;
		// Add validation logic here if needed
	}

	/**
	 * Search CycleStageLog by searchKeyword (stage name, action name, performer name) with pagination support
	 */
	public void findBySearchKeyword(String searchKeyword, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(CycleStageLog.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Apply search filter if provided
			if (StringUtils.isNotEmpty(searchKeyword)) {
				Disjunction disjunction = Restrictions.disjunction();

				// Search by stage name
				disjunction.add(Restrictions.ilike(CycleStageLog.PROP_STAGE_NAME, searchKeyword, MatchMode.ANYWHERE));

				// Search by action name
				disjunction.add(Restrictions.ilike(CycleStageLog.PROP_ACTION_NAME, searchKeyword, MatchMode.ANYWHERE));

				// Search by performer name
				disjunction.add(Restrictions.ilike(CycleStageLog.PROP_PERFORMER_NAME, searchKeyword, MatchMode.ANYWHERE));

				// Search by note
				disjunction.add(Restrictions.ilike(CycleStageLog.PROP_NOTE, searchKeyword, MatchMode.ANYWHERE));

				criteria.add(disjunction);
			}

			// Set pagination
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			// Order by created date (most recent first)
			criteria.addOrder(Order.desc(CycleStageLog.PROP_CREATED_DATE));

			tableModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Find cycle stage logs by cycle plan ID with pagination
	 * Includes logs for the CyclePlan itself AND all its ProtocolPhases
	 */
	public void findByCyclePlanId(String cyclePlanId, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(CycleStageLog.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Filter by cycle plan ID and its protocol phases
			if (StringUtils.isNotEmpty(cyclePlanId)) {
				// Load the CyclePlan to get its ProtocolPhases
				CyclePlan cyclePlan = (CyclePlan) session.get(CyclePlan.class, cyclePlanId);

				if (cyclePlan != null) {
					Disjunction disjunction = Restrictions.disjunction();

					// Include logs for the CyclePlan itself
					Conjunction cyclePlanCondition = Restrictions.conjunction();
					cyclePlanCondition.add(Restrictions.eq(CycleStageLog.PROP_ENTITY_TYPE, CyclePlan.REF));
					cyclePlanCondition.add(Restrictions.eq(CycleStageLog.PROP_ENTITY_ID, cyclePlanId));
					disjunction.add(cyclePlanCondition);

					// Include logs for all ProtocolPhases of this CyclePlan
					if (cyclePlan.getProtocolPhases() != null && !cyclePlan.getProtocolPhases().isEmpty()) {
						java.util.List<String> phaseIds = new java.util.ArrayList<>();
						for (Object phaseObj : cyclePlan.getProtocolPhases()) {
							com.floreantpos.model.ProtocolPhase phase = (com.floreantpos.model.ProtocolPhase) phaseObj;
							if (phase.getId() != null) {
								phaseIds.add(phase.getId());
							}
						}

						if (!phaseIds.isEmpty()) {
							Conjunction phaseCondition = Restrictions.conjunction();
							phaseCondition.add(Restrictions.eq(CycleStageLog.PROP_ENTITY_TYPE, com.floreantpos.model.ProtocolPhase.REF));
							phaseCondition.add(Restrictions.in(CycleStageLog.PROP_ENTITY_ID, phaseIds));
							disjunction.add(phaseCondition);
						}
					}

					criteria.add(disjunction);
				}
			}

			// Set pagination
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			// Order by created date (chronological order)
			criteria.addOrder(Order.asc(CycleStageLog.PROP_CREATED_DATE));

			tableModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Find cycle stage logs by stage name with pagination
	 */
	public void findByStageName(String stageName, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(CycleStageLog.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Filter by stage name
			if (StringUtils.isNotEmpty(stageName)) {
				criteria.add(Restrictions.eq(CycleStageLog.PROP_STAGE_NAME, stageName));
			}

			// Set pagination
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			// Order by created date (most recent first)
			criteria.addOrder(Order.desc(CycleStageLog.PROP_CREATED_DATE));

			tableModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Find cycle stage logs by entity type and entity ID with pagination
	 * This is a generic method to find logs for any entity type (CyclePlan, ProtocolPhase, etc.)
	 */
	public void findByEntityTypeAndId(String entityType, String entityId, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(CycleStageLog.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Filter by entity type and ID
			if (StringUtils.isNotEmpty(entityType)) {
				criteria.add(Restrictions.eq(CycleStageLog.PROP_ENTITY_TYPE, entityType));
			}
			if (StringUtils.isNotEmpty(entityId)) {
				criteria.add(Restrictions.eq(CycleStageLog.PROP_ENTITY_ID, entityId));
			}

			// Set pagination
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			// Order by created date (chronological order)
			criteria.addOrder(Order.asc(CycleStageLog.PROP_CREATED_DATE));

			tableModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Find cycle stage logs by entity type only with pagination
	 */
	public void findByEntityType(String entityType, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(CycleStageLog.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Filter by entity type
			if (StringUtils.isNotEmpty(entityType)) {
				criteria.add(Restrictions.eq(CycleStageLog.PROP_ENTITY_TYPE, entityType));
			}

			// Set pagination
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			// Order by created date (most recent first)
			criteria.addOrder(Order.desc(CycleStageLog.PROP_CREATED_DATE));

			tableModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

}
