package com.floreantpos.model.dao;

import java.io.Serializable;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.CyclePlan;
import com.floreantpos.model.Embryo;
import com.floreantpos.swing.PaginationSupport;


public class EmbryoDAO extends BaseEmbryoDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public EmbryoDAO () {}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		Serializable serializable = super.save(obj, s);
		return serializable;
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public void delete(Object obj, Session session) throws HibernateException {
		Embryo embryo = (Embryo) obj;
		if (embryo == null) {
			throw new PosException(Messages.getString("MenuItemDAO.0")); //$NON-NLS-1$
		}

		embryo.setDeleted(true);

		session.update(embryo);
	}

	private void doCheckValidation(Object obj) {
		Embryo embryo = (Embryo) obj;
		// Add validation logic here if needed
	}

	/**
	 * Search Embryo by searchKeyword (grade, status, biopsy status) with pagination support
	 */
	public void findBySearchKeyword(String searchKeyword, String status, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(Embryo.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Apply search filter if provided
			if (StringUtils.isNotEmpty(searchKeyword)) {
				Disjunction disjunction = Restrictions.disjunction();

				// Search by grade
				disjunction.add(Restrictions.ilike(Embryo.PROP_GRADE, searchKeyword, MatchMode.ANYWHERE));

				// Search by status
				disjunction.add(Restrictions.ilike(Embryo.PROP_STATUS, searchKeyword, MatchMode.ANYWHERE));

				// Search by biopsy status
				disjunction.add(Restrictions.ilike(Embryo.PROP_BIOPSY_STATUS, searchKeyword, MatchMode.ANYWHERE));

				criteria.add(disjunction);
			}

			// Apply status filter if provided
			if (StringUtils.isNotEmpty(status)) {
				criteria.add(Restrictions.eq(Embryo.PROP_STATUS, status));
			}

			// Set pagination
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			// Order by last update time (most recent first)
			criteria.addOrder(Order.desc(Embryo.PROP_LAST_UPDATE_TIME));

			java.util.List<Embryo> embryos = criteria.list();

			// Initialize CyclePlan proxies before session closes
			for (Embryo embryo : embryos) {
				if (embryo.getCyclePlan() != null) {
					org.hibernate.Hibernate.initialize(embryo.getCyclePlan());
				}
			}

			tableModel.setRows(embryos);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Find embryos by cycle plan ID with pagination
	 */
	public void findByCyclePlanId(String cyclePlanId, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(Embryo.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Filter by cycle plan ID directly without creating alias
			if (StringUtils.isNotEmpty(cyclePlanId)) {
				CyclePlan cyclePlan = (CyclePlan) session.get(CyclePlan.class, cyclePlanId);
				if (cyclePlan != null) {
					criteria.add(Restrictions.eq(Embryo.PROP_CYCLE_PLAN, cyclePlan));
				}
			}

			// Set pagination
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			// Order by day of development
			criteria.addOrder(Order.asc(Embryo.PROP_DAY_OF_DEVELOPMENT));

			java.util.List<Embryo> embryos = criteria.list();

			// Initialize CyclePlan proxies before session closes
			for (Embryo embryo : embryos) {
				if (embryo.getCyclePlan() != null) {
					org.hibernate.Hibernate.initialize(embryo.getCyclePlan());
				}
			}

			tableModel.setRows(embryos);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Find all embryos by transfer ID
	 * Returns list of embryos associated with a specific embryo transfer
	 * Note: CyclePlan is initialized but its nested collections are NOT initialized
	 * to avoid LazyInitializationException. Use service layer to handle nested data.
	 */
	public java.util.List<Embryo> findByTransferId(String transferId) {
		if (StringUtils.isEmpty(transferId)) {
			return new java.util.ArrayList<>();
		}

		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(Embryo.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Filter by transfer ID
			criteria.add(Restrictions.eq(Embryo.PROP_TRANSFER_ID, transferId));

			// Order by day of development
			criteria.addOrder(Order.asc(Embryo.PROP_DAY_OF_DEVELOPMENT));

			java.util.List<Embryo> embryos = criteria.list();

			// Initialize CyclePlan proxies before session closes
			// But DO NOT initialize CyclePlan's nested collections to prevent LazyInitializationException
			for (Embryo embryo : embryos) {
				if (embryo.getCyclePlan() != null) {
					CyclePlan cyclePlan = embryo.getCyclePlan();
					org.hibernate.Hibernate.initialize(cyclePlan);

					// Clear nested collections if already initialized to prevent serialization issues
					if (org.hibernate.Hibernate.isInitialized(cyclePlan.getProtocolPhases())) {
						cyclePlan.getProtocolPhases().clear();
					}
					if (org.hibernate.Hibernate.isInitialized(cyclePlan.getEmbryos())) {
						cyclePlan.getEmbryos().clear();
					}
					if (org.hibernate.Hibernate.isInitialized(cyclePlan.getCycleStageLogs())) {
						cyclePlan.getCycleStageLogs().clear();
					}
				}
			}

			return embryos;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

}
