package com.floreantpos.model.dao;

import java.io.Serializable;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.Couple;
import com.floreantpos.model.CyclePlan;
import com.floreantpos.model.EmbryoTransfer;
import com.floreantpos.swing.PaginationSupport;

public class EmbryoTransferDAO extends BaseEmbryoTransferDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public EmbryoTransferDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		Serializable serializable = super.save(obj, s);
		return serializable;
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public void delete(Object obj, Session session) throws HibernateException {
		EmbryoTransfer embryoTransfer = (EmbryoTransfer) obj;
		if (embryoTransfer == null) {
			throw new PosException(Messages.getString("MenuItemDAO.0")); //$NON-NLS-1$
		}

		embryoTransfer.setDeleted(true);

		session.update(embryoTransfer);
	}

	private void doCheckValidation(Object obj) {
		EmbryoTransfer embryoTransfer = (EmbryoTransfer) obj;
		// Add validation logic here if needed
	}

	/**
	 * Search EmbryoTransfer by searchKeyword (couple name, cycle plan, transfer type, outcome)
	 * with pagination support
	 */
	public void findBySearchKeyword(String searchKeyword, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(EmbryoTransfer.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			if (StringUtils.isNotEmpty(searchKeyword)) {
				// Create a disjunction (OR condition) for searching across multiple fields
				Disjunction disjunction = Restrictions.disjunction();

				// Search by transfer type
				disjunction.add(Restrictions.ilike(EmbryoTransfer.PROP_TRANSFER_TYPE, searchKeyword, MatchMode.ANYWHERE));

				// Search by outcome
				disjunction.add(Restrictions.ilike(EmbryoTransfer.PROP_OUTCOME, searchKeyword, MatchMode.ANYWHERE));

				// Search by performer name
				disjunction.add(Restrictions.ilike(EmbryoTransfer.PROP_PERFORMER_NAME, searchKeyword, MatchMode.ANYWHERE));

				// Search by couple ID (exact match)
				disjunction.add(Restrictions.ilike(EmbryoTransfer.PROP_COUPLE_ID, searchKeyword, MatchMode.ANYWHERE));

				// Search by cycle plan ID (exact match)
				disjunction.add(Restrictions.ilike(EmbryoTransfer.PROP_CYCLE_PLAN_ID, searchKeyword, MatchMode.ANYWHERE));

				criteria.add(disjunction);
			}

			// Set pagination
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			// Order by transfer date (most recent first)
			criteria.addOrder(Order.desc(EmbryoTransfer.PROP_TRANSFER_DATE));

			tableModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Load full embryo transfer with all lazy collections initialized
	 */
	public EmbryoTransfer loadFullEmbryoTransfer(String embryoTransferId) {
		Session session = null;
		try {
			session = createNewSession();
			EmbryoTransfer embryoTransfer = (EmbryoTransfer) get(EmbryoTransfer.class, embryoTransferId, session);
			if (embryoTransfer != null) {
				initializeEmbryoTransfer(embryoTransfer, session);
			}
			return embryoTransfer;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Load full embryo transfer with all lazy collections initialized
	 */
	public EmbryoTransfer loadFullEmbryoTransfer(EmbryoTransfer embryoTransfer) {
		Session session = null;
		try {
			session = createNewSession();
			session.refresh(embryoTransfer);
			initializeEmbryoTransfer(embryoTransfer, session);
			return embryoTransfer;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Initialize lazy collections for an embryo transfer within an active session
	 */
	public void initializeEmbryoTransfer(EmbryoTransfer embryoTransfer, Session session) {
		if (embryoTransfer == null) {
			return;
		}

		try {
			// Initialize embryos collection - force load by calling size()
			if (embryoTransfer.getEmbryos() != null) {
				embryoTransfer.getEmbryos().size();
			}

			// Initialize couple if couple ID exists
			if (embryoTransfer.getCoupleId() != null) {
				Couple couple = (Couple) session.get(Couple.class, embryoTransfer.getCoupleId());
				if (couple != null) {
					org.hibernate.Hibernate.initialize(couple);
				}
			}

			// Initialize cycle plan with all nested collections if cycle plan ID exists
			if (embryoTransfer.getCyclePlanId() != null) {
				CyclePlan cyclePlan = (CyclePlan) session.get(CyclePlan.class, embryoTransfer.getCyclePlanId());
				if (cyclePlan != null) {
					// Use CyclePlanDAO to initialize all nested collections
					CyclePlanDAO cyclePlanDAO = new CyclePlanDAO();
					cyclePlanDAO.initializeCyclePlan(cyclePlan, session);
				}
			}
		} catch (Exception e) {
			// Log but don't fail - some collections might be null
			com.floreantpos.PosLog.error(getReferenceClass(), e);
		}
	}

}