package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.SpermSample;
import com.floreantpos.swing.PaginationSupport;


public class SpermSampleDAO extends BaseSpermSampleDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public SpermSampleDAO () {}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		Serializable serializable = super.save(obj, s);
		return serializable;
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public void delete(Object obj, Session session) throws HibernateException {
		SpermSample spermSample = (SpermSample) obj;
		if (spermSample == null) {
			throw new PosException(Messages.getString("MenuItemDAO.0")); //$NON-NLS-1$
		}

		spermSample.setDeleted(true);

		session.update(spermSample);
	}

	private void doCheckValidation(Object obj) {
		SpermSample spermSample = (SpermSample) obj;
		// Add validation logic here if needed
	}

	/**
	 * Search SpermSample by searchKeyword (sperm source, fertilization method) with pagination support
	 */
	public void findBySearchKeyword(String searchKeyword, String spermSource, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(SpermSample.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Apply search filter if provided
			if (StringUtils.isNotEmpty(searchKeyword)) {
				Disjunction disjunction = Restrictions.disjunction();

				// Search by sperm source
				disjunction.add(Restrictions.ilike(SpermSample.PROP_SPERM_SOURCE, searchKeyword, MatchMode.ANYWHERE));

				// Search by fertilization method
				disjunction.add(Restrictions.ilike(SpermSample.PROP_FERTILIZATION_METHOD, searchKeyword, MatchMode.ANYWHERE));

				// Search by male partner ID
				disjunction.add(Restrictions.ilike(SpermSample.PROP_MALE_PARTNER_ID, searchKeyword, MatchMode.ANYWHERE));

				// Search by donor ID
				disjunction.add(Restrictions.ilike(SpermSample.PROP_DONOR_ID, searchKeyword, MatchMode.ANYWHERE));

				criteria.add(disjunction);
			}

			// Apply sperm source filter if provided
			if (StringUtils.isNotEmpty(spermSource)) {
				criteria.add(Restrictions.eq(SpermSample.PROP_SPERM_SOURCE, spermSource));
			}

			// Set pagination
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			// Order by collection date (most recent first)
			criteria.addOrder(Order.desc(SpermSample.PROP_COLLECTION_DATE));

			List<SpermSample> samples = criteria.list();

			tableModel.setRows(samples);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Find frozen, unused sperm samples by male partner ID
	 */
	public List<SpermSample> findAvailableFrozenByMalePartner(String malePartnerId) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(SpermSample.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Filter by male partner ID
			if (StringUtils.isNotEmpty(malePartnerId)) {
				criteria.add(Restrictions.eq(SpermSample.PROP_MALE_PARTNER_ID, malePartnerId));
			}

			// Only frozen samples
			criteria.add(Restrictions.eq(SpermSample.PROP_FROZEN, true));

			// Only unused samples
			criteria.add(Restrictions.eq(SpermSample.PROP_USED, false));

			// Order by freeze date
			criteria.addOrder(Order.asc(SpermSample.PROP_FREEZE_DATE));

			return criteria.list();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Find sperm sample used in a specific cycle plan
	 */
	public SpermSample findByCyclePlanId(String cyclePlanId) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(SpermSample.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Filter by cycle plan ID
			if (StringUtils.isNotEmpty(cyclePlanId)) {
				criteria.add(Restrictions.eq(SpermSample.PROP_USED_IN_CYCLE_PLAN_ID, cyclePlanId));
			}

			return (SpermSample) criteria.uniqueResult();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * Find all sperm samples by male partner ID (both frozen and used)
	 */
	public List<SpermSample> findByMalePartner(String malePartnerId) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(SpermSample.class);

			// Add deleted filter
			addDeletedFilter(criteria);

			// Filter by male partner ID
			if (StringUtils.isNotEmpty(malePartnerId)) {
				criteria.add(Restrictions.eq(SpermSample.PROP_MALE_PARTNER_ID, malePartnerId));
			}

			// Order by collection date (most recent first)
			criteria.addOrder(Order.desc(SpermSample.PROP_COLLECTION_DATE));

			return criteria.list();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

}
