package com.orocube.rest.service.mqtt;

import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang.StringUtils;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.MqttCallbackExtended;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;
import org.json.JSONObject;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.hl7.model.Result;
import com.floreantpos.hl7.model.Test;
import com.floreantpos.model.Store;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.MqttCommand;
import com.floreantpos.util.POSUtil;

public class OroMqttClient extends OroMqttMessageReceiver implements MqttCallbackExtended {
	private static final String MQTT_SERVER_URL = "tcp://siiopa.com:1883"; //$NON-NLS-1$
	private MqttClient mqttClient;
	private int qos = 0;
	private final ExecutorService executorService = Executors.newSingleThreadExecutor();

	private static OroMqttClient instance;
	private String deviceId;

	private OroMqttClient() {
	}

	public static OroMqttClient getInstance() {
		if (instance == null) {
			instance = new OroMqttClient();
		}
		return instance;
	}

	public MqttClient getMqttClient() {
		return getInstance().mqttClient;
	}

	public boolean isConnectedToServer() {
		if (mqttClient == null) {
			return false;
		}
		if (mqttClient.isConnected())
			return true;
		return false;
	}

	public void initMqttClient(String deviceIdStr) {
		this.deviceId = deviceIdStr;

		executorService.submit(() -> {
			try {
				mqttClient = new MqttClient(MQTT_SERVER_URL, deviceId, new MemoryPersistence());
				MqttConnectOptions connOpts = new MqttConnectOptions();
				connOpts.setCleanSession(true);
				connOpts.setConnectionTimeout(2000);
				connOpts.setAutomaticReconnect(true);
				mqttClient.setCallback(this);
				mqttClient.connect(connOpts);

			} catch (Exception e) {
				PosLog.error(getClass(), "Failed to initialize MQTT client: " + e.getMessage());
			}
		});
	}

	public void disconnect() {
		if (mqttClient == null) {
			return;
		}
		executorService.submit(() -> {
			try {
				mqttClient.disconnect();
			} catch (MqttException e) {
				PosLog.error(getClass(), "Error disconnecting MQTT client: " + e.getMessage());
			}
		});
	}

	public void shutdown() {
		try {
			executorService.shutdown();
			if (!executorService.awaitTermination(5, TimeUnit.SECONDS)) {
				executorService.shutdownNow();
			}
			disconnect();
		} catch (InterruptedException e) {
			executorService.shutdownNow();
			Thread.currentThread().interrupt();
		}
	}

	protected void showMqttNotification(String msg) {
		PosLog.info(getClass(), msg);
	}

	public void publishOnThread(final String topic, final String msg) {
		publishData(topic, msg);
	}

	public void publishData(final String topic, final String msg) {
		publishData(topic, msg, false);
	}

	public void publishData(final String topic, final String msg, boolean compress) {
		publishData(getStoreUuid(), topic, msg, compress);
	}

	public void publishData(final String storeId, final String topic, final String msg, boolean compress) {
		if (mqttClient == null) {
			return;
		}
		if (!isConnectedToServer())
			return;

		executorService.submit(() -> {
			try {
				PosLog.debug(getClass(), "publishing to topic: " + storeId + topic); //$NON-NLS-1$
				mqttClient.publish(storeId + topic, compress ? POSUtil.compress(msg) : msg.getBytes(), qos, false);
			} catch (Exception me) {
				PosLog.error(getClass(), "Notification service error. " + me.getMessage()); //$NON-NLS-1$
			}
		});
	}

	public String getDeviceId() {
		return deviceId;
	}

	private String getStoreUuid() {
		Store store = DataProvider.get().getStore();
		if (store != null) {
			return store.getUuid() + "/"; //$NON-NLS-1$
		}

		return UUID.randomUUID().toString() + "/"; //$NON-NLS-1$
	}

	public void publishToOutlet(String topic, String message, String outletId) {
		//		try {
		//			if (StringUtils.isBlank(outletId)) {
		//				return;
		//			}
		//			OroMqttClient.getInstance().getMqttClient().publish(getStoreUuid() + outletId + "/" + topic, message.getBytes(), qos, false);
		//		} catch (Exception e) {
		//			PosLog.error(OroMqttMessagePublisher.class, e.getMessage());
		//		}

		if (mqttClient == null) {
			return;
		}
		if (!isConnectedToServer())
			return;

		executorService.submit(() -> {
			try {
				String publishTo = getStoreUuid() + outletId + "/" + topic;
				PosLog.debug(getClass(), publishTo); //$NON-NLS-1$

				mqttClient.publish(publishTo, message.getBytes(), qos, false);
			} catch (Exception me) {
				PosLog.error(getClass(), "Notification service error. " + me.getMessage()); //$NON-NLS-1$
			}
		});
	}

	@Override
	public void connectionLost(Throwable arg0) {
		PosLog.debug(getClass(), "mqtt connection lost. client id: " + mqttClient.getClientId()); //$NON-NLS-1$
	}

	@Override
	public void deliveryComplete(IMqttDeliveryToken arg0) {
	}

	@Override
	public void connectComplete(boolean reconnect, String serverURI) {
		PosLog.debug(getClass(), "connected to mqtt server. client id: " + mqttClient.getClientId());
	}

	public void notifyDataUpdated(Class<?> beanClass) {
		if (beanClass == null) {
			return;
		}
		if (DataProvider.get().isCaching(beanClass)) {
			JSONObject jsonObject = new JSONObject();
			jsonObject.put("terminalKey", DataProvider.get().getMqttDeviceId()); //$NON-NLS-1$
			publishData(MqttCommand.TOPIC_REFRESH_CACHE, jsonObject.toString());
		}
	}

	public void notifyStoreClosed() {
		JSONObject jsonObject = new JSONObject();
		Terminal terminal = DataProvider.get().getCurrentTerminal();
		jsonObject.put("terminalKey", DataProvider.get().getMqttDeviceId()); //$NON-NLS-1$
		jsonObject.put("terminalName", terminal.getName()); //$NON-NLS-1$
		publishData(MqttCommand.TOPIC_STORE_CLOSED, jsonObject.toString());
	}

	public void subscribeToMedlogicsData(String storeId) {
		if (StringUtils.isNotBlank(storeId)) {
			subscribe(storeId + "/" + MqttCommand.TOPIC_MEDLOGICS_DATA, qos);
		}
	}

	public void subscribeToMedlogicsGlobalTopics() {
		subscribe(MqttCommand.TOPIC_MEDLOGICS_NOTIFICATION, qos);
	}

	private void subscribe(String topic, int qos) {
		if (mqttClient == null || topic == null) {
			return;
		}
		if (!mqttClient.isConnected()) {
			PosLog.error(getClass(), Messages.getString("OroMqttClient.4")); //$NON-NLS-1$
			return;
		}

		executorService.submit(() -> {
			try {
				mqttClient.subscribe(topic, 2, this);
				PosLog.debug(getClass(), "subscribed to topic: " + topic); //$NON-NLS-1$
			} catch (Exception e) {
				PosLog.error(getClass(), "Error subscribing to topic " + topic + ": " + e.getMessage());
			}
		});
	}

	public void publishHl7Data(Test test, String msg) {
		StringBuilder builder = new StringBuilder();
		builder.append(msg + "\n"); //$NON-NLS-1$

		Set<String> keySet = test.resultmap.keySet();
		for (String key : keySet) {
			Result result = test.resultmap.get(key);
			builder.append(key + ":" + result.result + "\n"); //$NON-NLS-1$ //$NON-NLS-2$
		}

		publishData(MqttCommand.TOPIC_MEDLOGICS_DATA, builder.toString());
	}

}