package com.orocube.siiopa.accounting.client.service;

import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.Tax;
import com.floreantpos.model.dao.LedgerEntryDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.report.model.BalanceSheetLedgerEntryModel;
import com.floreantpos.util.NumberUtil;

public class BalanceSheetService {

	private static final String ASSET = "asset"; //$NON-NLS-1$
	private static final String CURRENT_ASSET = "current_asset"; //$NON-NLS-1$
	private static final String INVENTORY = "inventory"; //$NON-NLS-1$
	private static final String CURRENT_LIABILITY = "current_liability"; //$NON-NLS-1$
	private static final String LIABILITIES = "liabilities"; //$NON-NLS-1$
	private static final String EQUITY = "equity"; //$NON-NLS-1$

	private Date startDate;
	private Date endDate;

	private List<BalanceSheetLedgerEntryModel> currentAssetList;
	private List<BalanceSheetLedgerEntryModel> assetList;

	private List<BalanceSheetLedgerEntryModel> currentLiabilityList;
	private List<BalanceSheetLedgerEntryModel> liabilitiesList;

	private List<BalanceSheetLedgerEntryModel> equityList;
	private double totalExpenseAamount;
	private BalanceSheetLedgerEntryModel retainedEarning;

	public BalanceSheetService(Date startDate, Date endDate) {
		this.startDate = startDate;
		this.endDate = endDate;

		searchData();

	}

	@SuppressWarnings("unchecked")
	private void searchData() {

		Map<String, ChartOfAccounts> accountMap = new HashMap<>();

		assetList = LedgerEntryDAO.getInstance().getCustomLedgerEntries(startDate, endDate, Arrays.asList(ASSET), Arrays.asList(CURRENT_ASSET, INVENTORY), ""); //$NON-NLS-1$
		if (assetList != null) {
			for (BalanceSheetLedgerEntryModel balanceSheetLedgerEntryModel : assetList) {
				balanceSheetLedgerEntryModel.setAmount(NumberUtil.round(balanceSheetLedgerEntryModel.getAmount()));
				getCoaName(accountMap, balanceSheetLedgerEntryModel);
			}
		}

		currentAssetList = LedgerEntryDAO.getInstance().getLedgerEntriesByTypeId(Arrays.asList(CURRENT_ASSET, INVENTORY), startDate, endDate);
		if (currentAssetList != null) {
			for (BalanceSheetLedgerEntryModel balanceSheetLedgerEntryModel : currentAssetList) {
				balanceSheetLedgerEntryModel.setGroupName("Current asset");
				balanceSheetLedgerEntryModel.setAmount(NumberUtil.round(balanceSheetLedgerEntryModel.getAmount()));
				getCoaName(accountMap, balanceSheetLedgerEntryModel);
			}
		}

		currentLiabilityList = LedgerEntryDAO.getInstance().getLedgerEntriesByTypeId(CURRENT_LIABILITY, startDate, endDate);
		if (currentLiabilityList != null) {
			for (BalanceSheetLedgerEntryModel balanceSheetLedgerEntryModel : currentLiabilityList) {
				balanceSheetLedgerEntryModel.setGroupName("Current Liability");

				double amount = -1 * balanceSheetLedgerEntryModel.getAmount();
				balanceSheetLedgerEntryModel.setAmount(NumberUtil.round(amount));
				getCoaName(accountMap, balanceSheetLedgerEntryModel);
			}
		}

		liabilitiesList = LedgerEntryDAO.getInstance().getCustomLedgerEntries(startDate, endDate, Arrays.asList(LIABILITIES), CURRENT_LIABILITY, ""); //$NON-NLS-1$
		if (liabilitiesList != null) {
			for (BalanceSheetLedgerEntryModel balanceSheetLedgerEntryModel : liabilitiesList) {
				balanceSheetLedgerEntryModel.setGroupName("Long-term liabilities");

				double amount = -1 * balanceSheetLedgerEntryModel.getAmount();
				balanceSheetLedgerEntryModel.setAmount(NumberUtil.round(amount));
				getCoaName(accountMap, balanceSheetLedgerEntryModel);
			}
		}

		equityList = LedgerEntryDAO.getInstance().getCustomLedgerEntries(startDate, endDate, Arrays.asList(EQUITY), CURRENT_LIABILITY, ""); //$NON-NLS-1$
		if (equityList != null) {
			for (BalanceSheetLedgerEntryModel balanceSheetLedgerEntryModel : equityList) {

				double amount = -1 * balanceSheetLedgerEntryModel.getAmount();
				balanceSheetLedgerEntryModel.setAmount(NumberUtil.round(amount));
				getCoaName(accountMap, balanceSheetLedgerEntryModel);
			}
		}

		retainedEarning = LedgerEntryDAO.getInstance().getRetainedEarning(startDate, endDate);
		retainedEarning.setAmount(NumberUtil.round(-1 * retainedEarning.getAmount()));

		totalExpenseAamount = currentLiabilityList.stream().mapToDouble(e -> e.getAmount()).sum();
		totalExpenseAamount += liabilitiesList.stream().mapToDouble(e -> e.getAmount()).sum();
		totalExpenseAamount += equityList.stream().mapToDouble(e -> e.getAmount()).sum();
		totalExpenseAamount += retainedEarning.getAmount();

	}

	private void getCoaName(Map<String, ChartOfAccounts> accountMap, BalanceSheetLedgerEntryModel balanceSheetLedgerEntryModel) {
		String coaId = balanceSheetLedgerEntryModel.getCoaId();
		ChartOfAccounts chartOfAccounts = accountMap.get(coaId);
		if (chartOfAccounts == null) {
			chartOfAccounts = (ChartOfAccounts) DataProvider.get().getObjectOf(ChartOfAccounts.class, balanceSheetLedgerEntryModel.getCoaId());
			accountMap.put(coaId, chartOfAccounts);
		}
		balanceSheetLedgerEntryModel.setName(chartOfAccounts.getName());
	}

	public List<BalanceSheetLedgerEntryModel> getCurrentAssetList() {
		return currentAssetList;
	}

	public List<BalanceSheetLedgerEntryModel> getCurrentLiabilityList() {
		return currentLiabilityList;
	}

	public List<BalanceSheetLedgerEntryModel> getLiabilitiesList() {
		return liabilitiesList;
	}

	public List<BalanceSheetLedgerEntryModel> getEquityList() {
		return equityList;
	}

	public BalanceSheetLedgerEntryModel getRetainedEarning() {
		return retainedEarning;
	}

	public double getTotalExpenseAamount() {
		return totalExpenseAamount;
	}

	public List<BalanceSheetLedgerEntryModel> getAssetList() {
		return assetList;
	}

}
