/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.IUnit;
import com.floreantpos.model.InventoryClosingBalance;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryStock;
import com.floreantpos.model.InventoryStockUnit;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.PurchaseOrderItem;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.UnitType;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.model.util.InventoryUnitConvertionUtil;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;

public class InventoryTransactionDAO extends BaseInventoryTransactionDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public InventoryTransactionDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public InventoryTransaction initialize(InventoryTransaction inventoryTransaction) {
		if (inventoryTransaction == null || inventoryTransaction.getId() == null)
			return inventoryTransaction;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(inventoryTransaction);

			//Hibernate.initialize(InventoryTransaction.);

			return inventoryTransaction;
		} finally {
			closeSession(session);
		}
	}

	public Boolean convertInventoryUnit(Session session, MenuItem menuItem, InventoryStock stock, IUnit sourceInventoryUnit, IUnit destinationInventoryUnit,
			Double convertValue) {
		Boolean isConverted = false;
		Transaction transaction = null;
		try {
			transaction = session.beginTransaction();
			double convertedQuantity = NumberUtil.round(menuItem.getUnitQuantity(sourceInventoryUnit, destinationInventoryUnit) * convertValue);
			Date currentDate = new Date();
			InventoryTransaction outTrans = new InventoryTransaction();
			outTrans.setReason(InventoryTransaction.REASON_UNIT_CONVERSION);
			outTrans.setTransactionDate(currentDate);
			outTrans.setMenuItem(menuItem);
			outTrans.setType(InventoryTransactionType.OUT.getType());
			outTrans.setQuantity(convertValue);
			outTrans.setUnit(sourceInventoryUnit.getId());
			String srcUnitType = sourceInventoryUnit.getUnitType();
			outTrans.setUnitType(srcUnitType);
			if (UnitType.match(srcUnitType, UnitType.PACKAGING_UNIT)) {
				outTrans.setUnitCost(stock.getUnitCost());
				outTrans.setUnitPrice(stock.getUnitPrice());
			}
			else {
				outTrans.setUnitCost(menuItem.getCost());
			}

			if (stock != null) {
				outTrans.setBatchNumber(stock.getBatchNumber());
				outTrans.setMrpPrice(stock.getMrpPrice());
			}
			outTrans.calculateTotal();

			InventoryLocation inventoryLocation = stock == null ? null : stock.getInventoryLocation();
			if (stock != null) {
				outTrans.setFromInventoryLocation(inventoryLocation);
			}
			InventoryTransaction inTrans = new InventoryTransaction();
			inTrans.setReason(InventoryTransaction.REASON_UNIT_CONVERSION);
			inTrans.setTransactionDate(currentDate);
			inTrans.setMenuItem(menuItem);
			inTrans.setType(InventoryTransactionType.IN.getType());
			inTrans.setQuantity(convertedQuantity);
			inTrans.setUnit(destinationInventoryUnit.getId());
			String destUnitType = destinationInventoryUnit.getUnitType();
			inTrans.setUnitType(destUnitType);
			if (UnitType.match(destUnitType, UnitType.PACKAGING_UNIT)) {
				double unitCost = stock.getUnitCost() / sourceInventoryUnit.getConversionRate() * destinationInventoryUnit.getConversionRate();
				unitCost = NumberUtil.round(unitCost);

				double unitPrice = stock.getUnitPrice() / sourceInventoryUnit.getConversionRate() * destinationInventoryUnit.getConversionRate();
				unitPrice = NumberUtil.round(unitPrice);

				inTrans.setUnitCost(unitCost);
				inTrans.setUnitPrice(unitPrice);
			}
			else {
				inTrans.setUnitCost(menuItem.getCost());
			}
			inTrans.setBatchNumber(stock.getBatchNumber());
			inTrans.setMrpPrice(stock.getMrpPrice());
			inTrans.calculateTotal();
			if (stock != null) {
				inTrans.setToInventoryLocation(inventoryLocation);
				inTrans.setOutletId(inventoryLocation == null ? null : inventoryLocation.getOutletId());
			}
			this.adjustOutTransactionInventoryStock(outTrans, session, true, stock);
			InventoryStock inventoryStock = this.adjustInTransactionInventoryStock(inTrans, session, true, stock);
			if (inventoryStock != null && destinationInventoryUnit instanceof InventoryStockUnit) {
				InventoryStockUnit inventoryStockUnit = (InventoryStockUnit) destinationInventoryUnit;
				Double oldPrice = inventoryStockUnit.getPrice();
				Double newPrice = inventoryStock.getUnitPrice();
				if (oldPrice < newPrice) {
					inventoryStockUnit.setPrice(newPrice);
					inventoryStockUnit.setCost(inventoryStock.getUnitCost());
					InventoryStockUnitDAO.getInstance().saveOrUpdate(inventoryStockUnit, session);
				}
			}
			transaction.commit();
			isConverted = true;
		} catch (Exception e0) {
			if (transaction != null) {
				transaction.rollback();
			}
			PosLog.error(this.getReferenceClass(), e0);
		}
		return isConverted;
	}

	private void adjustOutTransactionInventoryStock(InventoryTransaction outTrans, Session session, boolean isConversion, InventoryStock sourceStock) {
		try {
			MenuItem menuItem = outTrans.getMenuItem();
			PosLog.debug(getClass(),
					menuItem.getDisplayName() + " |" + outTrans.getTransactionType() + "| |" + outTrans.getQuantity() + "| " + outTrans.getReason());
			InventoryLocation inventoryFromLocation = outTrans.getFromInventoryLocation();
			InventoryStock stockItem = InventoryStockDAO.getInstance().getInventoryStock(menuItem, inventoryFromLocation, outTrans.getUnit(),
					outTrans.getUnitType(), session, outTrans.getBatchNumber());
			calculateStockOutOnHandQty(outTrans, stockItem);
			double unitQuantity = this.createStockItem(outTrans, session, menuItem, 1L, stockItem, inventoryFromLocation, null, sourceStock);
			this.saveOrUpdate(outTrans, session);
			MenuItemDAO.getInstance().updateStockQuantity(menuItem.getId(), outTrans.getOutletId(), unitQuantity, !isConversion, !isConversion, session);
		} catch (Exception e0) {
			PosLog.error(this.getReferenceClass(), e0);
		}
	}

	private void calculateStockOutOnHandQty(InventoryTransaction outTrans, InventoryStock stockItem) {
		double quantityInHand = stockItem == null ? 0 : stockItem.getQuantityInHand();
		outTrans.putBeforeOnHandQty(quantityInHand);
		outTrans.putAfterOnHandQty(quantityInHand - outTrans.getQuantity());
	}

	private InventoryStock adjustInTransactionInventoryStock(InventoryTransaction inTrans, Session session, boolean isConversion, InventoryStock sourceStock) {
		try {
			MenuItem menuItem = inTrans.getMenuItem();
			PosLog.debug(getClass(),
					menuItem.getDisplayName() + " |" + inTrans.getTransactionType() + "| |" + inTrans.getQuantity() + "| " + inTrans.getReason());
			InventoryLocation inventoryToLocation = inTrans.getToInventoryLocation();
			InventoryStock stockItem = InventoryStockDAO.getInstance().getInventoryStock(menuItem, inventoryToLocation, inTrans.getUnit(),
					inTrans.getUnitType(), session, inTrans.getBatchNumber());
			calculateStockInOnHandQty(inTrans, stockItem);
			double unitQuantity = this.createStockItem(inTrans, session, menuItem, 1L, stockItem, inventoryToLocation, null, sourceStock);
			this.saveOrUpdate(inTrans, session);
			MenuItemDAO.getInstance().updateStockQuantity(menuItem.getId(), inTrans.getOutletId(), unitQuantity, !isConversion, !isConversion, session);
			return stockItem;
		} catch (Exception e0) {
			PosLog.error(this.getReferenceClass(), e0);
		}
		return null;
	}

	private void calculateStockInOnHandQty(InventoryTransaction inTrans, InventoryStock stockItem) {
		double quantityInHand = stockItem == null ? 0 : stockItem.getQuantityInHand();
		inTrans.putBeforeOnHandQty(quantityInHand);
		inTrans.putAfterOnHandQty(quantityInHand + inTrans.getQuantity());
	}

	public void adjustInventoryStock(InventoryTransaction trans, Session session) {
		adjustInventoryStock(trans, session, true, true);
	}

	public void adjustInventoryStock(InventoryTransaction trans, Session session, InventoryStock sourceStock) {
		adjustInventoryStock(trans, session, true, true, true, null, sourceStock);
	}

	public void adjustInventoryStock(InventoryTransaction trans, Session session, boolean updateAvailableUnit, boolean updateOnHandUnit) {
		adjustInventoryStock(trans, session, updateAvailableUnit, updateOnHandUnit, true);
	}

	public void adjustInventoryStock(InventoryTransaction trans, Session session, boolean updateAvailableUnit, boolean updateOnHandUnit,
			boolean updateDisplayUnit) {
		adjustInventoryStock(trans, session, updateAvailableUnit, updateOnHandUnit, updateDisplayUnit, null);
	}

	public void adjustInventoryStock(InventoryTransaction trans, Session session, boolean updateAvailableUnit, boolean updateOnHandUnit,
			boolean updateDisplayUnit, PurchaseOrderItem poItem) {
		adjustInventoryStock(trans, session, updateAvailableUnit, updateOnHandUnit, updateDisplayUnit, poItem, null);
	}

	public void adjustInventoryStock(InventoryTransaction trans, Session session, boolean updateAvailableUnit, boolean updateOnHandUnit,
			boolean updateDisplayUnit, PurchaseOrderItem poItem, InventoryStock sourceStock) {
		MenuItem menuItem = trans.getMenuItem();
		double baseUnitQuantity = menuItem.getBaseUnitQuantity(trans.getUnit());
		String baseUnit = "";
		if (menuItem.getUnit() != null)
			baseUnit = menuItem.getUnit().getId();
		trans.setBaseUnit(baseUnit);
		trans.setBaseUnitQuantity(baseUnitQuantity);
		double transQty = trans.getQuantity();
		if (updateDisplayUnit) {
			IUnit transactionUnit = DataProvider.get().getUnitById(trans.getUnit(), trans.getUnitType());
			trans.setDisplayUnit(transactionUnit, transQty, menuItem);
		}
		InventoryLocation inventoryFromLocation = null;
		InventoryLocation inventoryToLocation = null;
		InventoryStock stockItem = null;
		double unitQuantity = 0;

		PosLog.debug(getClass(), menuItem.getDisplayName() + " |" + trans.getTransactionType() + "| |" + transQty + "| " + trans.getReason());
		String batchNumber = poItem == null ? trans.getBatchNumber() : poItem.getBatchNumber();
		if (trans.getTransactionType() == InventoryTransactionType.IN) {
			inventoryToLocation = trans.getToInventoryLocation();
			stockItem = InventoryStockDAO.getInstance().getInventoryStock(menuItem, inventoryToLocation, trans.getUnit(), trans.getUnitType(), session,
					batchNumber);
			calculateStockInOnHandQty(trans, stockItem);
			unitQuantity = this.createStockItem(trans, session, menuItem, baseUnitQuantity, stockItem, inventoryToLocation, poItem, sourceStock);
			this.saveOrUpdate(trans, session);
			MenuItemDAO.getInstance().updateStockQuantity(menuItem.getId(), inventoryToLocation.getOutletId(), unitQuantity, updateAvailableUnit,
					updateOnHandUnit, session);
		}
		if (trans.getTransactionType() == InventoryTransactionType.OUT) {
			inventoryFromLocation = trans.getFromInventoryLocation();
			stockItem = InventoryStockDAO.getInstance().getInventoryStock(menuItem, inventoryFromLocation, trans.getUnit(), trans.getUnitType(), session,
					batchNumber);
			calculateStockOutOnHandQty(trans, stockItem);
			unitQuantity = this.createStockItem(trans, session, menuItem, baseUnitQuantity, stockItem, inventoryFromLocation, poItem, sourceStock);
			this.saveOrUpdate(trans, session);
			MenuItemDAO.getInstance().updateStockQuantity(menuItem.getId(), inventoryFromLocation.getOutletId(), unitQuantity, updateAvailableUnit,
					updateOnHandUnit, session);
		}
		if (trans.getTransactionType() == InventoryTransactionType.UNCHANGED) {
			inventoryFromLocation = trans.getFromInventoryLocation();
			inventoryToLocation = trans.getToInventoryLocation();

			createStockItemForTransfer(trans, session, menuItem, baseUnitQuantity, inventoryFromLocation, inventoryToLocation);
			InventoryTransactionDAO.getInstance().saveOrUpdate(trans, session);
		}
		if (trans.getTransactionType() == InventoryTransactionType.CONVERSION) {
			inventoryFromLocation = trans.getFromInventoryLocation();
			inventoryToLocation = trans.getToInventoryLocation();
			InventoryStock fromStock = InventoryStockDAO.getInstance().getInventoryStock(menuItem, inventoryFromLocation, trans.getUnit(), trans.getUnitType(),
					session);
			InventoryStock toStock = InventoryStockDAO.getInstance().getInventoryStock(menuItem, inventoryToLocation, trans.getBaseUnit(), trans.getUnitType(),
					session);
			createConversionStockItem(trans, session, menuItem, baseUnitQuantity, fromStock, toStock, inventoryToLocation);
			InventoryTransactionDAO.getInstance().saveOrUpdate(trans, session);
		}
	}

	private void createStockItemForTransfer(InventoryTransaction trans, Session session, MenuItem menuItem, double baseUnitQuantity,
			InventoryLocation fromLocation, InventoryLocation toLocation) {

		InventoryStockDAO instanceDao = InventoryStockDAO.getInstance();
		InventoryStock stockItemFromLocation = instanceDao.getInventoryStock(menuItem, fromLocation, trans.getUnit(), trans.getUnitType(), session);
		InventoryStock stockItemToLocation = instanceDao.getInventoryStock(menuItem, toLocation, trans.getUnit(), trans.getUnitType(), session);
		if (stockItemFromLocation == null) {
			return;
		}
		if (stockItemToLocation == null) {
			stockItemToLocation = new InventoryStock();
			stockItemToLocation.setOutletId(trans.getOutletId());
			stockItemToLocation.setMenuItem(menuItem);
			stockItemToLocation.setLocationId(toLocation.getId());
			stockItemToLocation.setUnit(trans.getUnit());
			stockItemToLocation.setUnitType(trans.getUnitType());
		}
		stockItemFromLocation.setQuantityInHand(NumberUtil.round(stockItemFromLocation.getQuantityInHand() - trans.getQuantity()));
		stockItemToLocation.setQuantityInHand(NumberUtil.round(stockItemToLocation.getQuantityInHand() + trans.getQuantity()));

		if (stockItemFromLocation.getQuantityInHand() == 0) {
			instanceDao.delete(stockItemFromLocation, session);
		}
		else {
			instanceDao.saveOrUpdate(stockItemFromLocation, session);
		}
		instanceDao.saveOrUpdate(stockItemToLocation, session);
	}

	//	private double createStockItem(InventoryTransaction trans, Session session, MenuItem menuItem, double baseUnitQuantity, InventoryStock stockItem,
	//			InventoryLocation inventoryLocation, PurchaseOrderItem poItem) {
	//		return createStockItem(trans, session, menuItem, baseUnitQuantity, stockItem, inventoryLocation, poItem, null);
	//	}

	private double createStockItem(InventoryTransaction trans, Session session, MenuItem menuItem, double baseUnitQuantity, InventoryStock stockItem,
			InventoryLocation inventoryLocation, PurchaseOrderItem poItem, InventoryStock sourceStock) {
		if (stockItem == null) {
			stockItem = new InventoryStock();
			if (trans.getTransactionType() == InventoryTransactionType.IN) {
				stockItem.setOutletId(trans.getToOultetId());
			}
			else {
				stockItem.setOutletId(trans.getOutletId());
			}
			stockItem.setMenuItem(menuItem);
			if (inventoryLocation != null) {
				stockItem.setLocationId(inventoryLocation.getId());
				stockItem.setOutletId(inventoryLocation.getOutletId());
			}
			stockItem.setUnit(trans.getUnit());
			stockItem.setUnitType(trans.getUnitType());
			stockItem.setUnitCost(trans.getUnitCost());
			if (trans.getUnitPrice() > 0.0) {
				stockItem.setUnitPrice(trans.getUnitPrice());
			}
		}

		if (ProductType.match(menuItem.getProductType(), ProductType.MEDICINE) || ProductType.match(menuItem.getProductType(), ProductType.GOODS)) {
			if (sourceStock != null) {
				stockItem.setBatchNumber(sourceStock.getBatchNumber());
				stockItem.setExpireDate(sourceStock.getExpireDate());
			}
			else if (poItem != null) {
				stockItem.setBatchNumber(poItem.getBatchNumber());
				stockItem.setExpireDate(poItem.getExpireDate());
			}
			else {
				stockItem.setBatchNumber(trans.getBatchNumber());
				stockItem.setExpireDate(trans.getExpiryDate());
			}

			if (trans.getTransactionType() == InventoryTransactionType.IN) {
				Double unitCost = stockItem.getUnitCost();
				double avgCost = (unitCost * Math.abs(stockItem.getQuantityInHand()) + trans.getUnitCost() * trans.getQuantity())
						/ (Math.abs(stockItem.getQuantityInHand()) + trans.getQuantity());

				Double unitPrice = stockItem.getUnitPrice();
				double avgSalesPrice = (unitPrice * Math.abs(stockItem.getQuantityInHand()) + trans.getUnitPrice() * trans.getQuantity())
						/ (Math.abs(stockItem.getQuantityInHand()) + trans.getQuantity());

				stockItem.setUnitCost(NumberUtil.round(avgCost));
				stockItem.setUnitPrice(NumberUtil.round(avgSalesPrice));
			}
			else {
				stockItem.setUnitCost(trans.getUnitCost());
				if (trans.getUnitPrice() > 0.0) {
					stockItem.setUnitPrice(trans.getUnitPrice());
				}
			}
		}
		else if (poItem != null) {
			stockItem.setBatchNumber(poItem.getBatchNumber());
			stockItem.setExpireDate(poItem.getExpireDate());
			stockItem.setUnitCost(poItem.getUnitCost());
			if (poItem.getSalesPrice() > 0.0) {
				stockItem.setUnitPrice(poItem.getSalesPrice());
			}
		}
		else {
			stockItem.setBatchNumber(trans.getBatchNumber());
			stockItem.setExpireDate(trans.getExpiryDate());

			stockItem.setUnitCost(trans.getUnitCost());
			if (trans.getUnitPrice() > 0.0) {
				stockItem.setUnitPrice(trans.getUnitPrice());
			}
		}

		if (trans.getTransactionType() == InventoryTransactionType.IN) {
			stockItem.setQuantityInHand(NumberUtil.round(stockItem.getQuantityInHand() + trans.getQuantity()));
		}
		if (trans.getTransactionType() == InventoryTransactionType.OUT) {
			stockItem.setQuantityInHand(NumberUtil.round(stockItem.getQuantityInHand() - trans.getQuantity()));
		}

		double unitQuantity = 0;
		if (trans.getTransactionType() == InventoryTransactionType.IN) {
			unitQuantity += (baseUnitQuantity * trans.getQuantity());
		}
		if (trans.getTransactionType() == InventoryTransactionType.OUT) {
			unitQuantity -= (baseUnitQuantity * trans.getQuantity());
		}
		InventoryStockDAO instance = InventoryStockDAO.getInstance();
		instance.saveOrUpdate(stockItem, session);

		return unitQuantity;
	}

	private void createConversionStockItem(InventoryTransaction trans, Session session, MenuItem menuItem, double baseUnitQuantity,
			InventoryStock fromStockItem, InventoryStock toStockItem, InventoryLocation inventoryLocation) {
		if (toStockItem == null) {
			toStockItem = new InventoryStock();
			if (trans.getTransactionType() == InventoryTransactionType.IN) {
				toStockItem.setOutletId(trans.getToOultetId());
			}
			else {
				toStockItem.setOutletId(trans.getOutletId());
			}
			if (inventoryLocation != null) {
				toStockItem.setLocationId(inventoryLocation.getId());
				toStockItem.setOutletId(inventoryLocation.getOutletId());
			}
			toStockItem.setMenuItem(menuItem);
			toStockItem.setLocationId(inventoryLocation.getId());
			toStockItem.setUnit(trans.getBaseUnit());
			toStockItem.setUnitType(trans.getUnitType());
		}

		toStockItem.setQuantityInHand(NumberUtil.round(toStockItem.getQuantityInHand() + trans.getQuantity() * baseUnitQuantity));
		fromStockItem.setQuantityInHand(NumberUtil.round(fromStockItem.getQuantityInHand() - trans.getQuantity()));

		InventoryStockDAO instance = InventoryStockDAO.getInstance();
		if (fromStockItem.getQuantityInHand() == 0)
			instance.delete(fromStockItem, session);
		else
			instance.saveOrUpdate(fromStockItem, session);
		instance.saveOrUpdate(toStockItem, session);
	}

	public void performInventoryTransaction(InventoryTransaction inventoryTransaction) {
		try (Session session = this.createNewSession()) {
			this.performInventoryTransaction(inventoryTransaction, session);
		} catch (Exception e0) {
			PosLog.error(getClass(), e0);
		}
	}

	public void performInventoryTransaction(InventoryTransaction inventoryTransaction, Session session) throws Exception {
		Transaction transaction = null;
		try {
			transaction = session.beginTransaction();
			this.adjustInventoryStock(inventoryTransaction, session);
			transaction.commit();
		} catch (Exception e0) {
			if (transaction != null) {
				transaction.rollback();
			}
			throw e0;
		}
	}

	public void performInventoryTransactions(InventoryTransaction... inventoryTransactions) {
		if (inventoryTransactions == null || inventoryTransactions.length == 0) {
			return;
		}
		Transaction transaction = null;
		try (Session session = this.createNewSession()) {
			transaction = session.beginTransaction();
			this.performInventoryTransactions(session, inventoryTransactions);
			transaction.commit();
		} catch (Exception e0) {
			PosLog.error(getClass(), e0);
			if (transaction != null) {
				transaction.rollback();
			}
		}
	}

	public void performInventoryTransactions(Session session, InventoryTransaction... inventoryTransactions) {
		if (inventoryTransactions == null || inventoryTransactions.length == 0) {
			return;
		}
		for (InventoryTransaction inventoryTransaction : inventoryTransactions) {
			this.adjustInventoryStock(inventoryTransaction, session);
			InventoryVendorItemsDAO.getInstance().saveItems(inventoryTransaction, session);
		}
	}

	/**
	 * @deprecated use performInventoryTransactions(InventoryTransaction... inventoryTransactions) or performInventoryTransactions(Session session, InventoryTransaction... inventoryTransactions) instead
	 * @param inventoryTransactions
	 */
	@Deprecated
	public void saveInventoryTransactionList(List<InventoryTransaction> inventoryTransactions) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (InventoryTransaction inventoryTransaction : inventoryTransactions) {
				adjustInventoryStock(inventoryTransaction, session);
				if (inventoryTransaction.getReason().equals(InventoryTransaction.REASON_PURCHASE)) {
					//					MenuItem menuItem = inventoryTransaction.getMenuItem();
					//					MenuItemDAO.getInstance().updateLastPurchaseCost(menuItem.getId(), inventoryTransaction.getUnitCost(),
					//							!DataProvider.get().getStore().isInventoryAvgPricingMethod(), session);
					InventoryVendorItemsDAO.getInstance().saveItems(inventoryTransaction, session);
				}
				else if (inventoryTransaction.getReason().equals(InventoryTransaction.REASON_NEW_STOCK)) {
					InventoryVendorItemsDAO.getInstance().saveItems(inventoryTransaction, session);
				}
			}
			tx.commit();

			//			session = createNewSession();
			//			tx = session.beginTransaction();
			//			Store store = DataProvider.get().getStore();
			//			boolean avgPricingMethod = store.isInventoryAvgPricingMethod();
			//			for (InventoryTransaction inventoryTransaction : inventoryTransactions) {
			//				MenuItem menuItem = inventoryTransaction.getMenuItem();
			//				if (avgPricingMethod) {
			//					double itemAvgCost = findItemAvgCost(menuItem);
			//					menuItem.setCost(ToTwoDigit(itemAvgCost));
			//				}
			//				else {
			//					menuItem.setCost(ToTwoDigit(inventoryTransaction.getUnitCost()));
			//				}
			//
			//				session.update(menuItem);
			//			}
			//			tx.commit();
		} catch (Exception ex) {
			PosLog.error(getClass(), ex);
			tx.rollback();
		} finally {
			closeSession(session);
		}
	}

	public void performInventoryTransfer(InventoryTransaction inTrans, InventoryTransaction outTrans) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			InventoryStockDAO instanceDao = InventoryStockDAO.getInstance();
			InventoryStock stockIn = instanceDao.getInventoryStock(inTrans.getMenuItem(), outTrans.getToInventoryLocation(), inTrans.getUnit(),
					inTrans.getUnitType());
			if (stockIn == null) {
				stockIn = new InventoryStock();
				stockIn.setOutletId(inTrans.getOutletId());
				stockIn.setMenuItem(inTrans.getMenuItem());
				stockIn.setLocationId(inTrans.getToLocationId());
				stockIn.setQuantityInHand(inTrans.getQuantity());
				stockIn.setUnit(inTrans.getUnit());
				stockIn.setUnitType(inTrans.getUnitType());
			}
			else {
				stockIn.setQuantityInHand(stockIn.getQuantityInHand() + inTrans.getQuantity());
			}

			InventoryStock stockOut = instanceDao.getInventoryStock(outTrans.getMenuItem(), outTrans.getFromInventoryLocation(), outTrans.getUnit(),
					outTrans.getUnitType());
			if (stockOut == null) {
				stockOut = new InventoryStock();
				stockOut.setOutletId(outTrans.getOutletId());
				stockOut.setMenuItem(outTrans.getMenuItem());
				stockOut.setLocationId(outTrans.getFromLocationId());
				stockOut.setQuantityInHand(outTrans.getQuantity());
				stockOut.setUnit(outTrans.getUnit());
			}
			else {
				stockOut.setQuantityInHand(stockOut.getQuantityInHand() - outTrans.getQuantity());
			}
			saveOrUpdate(inTrans, session);
			saveOrUpdate(outTrans, session);
			instanceDao.saveOrUpdate(stockIn, session);
			instanceDao.saveOrUpdate(stockOut, session);
			tx.commit();
		} catch (Exception ex) {
			PosLog.error(getClass(), ex);
			tx.rollback();
		} finally {
			closeSession(session);
		}
	}

	public void delete(String id) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			InventoryTransaction transaction = get(id, session);
			//MenuItem menuItem = transaction.getMenuItem();
			//menuItem.setAvailableUnit(menuItem.getAvailableUnit() - transaction.getQuantity());

			//MenuItemDAO.getInstance().saveOrUpdate(menuItem, session);
			delete(transaction, session);
			tx.commit();
		} catch (Exception ex) {
			PosLog.error(getClass(), ex);
			tx.rollback();
		} finally {
			closeSession(session);
		}
	}

	public InventoryTransaction getInventoryTransaction(int referenceId) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = getSession();
			criteria = session.createCriteria(InventoryTransaction.class);

			/*criteria.add(Restrictions.eq(InventoryStock.PROP_MENU_ITEM, menuItem));
			return (InventoryTransaction) criteria.uniqueResult();*/

			//	if (referenceId instanceof PurchaseOrder) {

			//		if (referenceId instanceof PurchaseOrder) {
			criteria.createAlias(InventoryTransaction.PROP_REFERENCE_NO, "item"); //$NON-NLS-1$
			criteria.add(Restrictions.eq("item.id", referenceId)); //$NON-NLS-1$

			//		}

			//criteria.add(Restrictions.eq(InventoryTransaction.PROP_REFERENCE_NO, referenceId));
			// criteria.add(Restrictions.and(Restrictions.eq("", ""), Restrictions.eq("", "")));
			//return (InventoryTransaction) criteria.uniqueResult();

			List<InventoryTransaction> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return (InventoryTransaction) result.get(0);
			//		}

		} catch (Exception e) {
			PosLog.error(getClass(), "" + e);
		} finally {
			closeSession(session);
		}
		return null;
	}

	public PurchaseOrder getPurchaseOrderId(String referenceId) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = getSession();
			criteria = session.createCriteria(PurchaseOrder.class);

			//criteria.createAlias(PurchaseOrder.PROP_ID, "item"); //$NON-NLS-1$
			//criteria.add(Restrictions.eq("item.id", referenceId)); //$NON-NLS-1$
			criteria.add(Restrictions.eq(PurchaseOrder.PROP_ORDER_ID, referenceId));

			List<PurchaseOrder> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return (PurchaseOrder) result.get(0);

		} catch (Exception e) {
			PosLog.debug(getClass(), "" + e);
		} finally {
			closeSession(session);
		}
		return null;
	}

	public List<InventoryTransaction> findTransactionsByType(String menuItemNameOrSku, List<Integer> transactionTypes, MenuGroup menuGroup,
			MenuCategory menuCategory, Date fromDate, Date toDate) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.createAlias(InventoryTransaction.PROP_MENU_ITEM, "menuItem");
			criteria.add(Restrictions.between(InventoryTransaction.PROP_TRANSACTION_DATE, fromDate, toDate));
			criteria.add(Restrictions.or(Restrictions.isNull("menuItem." + AppConstants.PROP_DELETED),
					Restrictions.eq("menuItem." + AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (transactionTypes != null && !transactionTypes.isEmpty()) {
				criteria.add(Restrictions.in(InventoryTransaction.PROP_TYPE, transactionTypes));
			}
			if (StringUtils.isNotEmpty(menuItemNameOrSku)) {
				criteria.add(Restrictions.or(Restrictions.ilike("menuItem.name", menuItemNameOrSku, MatchMode.ANYWHERE),
						Restrictions.eq("menuItem.sku", menuItemNameOrSku)));
			}
			if (menuGroup != null) {
				criteria.add(Restrictions.eq("menuItem." + MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}
			if (menuCategory != null) {
				criteria.add(Restrictions.eq("menuItem." + MenuItem.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}

			//criteria.addOrder(Order.asc(MenuItem.PROP_MENU_GROUP_ID));
			criteria.addOrder(Order.asc("menuItem.name"));
			criteria.addOrder(Order.asc(InventoryTransaction.PROP_TRANSACTION_DATE));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	@SuppressWarnings("unchecked")
	public Map<String, InventoryTransaction> populateOpeningBalance(Map<String, MenuItem> itemMap, Date endOfPreviousMonth, Date reportDate) {
		List<String> menuItemIds = new ArrayList<>(itemMap.keySet());
		if (menuItemIds == null || menuItemIds.isEmpty()) {
			return null;
		}
		Date lastClosingDate = InventoryClosingBalanceDAO.getInstance().getLastClosingDate(reportDate);
		List<InventoryTransaction> lastClosingTrans = lastClosingDate == null ? null : getClosingBalances(itemMap, menuItemIds, lastClosingDate);
		List<InventoryTransaction> afterClosingTrans = getInTransactions(menuItemIds, lastClosingDate, endOfPreviousMonth);
		return mergeLastClosingBalanceAndTrans(lastClosingTrans, afterClosingTrans);
	}

	private List<InventoryTransaction> getClosingBalances(Map<String, MenuItem> itemMap, List<String> menuItemIds, Date lastClosingDate) {
		List<InventoryClosingBalance> closingBalances = InventoryClosingBalanceDAO.getInstance().getClosingBalances(menuItemIds, lastClosingDate);
		if (closingBalances == null || closingBalances.isEmpty()) {
			return null;
		}
		List<InventoryTransaction> inventoryTransactions = new ArrayList<>();
		for (InventoryClosingBalance inventoryClosingBalance : closingBalances) {
			InventoryTransaction inventoryTransaction = new InventoryTransaction();
			inventoryTransaction.setUnit(inventoryClosingBalance.getUnit());
			inventoryTransaction.setQuantity(inventoryClosingBalance.getBalance());
			inventoryTransaction.setUnitCost(inventoryClosingBalance.getUnitCost());
			inventoryTransaction.setTotal(inventoryClosingBalance.getTotal());
			inventoryTransaction.setMenuItem(itemMap.get(inventoryClosingBalance.getMenuItemId()));
			inventoryTransactions.add(inventoryTransaction);
		}
		return inventoryTransactions;
	}

	private List<InventoryTransaction> getInTransactions(List<String> menuItemIds, Date fromDate, Date toDate) {
		List<InventoryTransaction> closingAfterTransactions;
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.groupProperty(InventoryTransaction.PROP_MENU_ITEM), InventoryTransaction.PROP_MENU_ITEM);
			projectionList.add(Projections.groupProperty(InventoryTransaction.PROP_UNIT), InventoryTransaction.PROP_UNIT);
			projectionList.add(Projections.sum(InventoryTransaction.PROP_QUANTITY), InventoryTransaction.PROP_QUANTITY);
			projectionList.add(Projections.groupProperty(InventoryTransaction.PROP_UNIT_COST), InventoryTransaction.PROP_UNIT_COST);
			projectionList.add(Projections.sum(InventoryTransaction.PROP_TOTAL), InventoryTransaction.PROP_TOTAL);

			criteria.createAlias(InventoryTransaction.PROP_MENU_ITEM, "item"); //$NON-NLS-1$
			criteria.add(Restrictions.in("item.id", menuItemIds)); //$NON-NLS-1$
			criteria.add(Restrictions.eq(InventoryTransaction.PROP_TYPE, InventoryTransactionType.IN.getType()));
			criteria.add(Restrictions.or(Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_NEW_STOCK),
					Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_PURCHASE)));
			if (fromDate != null) {
				criteria.add(Restrictions.gt(InventoryTransaction.PROP_TRANSACTION_DATE, fromDate));
			}
			criteria.add(Restrictions.le(InventoryTransaction.PROP_TRANSACTION_DATE, toDate));
			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(InventoryTransaction.class));
			closingAfterTransactions = criteria.list();
		}
		return closingAfterTransactions;
	}

	private Map<String, InventoryTransaction> mergeLastClosingBalanceAndTrans(List<InventoryTransaction> lastClosingTrans,
			List<InventoryTransaction> afterClosingTrans) {

		Map<String, InventoryTransaction> balanceMap = new HashMap<String, InventoryTransaction>();
		if (lastClosingTrans != null && lastClosingTrans.size() > 0) {
			for (Iterator<InventoryTransaction> iterator = lastClosingTrans.iterator(); iterator.hasNext();) {
				InventoryTransaction trans = (InventoryTransaction) iterator.next();
				mergeAndPopulateToMap(balanceMap, trans);
			}
		}
		if (afterClosingTrans != null && afterClosingTrans.size() > 0) {
			for (Iterator<InventoryTransaction> iterator = afterClosingTrans.iterator(); iterator.hasNext();) {
				InventoryTransaction trans = (InventoryTransaction) iterator.next();
				mergeAndPopulateToMap(balanceMap, trans);
			}
		}
		return balanceMap;
	}

	private void mergeAndPopulateToMap(Map<String, InventoryTransaction> transMap, InventoryTransaction trans) {
		InventoryTransaction openingBalance = transMap.get(trans.getMenuItem().getId());
		double openingQuantity = 0;
		double total = 0;
		if (openingBalance == null) {
			openingBalance = trans;
			transMap.put(trans.getMenuItem().getId(), openingBalance);
		}
		else {
			openingQuantity = openingBalance.getQuantity();
			total = openingBalance.getTotal();
		}
		IUnit targetUnit = DataProvider.get().getInventoryUnitById(trans.getUnit());
		Double cost = InventoryUnitConvertionUtil.calculateCost(trans.getUnitCost(), targetUnit, trans.getMenuItem().getUnit(), trans.getMenuItem());
		Double baseUnitQuantity = trans.getTotal() == 0 || trans.getUnitCost() == 0 ? 0D : NumberUtil.round(trans.getTotal() / cost);
		openingBalance.setQuantity(openingQuantity + baseUnitQuantity);
		openingBalance.setUnitCost(cost);
		openingBalance.setTotal(total + trans.getTotal());
	}

	public InventoryTransaction findOpeningBalance(MenuItem menuItem, Date fromdate, Date toDate) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(InventoryTransaction.PROP_MENU_ITEM, menuItem));
			criteria.add(Restrictions.eq(InventoryTransaction.PROP_TYPE, InventoryTransactionType.IN.getType()));
			criteria.add(Restrictions.or(Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_NEW_STOCK),
					Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_PURCHASE)));
			criteria.add(Restrictions.ge(InventoryTransaction.PROP_TRANSACTION_DATE, fromdate));
			criteria.add(Restrictions.le(InventoryTransaction.PROP_TRANSACTION_DATE, toDate));

			List<InventoryTransaction> list = criteria.list();
			InventoryTransaction sum = new InventoryTransaction();
			double openingQty = 0;
			double openingCost = 0;
			double openingTotalCost = 0;
			for (InventoryTransaction inventoryTransaction : list) {
				openingQty += inventoryTransaction.getQuantity();
				openingTotalCost += inventoryTransaction.getTotal();
			}
			openingCost = openingTotalCost / openingQty;
			if (Double.isNaN(openingCost)) {
				openingCost = 0;
			}
			sum.setQuantity(openingQty);
			sum.setTotal(openingTotalCost);
			sum.setUnitCost(openingCost);

			return sum;
		} finally {
			closeSession(session);
		}
	}

	public List<InventoryTransaction> findTransactionsForAvgCosting(MenuGroup menuGroup, Date fromdate, Date toDate) {
		return findTransactionsForAvgCosting(menuGroup, fromdate, toDate, null, null);
	}

	public List<InventoryTransaction> findTransactionsForAvgCosting(MenuGroup menuGroup, Date fromdate, Date toDate, String itemNameOrSKU,
			MenuCategory menuCategory) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			//criteria.createAlias("menuItem.parent", "group");

			criteria.add(Restrictions.eq(InventoryTransaction.PROP_TYPE, InventoryTransactionType.IN.getType()));
			criteria.add(Restrictions.or(Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_NEW_STOCK),
					Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_PURCHASE)));
			criteria.add(Restrictions.ge(InventoryTransaction.PROP_TRANSACTION_DATE, fromdate));
			criteria.add(Restrictions.le(InventoryTransaction.PROP_TRANSACTION_DATE, toDate));
			criteria.createAlias(InventoryTransaction.PROP_MENU_ITEM, "menuItem");
			criteria.add(Restrictions.or(Restrictions.isNull("menuItem." + AppConstants.PROP_DELETED),
					Restrictions.eq("menuItem." + AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (menuGroup != null) {
				criteria.add(Restrictions.eq("menuItem." + MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}
			//			if (menuGroup != null) {
			//				criteria.add(Restrictions.eq("group.id", menuGroup.getId()));
			//			}

			//			criteria.addOrder(Order.asc("menuItem.parent"));

			if (StringUtils.isNotBlank(itemNameOrSKU)) {
				criteria.add(Restrictions.or(Restrictions.ilike("menuItem." + MenuItem.PROP_NAME, itemNameOrSKU.trim(), MatchMode.ANYWHERE),
						Restrictions.ilike("menuItem." + MenuItem.PROP_SKU, itemNameOrSKU.trim(), MatchMode.ANYWHERE)));
			}
			if (menuCategory != null) {
				criteria.add(Restrictions.eq("menuItem." + MenuItem.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}

			criteria.addOrder(Order.asc(InventoryTransaction.PROP_MENU_ITEM));
			criteria.addOrder(Order.asc(InventoryTransaction.PROP_TRANSACTION_DATE));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public int rowCount(String itemName, Object selectedType, Date fromDate, Date toDate) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(InventoryTransaction.class);
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.between(InventoryTransaction.PROP_TRANSACTION_DATE, fromDate, toDate));
			if (StringUtils.isNotEmpty(itemName)) {
				criteria.createAlias(InventoryTransaction.PROP_MENU_ITEM, "item"); //$NON-NLS-1$
				criteria.add(Restrictions.ilike("item.name", "%" + itemName + "%"));
			}
			if (selectedType instanceof InventoryLocation) {
				criteria.add(Restrictions.eq(InventoryTransaction.PROP_FROM_LOCATION_ID, ((InventoryLocation) selectedType).getId()));
				//				criteria.add(Restrictions.eq(InventoryTransaction.PROP_TO_INVENTORY_LOCATION, selectedType));
			}
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		} finally {
			closeSession(session);
		}
	}

	public void getInventoryTransactions(PaginationSupport model, String nameOrBarcodeOrSku, Object selectedLocation, Date fromDate, Date toDate) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.between(InventoryTransaction.PROP_TRANSACTION_DATE, fromDate, toDate));
			criteria.createAlias(InventoryTransaction.PROP_MENU_ITEM, "item"); //$NON-NLS-1$
			String itemAlias = "item."; //$NON-NLS-1$
			if (StringUtils.isNotEmpty(nameOrBarcodeOrSku)) {
				nameOrBarcodeOrSku = nameOrBarcodeOrSku.trim();
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.eq(itemAlias + MenuItem.PROP_BARCODE, nameOrBarcodeOrSku));
				disjunction.add(Restrictions.eq(itemAlias + MenuItem.PROP_SKU, nameOrBarcodeOrSku));
				disjunction.add(Restrictions.ilike(itemAlias + MenuItem.PROP_NAME, nameOrBarcodeOrSku, MatchMode.ANYWHERE));
				criteria.add(disjunction);
			}
			criteria.add(Restrictions.or(Restrictions.isNull(itemAlias + AppConstants.PROP_DELETED),
					Restrictions.eq(itemAlias + AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (selectedLocation instanceof InventoryLocation) {
				Disjunction disjunction = Restrictions.disjunction();
				String locationId = ((InventoryLocation) selectedLocation).getId();
				disjunction.add(Restrictions.eq(InventoryTransaction.PROP_FROM_LOCATION_ID, locationId));
				disjunction.add(Restrictions.eq(InventoryTransaction.PROP_TO_LOCATION_ID, locationId));
				criteria.add(disjunction);
			}
			addDeletedFilter(criteria);
			model.setNumRows(super.rowCount(criteria));
			criteria.addOrder(Order.asc(InventoryTransaction.PROP_TRANSACTION_DATE));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			model.setRows(criteria.list());
		}
	}

	public List<InventoryTransaction> getInventoryTransactionsByItemName(String itemName) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = getSession();
			criteria = session.createCriteria(InventoryTransaction.class);

			if (StringUtils.isNotEmpty(itemName)) {
				criteria.createAlias(InventoryTransaction.PROP_MENU_ITEM, "item"); //$NON-NLS-1$
				criteria.add(Restrictions.ilike("item.name", "%" + itemName + "%")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}

			List<InventoryTransaction> similarItems = criteria.list();
			if (similarItems == null || similarItems.size() == 0) {
				return null;
			}
			return similarItems;

		} catch (Exception e) {
		}
		return criteria.list();
	}

	public Ticket getTicketId(int referenceId) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = getSession();
			criteria = session.createCriteria(Ticket.class);

			criteria.add(Restrictions.eq(Ticket.PROP_ID, referenceId));

			List<Ticket> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return (Ticket) result.get(0);

		} catch (Exception e) {
			PosLog.debug(getClass(), "" + e);
		} finally {
			closeSession(session);
		}
		return null;
	}

	public double findItemAvgCost(MenuItem menuItem) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(InventoryTransaction.PROP_MENU_ITEM, menuItem));
			criteria.add(Restrictions.eq(InventoryTransaction.PROP_TYPE, InventoryTransactionType.IN.getType()));
			criteria.add(Restrictions.or(Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_NEW_STOCK),
					Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_PURCHASE)));

			Calendar calendar = Calendar.getInstance();
			calendar.add(Calendar.MONTH, -1);
			calendar.set(Calendar.DAY_OF_MONTH, 1);
			Date lastMonthStart = DateUtil.startOfDay(calendar.getTime());

			calendar = Calendar.getInstance();
			calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
			Date thisMonthEnd = DateUtil.endOfDay(calendar.getTime());
			criteria.add(Restrictions.ge(InventoryTransaction.PROP_TRANSACTION_DATE, lastMonthStart));
			criteria.add(Restrictions.le(InventoryTransaction.PROP_TRANSACTION_DATE, thisMonthEnd));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(InventoryTransaction.PROP_QUANTITY));
			projectionList.add(Projections.sum(InventoryTransaction.PROP_TOTAL));
			criteria.setProjection(projectionList);

			List list = criteria.list();
			Object[] object = (Object[]) list.get(0);

			double quantity = 0;
			if (object[0] != null) {
				quantity = ((Number) object[0]).doubleValue();
			}
			double total = 0;
			if (object[1] != null) {
				total = ((Number) object[1]).doubleValue();
			}

			if (quantity == 0) {
				return 0;
			}

			return total / quantity;
		} finally {
			closeSession(session);
		}
	}

	public double findItemAvgCost(String menuItemId, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.createAlias(InventoryTransaction.PROP_MENU_ITEM, "item");
		criteria.add(Restrictions.eq("item.id", menuItemId));
		criteria.add(Restrictions.eq(InventoryTransaction.PROP_TYPE, InventoryTransactionType.IN.getType()));
		criteria.add(Restrictions.or(Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_NEW_STOCK),
				Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_PURCHASE)));

		Calendar calendar = Calendar.getInstance();
		calendar.add(Calendar.MONTH, -1);
		calendar.set(Calendar.DAY_OF_MONTH, 1);
		Date lastMonthStart = DateUtil.startOfDay(calendar.getTime());

		calendar = Calendar.getInstance();
		calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
		Date thisMonthEnd = DateUtil.endOfDay(calendar.getTime());
		criteria.add(Restrictions.ge(InventoryTransaction.PROP_TRANSACTION_DATE, lastMonthStart));
		criteria.add(Restrictions.le(InventoryTransaction.PROP_TRANSACTION_DATE, thisMonthEnd));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.sum(InventoryTransaction.PROP_QUANTITY));
		projectionList.add(Projections.sum(InventoryTransaction.PROP_TOTAL));
		criteria.setProjection(projectionList);

		List list = criteria.list();
		Object[] object = (Object[]) list.get(0);

		double quantity = 0;
		if (object[0] != null) {
			quantity = ((Number) object[0]).doubleValue();
		}
		double total = 0;
		if (object[1] != null) {
			total = ((Number) object[1]).doubleValue();
		}

		if (quantity == 0) {
			return 0;
		}

		return total / quantity;
	}

	public List<InventoryTransaction> findTransactions(String reason, InventoryTransactionType transactionType) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (transactionType != null) {
				criteria.add(Restrictions.eq(InventoryTransaction.PROP_TYPE, transactionType.getType()));
			}
			if (reason != null) {
				criteria.add(Restrictions.eq(InventoryTransaction.PROP_REASON, reason));
			}
			criteria.addOrder(Order.asc(InventoryTransaction.PROP_TRANSACTION_DATE));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<InventoryTransaction> findUnsyncedByTicketId(String ticketId) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (ticketId != null) {
				criteria.add(Restrictions.eq(InventoryTransaction.PROP_TICKET_ID, ticketId));

				Criterion nullUpdateTime = Restrictions.isNull(InventoryTransaction.PROP_LAST_UPDATE_TIME);
				Criterion nullSyncTime = Restrictions.isNull(InventoryTransaction.PROP_LAST_SYNC_TIME);
				Criterion gtQuery = Restrictions.gtProperty(InventoryTransaction.PROP_LAST_UPDATE_TIME, InventoryTransaction.PROP_LAST_SYNC_TIME);

				criteria.add(Restrictions.or(nullUpdateTime, nullSyncTime, gtQuery));
			}

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public void doStockTransfer(InventoryTransaction outTrans, InventoryTransaction inTrans) {
		doStockTransfer(outTrans, inTrans, null);
	}

	public void doStockTransfer(InventoryTransaction outTrans, InventoryTransaction inTrans, InventoryStock sourceStock) {
		try (Session session = createNewSession()) {
			InventoryTransactionDAO inventoryTransactionDAO = InventoryTransactionDAO.getInstance();
			Transaction tx = session.beginTransaction();
			inventoryTransactionDAO.adjustInventoryStock(outTrans, session, sourceStock);
			inventoryTransactionDAO.adjustInventoryStock(inTrans, session, sourceStock);
			tx.commit();
		}
	}

	public boolean isPresent(String refNo) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(InventoryTransaction.class);
			criteria.add(Restrictions.eq(InventoryTransaction.PROP_REFERENCE_NO, refNo));
			criteria.setMaxResults(1);
			List list = criteria.list();
			return list != null && !list.isEmpty();
		}
	}

	//	public void saveOrUpdateInventoryTrans(List<InventoryTransaction> dataList) throws Exception {
	//
	//		if (dataList == null)
	//			return;
	//
	//		Transaction tx = null;
	//		Session session = null;
	//		try {
	//			session = createNewSession();
	//			tx = session.beginTransaction();
	//
	//			for (Iterator<InventoryTransaction> iterator = dataList.iterator(); iterator.hasNext();) {
	//				InventoryTransaction item = (InventoryTransaction) iterator.next();
	//				InventoryTransaction existingItem = get(item.getId());
	//				if (existingItem != null) {
	//					long version = existingItem.getVersion();
	//					PropertyUtils.copyProperties(existingItem, item);
	//					existingItem.setVersion(version);
	//					update(existingItem, session);
	//				}
	//				else {
	//					save(item, session);
	//				}
	//			}
	//			tx.commit();
	//		} catch (Exception e) {
	//			tx.rollback();
	//			throw e;
	//		} finally {
	//			closeSession(session);
	//		}
	//
	//	}

	public Criteria buildCriteriaForTransactionReport(Session session, String menuItemNameOrSku, List<Integer> transactionTypes, MenuGroup menuGroup,
			MenuCategory menuCategory) {

		Criteria criteria = session.createCriteria(InventoryTransaction.class);
		criteria.createAlias(InventoryTransaction.PROP_MENU_ITEM, "menuItem"); //$NON-NLS-1$

		criteria.add(Restrictions.or(Restrictions.isNull("menuItem." + AppConstants.PROP_DELETED), //$NON-NLS-1$
				Restrictions.eq("menuItem." + AppConstants.PROP_DELETED, Boolean.FALSE))); //$NON-NLS-1$

		if (transactionTypes != null && !transactionTypes.isEmpty()) {
			criteria.add(Restrictions.in(InventoryTransaction.PROP_TYPE, transactionTypes));
		}

		if (StringUtils.isNotEmpty(menuItemNameOrSku)) {
			Disjunction disjuction = Restrictions.disjunction();
			disjuction.add(Restrictions.eq("menuItem." + MenuItem.PROP_ID, menuItemNameOrSku)); //$NON-NLS-1$
			disjuction.add(Restrictions.ilike("menuItem." + MenuItem.PROP_NAME, menuItemNameOrSku, MatchMode.START)); //$NON-NLS-1$
			disjuction.add(Restrictions.ilike("menuItem." + MenuItem.PROP_SKU, menuItemNameOrSku, MatchMode.EXACT)); //$NON-NLS-1$
			criteria.add(disjuction);
		}

		if (menuGroup != null) {
			criteria.add(Restrictions.eq("menuItem." + MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId())); //$NON-NLS-1$
		}
		if (menuCategory != null) {
			criteria.add(Restrictions.eq("menuItem." + MenuItem.PROP_MENU_CATEGORY_ID, menuCategory.getId())); //$NON-NLS-1$
		}

		return criteria;
	}

}