package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.ModifierMultiplierPrice;
import com.floreantpos.model.Multiplier;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class MultiplierDAO extends BaseMultiplierDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public MultiplierDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof Multiplier) {
			Multiplier multiplier = (Multiplier) obj;
			String hql = "DELETE FROM %s MMP WHERE MMP.%s=:m_id";
			hql = String.format(hql, ModifierMultiplierPrice.REF, ModifierMultiplierPrice.PROP_MULTIPLIER_ID);
			Query query = s.createQuery(hql);
			query.setParameter("m_id", multiplier.getId());
			query.executeUpdate();
			multiplier.setDeleted(Boolean.TRUE);
			super.update(multiplier, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<Multiplier> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(Multiplier.PROP_SORT_ORDER));
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public void saveOrUpdateMultipliers(List<Multiplier> items) {
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<Multiplier> iterator = items.iterator(); iterator.hasNext();) {
				Multiplier multiplier = iterator.next();
				session.saveOrUpdate(multiplier);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			PosLog.error(getClass(), e);
		}
	}

	@SuppressWarnings("unchecked")
	public Multiplier getDefaultMutltiplier() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Multiplier.PROP_DEFAULT_MULTIPLIER, Boolean.TRUE));
			List<Multiplier> list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return list.get(0);
			}
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public boolean nameExists(String name) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Multiplier.PROP_ID, name).ignoreCase());
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount == null) {
				return false;
			}
			return rowCount.intValue() > 0;
		}
	}

	public void deleteMultiplier(Multiplier multiplier) {
		this.delete(multiplier);
	}

	public void saveOrUpdateMultiplierList(List<Multiplier> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<Multiplier> iterator = dataList.iterator(); iterator.hasNext();) {
				Multiplier item = (Multiplier) iterator.next();
				Multiplier existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}

}