/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.SerializationUtils;
import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.base.BaseKitchenTicket;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.model.util.DataProvider;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "parentTicket", "orderType" })
@XmlRootElement()
public class KitchenTicket extends BaseKitchenTicket implements PropertyContainer {
	private static final long serialVersionUID = 1L;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public KitchenTicket() {
	}

	/**
	 * Constructor for primary key
	 */
	public KitchenTicket(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	private String customerName;
	private Printer printer;
	private Ticket parentTicket;
	private Boolean filterItem;
	private Integer sortOrder;
	private transient com.google.gson.JsonObject propertiesContainer;

	@JsonIgnore
	public KitchenStatus getKitchenStatusValue() {
		return KitchenStatus.fromString(super.getStatus());
	}

	public void setKitchenStatusValue(KitchenStatus kitchenStatus) {
		super.setStatus(kitchenStatus.name());
	}

	@JsonIgnore
	public List<KitchenTicketItem> getModifiersForTicketItem(String ticketItemId) {
		List<KitchenTicketItem> modifiers = new ArrayList<>();
		List<KitchenTicketItem> ticketItems = getTicketItems();
		if (ticketItems != null) {
			for (KitchenTicketItem kitchenTicketItem : ticketItems) {
				if (kitchenTicketItem.isModifierItem() && ticketItemId.equals(kitchenTicketItem.getTicketItemId())) {
					modifiers.add(kitchenTicketItem);
				}
			}
		}
		return modifiers;
	}

	@JsonIgnore
	public List<KitchenTicketItem> getCookingInstructionForTicketItem(String ticketItemId) {
		List<KitchenTicketItem> instructions = new ArrayList<>();
		List<KitchenTicketItem> ticketItems = getTicketItems();
		if (ticketItems != null) {
			for (KitchenTicketItem kitchenTicketItem : ticketItems) {
				if (kitchenTicketItem.isCookingInstruction() && ticketItemId.equals(kitchenTicketItem.getTicketItemId())) {
					instructions.add(kitchenTicketItem);
				}
			}
		}
		return instructions;
	}

	public void setPrinter(Printer printer) {
		this.printer = printer;
	}

	@JsonIgnore
	public Printer getPrinter() {
		return this.printer;
	}

	@JsonIgnore
	public List<Printer> getPrinters() {
		List<Printer> printers = new ArrayList<Printer>();

		PosPrinters posPrinters = DataProvider.get().getPrinters();
		PrinterGroup printerGroup = getPrinterGroup();

		if (printerGroup == null) {
			printers.addAll(posPrinters.getKitchenPrinters());
			return printers;
		}

		List<String> printerNames = printerGroup.getPrinterNames();
		List<Printer> kitchenPrinters = posPrinters.getKitchenPrinters();
		for (Printer printer : kitchenPrinters) {
			if (printerNames.contains(printer.getVirtualPrinter().getName())) {
				printers.add(printer);
			}
		}

		return printers;
	}

	@Override
	public List<KitchenTicketItem> getTicketItems() {
		List<KitchenTicketItem> items = super.getTicketItems();

		if (items == null) {
			items = new ArrayList<KitchenTicketItem>();
			super.setTicketItems(items);
		}
		return items;
	}

	public static List<KitchenTicket> fromTicket(Ticket ticket, boolean isFilterKitchenPrintedItems) {
		return fromTicket(ticket, isFilterKitchenPrintedItems, null);
	}

	public static List<KitchenTicket> fromTicket(Ticket ticket, boolean isFilterKitchenPrintedItems, List<TicketItem> ticketItems) {
		return fromTicket(ticket, isFilterKitchenPrintedItems, null, null);
	}

	public static List<KitchenTicket> fromTicket(Ticket ticket, boolean isFilterKitchenPrintedItems, List<TicketItem> ticketItems, Printer printer2) {
		Map<Printer, KitchenTicket> itemMap = new HashMap<Printer, KitchenTicket>();
		List<KitchenTicket> kitchenTickets = new ArrayList<KitchenTicket>(4);

		Ticket clonedTicket = (Ticket) SerializationUtils.clone(ticket);
		boolean isItemBasedKitPrint = (ticketItems == null);//TODO: REVISE CODE. SHOULD WE REMOVE IT?

		if (ticketItems == null) {
			ticketItems = clonedTicket.getTicketItems();
		}

		if (ticketItems == null) {
			return kitchenTickets;
		}
		OrderType orderType = ticket.getOrderType();
		
		for (TicketItem ticketItem : ticketItems) {
			if ((isFilterKitchenPrintedItems && !ticketItem.isKitchenPrintable())) {
				continue;
			}

			List<Printer> printers = ticketItem.getPrinters(orderType);
			if (printers.isEmpty() && printer2 != null) {
				printers.add(printer2);
			}
			List<TicketItem> oldTicketItems = ticket.getTicketItems();
			if (oldTicketItems != null && !oldTicketItems.isEmpty()) {
				for (TicketItem oldTicketItem : oldTicketItems) {
					if (oldTicketItem == null || oldTicketItem.getId() == null || ticketItem.getId() == null) {
						continue;
					}
					if (ticketItem.getId().equals(oldTicketItem.getId())) {
						ticketItem.setProperties(oldTicketItem.getProperties());
					}
				}
			}
			
			boolean isPrintTranslatedName = Boolean.parseBoolean(DataProvider.get().getStore().getProperty(AppConstants.PROP_PRINT_TRANSLATED_NAME));
			for (Printer printer : printers) {
				VirtualPrinter virtualPrinter = printer.getVirtualPrinter();
				KitchenTicket kitchenTicket = itemMap.get(printer);
				if (kitchenTicket == null) {
					kitchenTicket = new KitchenTicket();
					kitchenTicket.setPrinterGroup(ticketItem.getPrinterGroup());
					kitchenTicket.setTicketId(ticket.getId());
					kitchenTicket.setTokenNo(ticket.getTokenNo());
					kitchenTicket.setCreateDate(StoreDAO.getServerTimestamp());
					kitchenTicket.setOrderType(orderType);

					if (ticket.getTableNumbers() != null) {
						kitchenTicket.setTableNumbers(new ArrayList<Integer>(ticket.getTableNumbers()));
					}

					kitchenTicket.setServerName(ticket.getOwner().getFirstName());
					kitchenTicket.setStatus(KitchenStatus.WAITING.name());

					if (StringUtils.isNotEmpty(ticket.getProperty(Ticket.CUSTOMER_NAME))) {
						kitchenTicket.setCustomerName(ticket.getProperty(Ticket.CUSTOMER_NAME));
					}
					kitchenTicket.setPrinter(printer);
					if (virtualPrinter != null) {
						kitchenTicket.setPrinterName(virtualPrinter.getName());
					}
					itemMap.put(printer, kitchenTicket);
				}
				addKitchenItem(ticketItem, kitchenTicket, isPrintTranslatedName);
				includeModifiers(ticketItem, kitchenTicket, isPrintTranslatedName);
				ticketItem.setPrintedToKitchen(true);
				includeCookintInstructions(ticketItem, kitchenTicket);

			}
		}

		Collection<KitchenTicket> values = itemMap.values();

		for (KitchenTicket kitchenTicket : values) {
			if (kitchenTicket.getTicketItems() == null || kitchenTicket.getTicketItems().isEmpty()) {
				continue;
			}
			kitchenTicket.setParentTicket(ticket);
			kitchenTicket.setFilterItem(isFilterKitchenPrintedItems);
			kitchenTickets.add(kitchenTicket);
			String kitchenTicketNumber = ticket.getProperty("KITCHEN_TICKET_NUMBER"); //$NON-NLS-1$
			if (kitchenTicketNumber == null) {
				kitchenTicketNumber = "1"; //$NON-NLS-1$
			}
			else {
				kitchenTicketNumber = String.valueOf(Integer.valueOf(kitchenTicketNumber) + 1);
			}
			ticket.addProperty("KITCHEN_TICKET_NUMBER", kitchenTicketNumber); //$NON-NLS-1$
			kitchenTicket.setSequenceNumber(Integer.valueOf(kitchenTicketNumber));
		}
		if (!isItemBasedKitPrint) {
			ticket.markPrintedToKitchen(ticketItems);
		}
		else {
			ticket.markPrintedToKitchen();
		}
		return kitchenTickets;
	}

	private static void addKitchenItem(TicketItem ticketItem, KitchenTicket kitchenTicket, boolean isPrintTranslatedName) {
		KitchenTicketItem item = new KitchenTicketItem();
		item.setTicketItemId(ticketItem.getId());
		item.setMenuItemCode(ticketItem.getItemCode());
		item.setVoided(ticketItem.isVoided());
		item.setVoidedItemId(ticketItem.getVoidedItemId());

		String prefix = ""; //$NON-NLS-1$
		
		String translatedName = ticketItem.getProperty(AppConstants.TRANSLATED_NAME);
		if (isPrintTranslatedName && translatedName != null)
			item.setMenuItemName(prefix + ticketItem.getNameDisplay(translatedName));
		else
			item.setMenuItemName(prefix + ticketItem.getNameDisplay());
		item.setPrintKitchenSticker(ticketItem.isPrintKitchenSticker());
		if (ticketItem.getMenuItem() == null) {
			item.setMenuItemGroupName("MISC."); //$NON-NLS-1$
			item.setSortOrder(10001);
		}
		else {
			item.setMenuItemGroupName(ticketItem.getGroupName());
			item.setSortOrder(ticketItem.getTableRowNum());
		}
		item.setUnitName(ticketItem.getUnitName());
		item.setQuantity(ticketItem.getQuantity());
		item.setKitchenStatusValue(KitchenStatus.WAITING);
		item.setKitchenTicket(kitchenTicket);
		kitchenTicket.addToticketItems(item);
	}

	private static void includeCookintInstructions(TicketItem ticketItem, KitchenTicket kitchenTicket) {
		List<TicketItemCookingInstruction> cookingInstructions = ticketItem.getCookingInstructions();
		if (cookingInstructions != null) {
			for (TicketItemCookingInstruction ticketItemCookingInstruction : cookingInstructions) {
				KitchenTicketItem item = new KitchenTicketItem();
				item.setCookingInstruction(true);
				item.setTicketItemId(ticketItem.getId());
				item.setMenuItemName(ticketItemCookingInstruction.getNameDisplay());
				if (ticketItem.getMenuItem() == null) {
					item.setMenuItemGroupName("MISC."); //$NON-NLS-1$
					item.setSortOrder(10001);
				}
				else {
					item.setMenuItemGroupName(ticketItem.getGroupName());
				}
				item.setKitchenTicket(kitchenTicket);
				kitchenTicket.addToticketItems(item);
			}
		}
	}

	private static void includeModifiers(TicketItem ticketItem, KitchenTicket kitchenTicket, boolean isPrintTranslatedName) {
		List<TicketItemModifier> ticketItemModifiers = ticketItem.getTicketItemModifiers();
		if (ticketItemModifiers != null) {
			for (TicketItemModifier itemModifier : ticketItemModifiers) {
				if (!itemModifier.isShouldPrintToKitchen()) {
					continue;
				}

				KitchenTicketItem item = new KitchenTicketItem();
				item.setTicketItemId(ticketItem.getId());
				item.setTicketItemModifierId(itemModifier.getId());
				item.setModifierItem(true);

				String translatedName = itemModifier.getProperty(AppConstants.TRANSLATED_NAME);
				if (isPrintTranslatedName && translatedName != null)
					item.setMenuItemName(translatedName);
				else
					item.setMenuItemName(itemModifier.getNameDisplay(true));

				if (ticketItem.getMenuItem() == null) {
					item.setMenuItemGroupName("MISC."); //$NON-NLS-1$
					item.setSortOrder(10001);
				}
				else {
					item.setMenuItemGroupName(ticketItem.getGroupName());
				}
				item.setQuantity(itemModifier.getItemQuantity());
				item.setKitchenStatusValue(KitchenStatus.WAITING);
				item.setKitchenTicket(kitchenTicket);
				kitchenTicket.addToticketItems(item);

				itemModifier.setPrintedToKitchen(true);
			}
		}
	}

	@JsonIgnore
	public String getCustomerName() {
		return customerName;
	}

	public void setCustomerName(String customerName) {
		this.customerName = customerName;
	}

	@XmlTransient
	@JsonIgnore
	public Ticket getParentTicket() {
		if (parentTicket == null) {
			parentTicket = TicketDAO.getInstance().get(getTicketId(), getOutletId());
		}
		return parentTicket;
	}

	public void setParentTicket(Ticket parentTicket) {
		this.parentTicket = parentTicket;
	}

	public Boolean isFilterItem() {
		return filterItem;
	}

	public void setFilterItem(Boolean filterItem) {
		this.filterItem = filterItem;
	}

	public void setOrderType(OrderType orderType) {
		if (orderType != null)
			super.setOrderTypeId(orderType.getId());
		else
			super.setOrderTypeId(null);
	}

	@XmlTransient
	@JsonIgnore
	public OrderType getOrderType() {
		if (StringUtils.isEmpty(getOrderTypeId()))
			return null;
		return DataProvider.get().getOrderType();
	}

	public void setSortOrder(Integer sortOrder) {
		this.sortOrder = sortOrder;
	}

	@JsonIgnore
	public Integer getSortOrder() {
		return sortOrder == null ? 0 : sortOrder;
	}
	
	public void setSubOrderType(SubOrderType subOrderType) {
		if (subOrderType != null) {
			addProperty(Ticket.PROPERTY_SUB_ORDER_TYPE, subOrderType.getName());
		}
	}

	@JsonIgnore
	public SubOrderType getSubOrderType() {
		return SubOrderType.fromName(getProperty(Ticket.PROPERTY_SUB_ORDER_TYPE));
	}
	
	private void initPropertyContainer() {
		if (propertiesContainer == null) {
			if (StringUtils.isBlank(super.getProperties())) {
				propertiesContainer = new com.google.gson.JsonObject();
			}
			else {
				propertiesContainer = new Gson().fromJson(super.getProperties(), com.google.gson.JsonObject.class);
			}
		}
	}
	
	@Override
	public void addProperty(String key, String value) {
		initPropertyContainer();
		propertiesContainer.addProperty(key, value);
		super.setProperties(propertiesContainer.toString());
	}

	@Override
	@JsonIgnore
	public JsonObject getPropertyStore() {
		initPropertyContainer();
		return propertiesContainer;
	}
}