package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Conjunction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.PriceRule;
import com.floreantpos.model.PriceTable;

public class PriceRuleDAO extends BasePriceRuleDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PriceRuleDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session session) {
		validateData((PriceRule) obj, session);
		updateTime(obj);
		return super.save(obj, session);
	}

	@Override
	protected void update(Object obj, Session session) {
		validateData((PriceRule) obj, session);
		updateTime(obj);
		super.update(obj, session);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session session) {
		validateData((PriceRule) obj, session);
		updateTime(obj);
		super.saveOrUpdate(obj, session);
	}

	private void validateData(PriceRule priceRule, Session session) {
		if (priceRuleExists(priceRule, session)) {
			throw new DuplicateDataException(Messages.getString("PriceRuleDAO.1")); //$NON-NLS-1$
		}
	}

	private boolean priceRuleExists(PriceRule priceRule, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		String existingId = priceRule.getId();
		if (StringUtils.isNotEmpty(existingId)) {
			Criterion existingIdExp = Restrictions.ne("id", existingId); //$NON-NLS-1$
			criteria.add(existingIdExp);
		}
		addDeletedFilter(criteria);

		String outletId = priceRule.getOutletId();
		Criterion outletExp = outletId != null ? Restrictions.eq(PriceRule.PROP_OUTLET_ID, outletId) : Restrictions.isNull(PriceRule.PROP_OUTLET_ID);

		String departmentId = priceRule.getDepartmentId();
		Criterion departmentExp = departmentId != null ? Restrictions.eq(PriceRule.PROP_DEPARTMENT_ID, departmentId)
				: Restrictions.isNull(PriceRule.PROP_DEPARTMENT_ID);

		String salesAreaId = priceRule.getSalesAreaId();
		Criterion salesAreaExp = salesAreaId != null ? Restrictions.eq(PriceRule.PROP_SALES_AREA_ID, salesAreaId)
				: Restrictions.isNull(PriceRule.PROP_SALES_AREA_ID);

		String orderTypeId = priceRule.getOrderTypeId();
		Criterion orderTypeExp = orderTypeId != null ? Restrictions.eq(PriceRule.PROP_ORDER_TYPE_ID, orderTypeId)
				: Restrictions.isNull(PriceRule.PROP_ORDER_TYPE_ID);

		String customerGroupId = priceRule.getCustomerGroupId();
		Criterion customerGroupExp = customerGroupId != null ? Restrictions.eq(PriceRule.PROP_CUSTOMER_GROUP_ID, customerGroupId)
				: Restrictions.isNull(PriceRule.PROP_CUSTOMER_GROUP_ID);

		String priceShiftId = priceRule.getPriceShiftId();
		Criterion priceShiftExp = priceShiftId != null ? Restrictions.eq(PriceRule.PROP_PRICE_SHIFT_ID, priceShiftId)
				: Restrictions.isNull(PriceRule.PROP_PRICE_SHIFT_ID);

		String priceTableId = priceRule.getPriceTableId();
		Criterion priceTableExp = priceTableId != null ? Restrictions.eq(PriceRule.PROP_PRICE_TABLE_ID, priceTableId)
				: Restrictions.isNull(PriceRule.PROP_PRICE_TABLE_ID);

		Conjunction priceRuleExp = Restrictions.and(outletExp, departmentExp, salesAreaExp, orderTypeExp, customerGroupExp, priceShiftExp, priceTableExp);
		criteria.add(priceRuleExp);

		List list = criteria.list();
		if (list != null && list.size() > 0) {
			return true;
		}
		return false;
	}

	@Override
	public List<PriceRule> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof PriceRule) {
			PriceRule priceRule = (PriceRule) obj;
			priceRule.setDeleted(Boolean.TRUE);
			super.update(priceRule, s);
		}
		else {
			throw new PosException(Messages.getString("PriceRuleDAO.0")); //$NON-NLS-1$
		}
	}

	public List<PriceRule> findByPriceShiftId(String shiftId, Session session) {
		return this.findByPriceShiftId(shiftId, session, Boolean.FALSE);
	}

	public List<PriceRule> findByPriceShiftId(String shiftId, Session session, boolean addDeletedFilter) {
		if (StringUtils.isBlank(shiftId)) {
			return null;
		}

		Criteria criteria = session.createCriteria(getReferenceClass());
		if (addDeletedFilter) {
			this.addDeletedFilter(criteria);
		}
		criteria.add(Restrictions.eq(PriceRule.PROP_PRICE_SHIFT_ID, shiftId));
		criteria.setProjection(Projections.alias(Projections.property(PriceRule.PROP_NAME), PriceRule.PROP_NAME));
		criteria.setResultTransformer(Transformers.aliasToBean(PriceRule.class));
		return criteria.list();
	}

	@SuppressWarnings("unchecked")
	public List<PriceRule> getPriceRulesByPriceTable(PriceTable priceTable, Session session) {
		if (priceTable == null || priceTable.getId() == null) {
			return null;
		}
		Criteria criteria = session.createCriteria(this.getReferenceClass());
		this.addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(PriceRule.PROP_PRICE_TABLE_ID, priceTable.getId()));
		criteria.setProjection(Projections.alias(Projections.property(PriceRule.PROP_NAME), PriceRule.PROP_NAME));
		return criteria.setResultTransformer(Transformers.aliasToBean(this.getReferenceClass())).list();
	}

	public void saveOrUpdatePriceRuleList(List<PriceRule> dataList, String clientOutletId, boolean updateLastUpdateTime, boolean updateSyncTime)
			throws Exception {
		saveOrUpdatePriceRuleList(dataList, clientOutletId, updateLastUpdateTime, updateSyncTime, false, false);
	}

	public void saveOrUpdatePriceRuleList(List<PriceRule> dataList, String clientOutletId, boolean updateLastUpdateTime, boolean updateSyncTime,
			boolean saveNewDataOnly, boolean forceUpdate) throws Exception {
		if (dataList != null && dataList.size() > 0) {
			for (Iterator<PriceRule> iterator = dataList.iterator(); iterator.hasNext();) {
				PriceRule item = (PriceRule) iterator.next();
				PriceRuleDAO dao = PriceRuleDAO.getInstance();
				PriceRule existingItem = dao.get(item.getId());
				if (existingItem != null) {
					if (!forceUpdate && saveNewDataOnly) {
						PosLog.info(getClass(), item.getName() + " already exists"); //$NON-NLS-1$
						continue;
					}
					final String id = existingItem.getId();
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setId(id);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					dao.update(existingItem);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					dao.save(item);
				}
			}
		}
	}

}