package com.floreantpos.report;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;

import org.apache.commons.lang3.StringUtils;
import org.castor.core.util.Base64Encoder;

import com.floreantpos.PosLog;
import com.floreantpos.model.BloodGroupType;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Department;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.ImageResource;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.ReportType;
import com.floreantpos.model.Specimen;
import com.floreantpos.model.Store;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.ImageResourceDAO;
import com.floreantpos.model.dao.SpecimenDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.UserDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.ImageUtil;
import com.floreantpos.util.POSUtil;

import net.sf.jasperreports.engine.JRBand;
import net.sf.jasperreports.engine.JRElement;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRGroup;
import net.sf.jasperreports.engine.JRSection;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;
import net.sf.jasperreports.engine.type.FooterPositionEnum;
import net.sourceforge.barbecue.Barcode;
import net.sourceforge.barbecue.BarcodeFactory;
import net.sourceforge.barbecue.BarcodeImageHandler;

public class LabTestReportUtil {
	public static JasperPrint createJasperPrint(TicketItem ticketItem) throws JRException {
		return createJasperPrint(ticketItem, null);
	}

	public static JasperPrint createJasperPrint(TicketItem ticketItem, Map<String, Object> mapForPreview) throws JRException {
		LabTestResultReportModel testResultReportModel = createModels(ticketItem);

		HashMap<String, Object> map = new HashMap<String, Object>();
		Outlet outlet = DataProvider.get().getOutlet();
		ReportUtil.populateMedlogicsProperties(map, outlet);
		map.put("reportTitle", "Lab test result report"); //$NON-NLS-1$ 
		JasperReport header = ReportUtil.getReport("medlogics_report_customer_header"); //$NON-NLS-1$
		map.put("head", header); //$NON-NLS-1$
		populateReportParameter(map, Arrays.asList(ticketItem));

		int reportHeaderMargin = outlet.getReportHeaderMargin();
		int reportFooterMargin = outlet.getReportFooterMargin();
		if (mapForPreview != null) {
			map.putAll(mapForPreview);
			reportHeaderMargin = POSUtil.parseInteger((String) mapForPreview.get("headerMarginValue"));
			reportFooterMargin = POSUtil.parseInteger((String) mapForPreview.get("footerMarginValue"));
		}
		JasperReport masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-result-report", reportFooterMargin, reportHeaderMargin); //$NON-NLS-1$

		String reportType = ticketItem.getReportType();
		//		if (ticketItem.getLabWorkStatusValue() != LabWorkStatus.RESULT_RECORDED && ticketItem.getLabWorkStatusValue() != LabWorkStatus.APPROVED) {
		//			MenuItem menuItem = ticketItem.getMenuItem();
		//			if (menuItem != null) {
		//				reportType = menuItem.getReportType();
		//			}
		//		}

		if (!reportType.equals(ReportType.TABULAR_REPORT.name())) {
			masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-result-plain-report", reportFooterMargin, reportHeaderMargin); //$NON-NLS-1$
		}

		//		if (reportType.equals(ReportType.PLAIN_TABULAR_REPORT.name())) {
		//			masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-result-plain-tabular-report", reportFooterMargin, reportHeaderMargin); //$NON-NLS-1$
		//		}

		if (reportType.equals(ReportType.POSITIVE_NEGATIVE.name())) {
			masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-result-positive-negative-report", reportFooterMargin, reportHeaderMargin); //$NON-NLS-1$
		}

		if (reportType.equals(ReportType.PLAIN_REPORT_SIDE_BY_SIDE.name())) {
			masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-result-plain-side-by-side-report", reportFooterMargin, reportHeaderMargin); //$NON-NLS-1$
		}

		JRGroup[] groups = masterReport.getGroups();
		for (JRGroup group : groups) {
			if (group.getFooterPositionValue() == FooterPositionEnum.STACK_AT_BOTTOM) {
				JRSection section = group.getGroupFooterSection();
				JRBand[] bands = section.getBands();
				for (JRBand band : bands) {
					if (!outlet.isShowLabTestTechnicianInReport()) {
						if (band != null) {
							JRElement inchargeFrame = band.getElementByKey("lab_incharge");
							if (inchargeFrame != null) {
								inchargeFrame.setX(0);
							}
						}
					}
				}
			}
		}

		JasperPrint jasperPrint = JasperFillManager.fillReport(masterReport, map, new JRTableModelDataSource(testResultReportModel));
		return jasperPrint;
	}

	public static void populateTestItemParameter(Map<String, Object> contextMap, List<TicketItem> ticketItems) {
		final StringBuilder items = new StringBuilder(""); //$NON-NLS-1$

		for (TicketItem ticketItem : ticketItems) {
			List<TestItem> testItems = ticketItem.getTestItems();
			MenuItem menuItem = ticketItem.getMenuItem();
			if (testItems != null && testItems.size() > 0) {
				testItems.forEach(item -> {
					StringBuilder itemRow = new StringBuilder("<tr>"); //$NON-NLS-1$

					if (!ReportType.PLAIN_REPORT.name().equals(menuItem.getReportType())) {
						itemRow.append("<td style = \"padding-bottom: 5px\" ><b>"); //$NON-NLS-1$
						itemRow.append(item.getName());
						itemRow.append("</b></td>"); //$NON-NLS-1$
						itemRow.append("</tr>"); //$NON-NLS-1$
						itemRow.append("<tr>"); //$NON-NLS-1$
					}
					itemRow.append("<td>"); //$NON-NLS-1$
					itemRow.append(item.getResult());
					itemRow.append("</td>"); //$NON-NLS-1$
					itemRow.append("</tr>"); //$NON-NLS-1$

					items.append(itemRow.toString());
				});
			}
		}
		contextMap.put("testItemsValue", "<table style = 'width: 100%'>" + items.toString() + "</table>");

	}

	public static void populateReportParameter(Map<String, Object> map, List<TicketItem> ticketItems) {
		TicketItem ticketItem = ticketItems.get(0);
		Ticket ticket = ticketItem.getTicket();
		Customer patient = ticket.getCustomer();
		Store store = DataProvider.get().getStore();
		Outlet outlet = DataProvider.get().getOutlet();

		String method = ticketItem.getProperty("method"); //$NON-NLS-1$
		String billName = StringUtils.isBlank(ticketItem.getGroupName()) ? "Laboratory Report" : ticketItem.getGroupName() + " report";

		map.put("visibleReportHeader", outlet.isShowReportPrintHeader());
		map.put("visibleGroupInReport", outlet.isShowGroupInReport());
		map.put("visibleLabTestFooterInReport", outlet.isShowLabTestFooterInReport()); //$NON-NLS-1$
		map.put("visibleLabTechnician", outlet.isShowLabTestTechnicianInReport()); //$NON-NLS-1$
		map.put("visibleLabIncharge", outlet.isShowLabTestInchargeInReport()); //$NON-NLS-1$
		map.put("visibleLabDoctor", outlet.isShowLabDoctorInReport()); //$NON-NLS-1$

		map.put("barcode", ticket.getId()); //$NON-NLS-1$
		map.put("billNo", ticket.getId()); //$NON-NLS-1$

		map.put("instrumentUsed", "Instrument Used: " + method); //$NON-NLS-1$
		map.put("bill", billName); //$NON-NLS-1$
		map.put("methodName", method == null ? "" : method); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("date", DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(ticketItem.getCreateDate()), "dd/MM/yy hh:mm a")); //$NON-NLS-1$
		Date deliveryDate = ticketItem.getDeliveryDate();
		if (deliveryDate != null) {
			map.put("dateReceived", DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(deliveryDate), "dd/MM/yy")); //$NON-NLS-1$
		}
		map.put("labId", store.getUuid()); //$NON-NLS-1$

		map.put("patientId", patient.getId()); //$NON-NLS-1$
		String patientName = patient.getName();
		if (patientName != null) {
			patientName = patientName.trim();
		}
		map.put("ptName", patientName); //$NON-NLS-1$
		map.put("phone", patient.getMobileNo()); //$NON-NLS-1$
		map.put("sex", patient.getGender()); //$NON-NLS-1$
		Date dateOfBirth = patient.getDateOfBirth();
		if (dateOfBirth != null) {
			map.put("age", DateUtil.calculateAge(dateOfBirth)); //$NON-NLS-1$
		}
		String doctorId = ticket.getDoctorId();
		if (StringUtils.isNotBlank(doctorId)) {
			Doctor doctor = DoctorDAO.getInstance().get(doctorId);
			String doctorName = doctor.getName();
			String designation = doctor.getDoctorDesignation();
			if (StringUtils.isNotBlank(designation)) {
				doctorName = doctorName + ", " + "<span style = 'font-weight: normal;'>" + designation + "</span>"; //$NON-NLS-1$
			}
			map.put("consultant", doctorName); //$NON-NLS-1$
		}
		String departmentName = ""; //$NON-NLS-1$
		if (StringUtils.isNotBlank(doctorId)) {
			String departmentId = DoctorDAO.getInstance().get(doctorId).getDoctorDepartmentId();
			Department department = DataProvider.get().getDepartmentById(departmentId);
			if (department != null) {
				departmentName = department.getName();
			}
		}
		map.put("department", departmentName); //$NON-NLS-1$

		boolean isSingleItem = ticketItems.size() == 1;
		map.put("isSingleItem", isSingleItem); //$NON-NLS-1$

		String testName = "";
		String sampleId = "";
		for (Iterator<TicketItem> iterator = ticketItems.iterator(); iterator.hasNext();) {
			TicketItem ticketItem2 = (TicketItem) iterator.next();
			testName += ticketItem2.getName();
			sampleId += ticketItem2.getLabTestId();

			if (iterator.hasNext()) {
				testName += ", ";
				sampleId += ", ";
			}
		}

		map.put("testName", testName); //$NON-NLS-1$
		map.put("sampleId", sampleId); //$NON-NLS-1$

		String bloodGroup = patient.getBloodGroup();
		if (StringUtils.isNotBlank(bloodGroup)) {
			BloodGroupType bloodGroupType = BloodGroupType.fromNameString(bloodGroup);
			if (bloodGroupType != null) {
				bloodGroup = bloodGroupType.getDisplayString();
			}
			else {
				bloodGroup = ""; //$NON-NLS-1$
			}
		}
		map.put("bloodGroup", bloodGroup); //$NON-NLS-1$
		map.put("colCategoryName", ticketItem.getCategoryName()); //$NON-NLS-1$
		map.put("colNameOfTest", "<b>Name of test</b>"); //$NON-NLS-1$ 
		map.put("colResult", "<b>Result</b>"); //$NON-NLS-1$ 
		map.put("colReferenceValue", "<b>Reference value</b>"); //$NON-NLS-1$ 
		map.put("colApproveResultNote", "<b>Comment</b>"); //$NON-NLS-1$ 
		map.put("colUnit", "Unit"); //$NON-NLS-1$ 
		User owner = ticket.getOwner();
		String serverName = "";
		if (owner != null) {
			serverName = owner.getFullName();
		}
		else {
			serverName += ticket.getProperty("owner.firstname", "") + " " + ticket.getProperty("owner.lastname", ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		}

		map.put("colPrintedBy", //$NON-NLS-1$
				"Printed: " + DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd/MM/yy hh:mm a") + " " //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
						+ "by" + " " + serverName); //$NON-NLS-2$

		String note = ticketItem.getReportDescription();
		map.put("colNote", "Comment:"); //$NON-NLS-1$ //$NON-NLS-2$ 
		if (StringUtils.isNotBlank(note)) {
			map.put("note", skipInlineFont(outlet, note)); //$NON-NLS-1$
		}

		String headerNote = ticketItem.getHeaderNote();
		map.put("headerNote", headerNote == null ? StringUtils.EMPTY : skipInlineFont(outlet, headerNote)); //$NON-NLS-1$

		String specimenName = ""; //$NON-NLS-1$
		String specimenId = ticketItem.getSpecimenId();
		if (StringUtils.isNotBlank(specimenId)) {
			Specimen specimen = SpecimenDAO.getInstance().get(specimenId);
			if (specimen != null) {
				specimenName = specimen.getName() != null ? specimen.getName().trim() : specimen.getName();
			}
		}
		map.put("sampleType", specimenName); //$NON-NLS-1$

		boolean hasSignature = false;
		String labDoctorId = ticketItem.getLabDoctorId();
		String labDoctorName = "";
		String doctorDesignation = "";
		String doctorSignature = null;
		if (StringUtils.isNotBlank(labDoctorId)) {
			Doctor labDoctor = DoctorDAO.getInstance().get(labDoctorId);
			if (labDoctor != null) {
				labDoctorName = labDoctor.getName();
				doctorDesignation = labDoctor.getDoctorDesignation();
				if (doctorDesignation != null) {
					doctorDesignation = doctorDesignation.trim();
				}
				else {
					labDoctorName = "";
					doctorDesignation = "";
				}
				doctorSignature = getDoctorSignature(labDoctor);
			}
		}
		if (doctorSignature != null) {
			hasSignature = true;
			map.put("labDoctorSignature", doctorSignature); //$NON-NLS-1$
		}
		map.put("hasLabDoctorSignature", StringUtils.isNotBlank(doctorSignature)); //$NON-NLS-1$
		map.put("colLabDoctorName", labDoctorName);
		map.put("colLabDoctorDesignation", doctorDesignation); //$NON-NLS-1$
		//		map.put("colLabDoctorDesignation", POSUtil.simplifySentence(doctorDesignation, 1)); //$NON-NLS-1$

		String labInchargeId = ticketItem.getLabInchargeId();
		String labInchargeName = "";
		String inchargeDesignation = "";
		String inchargeSignature = null;
		if (StringUtils.isNotBlank(labInchargeId)) {
			User labInchargeUser = UserDAO.getInstance().get(labInchargeId, DataProvider.get().getCurrentOutletId());
			if (labInchargeUser != null) {
				labInchargeName = labInchargeUser.getFullName(); //$NON-NLS-1$
				inchargeDesignation = labInchargeUser.getDesignation();
				if (inchargeDesignation != null) {
					inchargeDesignation = inchargeDesignation.trim();
				}
				else {
					labInchargeName = "";
					inchargeDesignation = "";
				}
				inchargeSignature = getSignature(labInchargeUser);
			}
		}
		if (inchargeSignature != null) {
			hasSignature = true;
			map.put("labInchargeSignature", inchargeSignature); //$NON-NLS-1$
		}
		map.put("hasLabInchargeSignature", StringUtils.isNotBlank(inchargeSignature)); //$NON-NLS-1$
		map.put("colLabInchargeName", labInchargeName);
		map.put("colLabInchargeDesignation", inchargeDesignation); //$NON-NLS-1$

		String labTechnicianId = ticketItem.getLabTechnicianId();
		String technicianName = "";
		String technicianDesignation = "";
		String technicianSignature = null;
		if (StringUtils.isNotBlank(labTechnicianId)) {
			User labTechnicianUser = UserDAO.getInstance().get(labTechnicianId, DataProvider.get().getCurrentOutletId());
			if (labTechnicianUser != null) {
				technicianName = labTechnicianUser.getFullName(); //$NON-NLS-1$
				technicianDesignation = labTechnicianUser.getDesignation();
				if (technicianDesignation != null) {
					technicianDesignation = technicianDesignation.trim();
				}
				else {
					technicianName = "";
					technicianDesignation = "";
				}
				technicianSignature = getSignature(labTechnicianUser);
			}
		}
		if (technicianSignature != null) {
			hasSignature = true;
			map.put("labTechnicianSignature", technicianSignature); //$NON-NLS-1$
		}
		map.put("hasLabTechnicianSignature", StringUtils.isNotBlank(technicianSignature)); //$NON-NLS-1$
		map.put("colLabTechnician", technicianName);
		map.put("colLabTechnicianDesignation", technicianDesignation); //$NON-NLS-1$
		map.put("hasSignature", hasSignature);

		map.put("reportDescription", ticketItem.getReportDescription()); //$NON-NLS-1$

		map.put("isVerifiedLabDoctorSignature", ticketItem.isLabDoctorSignatureVerified()); //$NON-NLS-1$
		map.put("isVerifiedLabTechnicianSignature", ticketItem.isLabTechnicianSignatureVerified()); //$NON-NLS-1$
		map.put("isVerifiedLabInchargeSignature", ticketItem.isLabInchargeSignatureVerified()); //$NON-NLS-1$

		if (outlet.isShowPrintReportStoreLogo()) {
			ImageIcon storeLogo = outlet.getStoreLogo();
			if (storeLogo != null) {
				ImageIcon imageIcon = ImageUtil.pngToJpg(storeLogo);
				if (imageIcon != null) {
					//for back office view
					map.put("storeLogo", imageIcon.getImage()); //$NON-NLS-1$
				}
			}
			ImageResource logoImage = outlet.getLogoImageResource();
			if (logoImage != null) {
				//for html report view
				map.put("logo", new String(Base64Encoder.encode(logoImage.getImageBytes()))); //$NON-NLS-1$
			}
		}

		try {
			Barcode barcode = BarcodeFactory.createCode128(ticket.getId());
			BufferedImage bufferedImage = BarcodeImageHandler.getImage(barcode);
			ByteArrayOutputStream bos = new ByteArrayOutputStream();
			ImageIO.write(bufferedImage, "PNG", bos);
			byte[] data = bos.toByteArray();
			map.put("barcode", new String(Base64Encoder.encode(data)));
		} catch (Exception e) {
			PosLog.error(LabTestReportUtil.class, e);
		}
		if (outlet.isShowPadImageInReport()) {
			String imageId = outlet.getPreviewImageId();
			if (StringUtils.isNotBlank(imageId)) {
				map.put("imagePath", new String(Base64Encoder.encode(DataProvider.get().getImageResource(imageId).getImageBytes()))); //$NON-NLS-1$);
			}
		}
	}

	public static String skipInlineFont(Outlet outlet, String htmlString) {
		if (outlet.isBengaliFontOnLabReport()) {
			return "<div>" + htmlString.replaceAll("&nbsp;", " ").replaceAll("font-family", "family") + "</div>"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		return htmlString;
	}

	private static String getSignature(User labTechnicianUser) {
		try {
			if (labTechnicianUser.isAllowDigitalSignature()) {
				String signatureImageId = labTechnicianUser.getSignatureImageId();
				if (StringUtils.isNotBlank(signatureImageId)) {
					ImageResource imageResource = ImageResourceDAO.getInstance().findById(signatureImageId);
					if (imageResource != null) {
						return Base64.getEncoder().encodeToString(imageResource.getImageBytes());
					}
				}
			}
		} catch (Exception e) {
			PosLog.error(LabTestReportUtil.class, e);
		}
		return null;
	}

	private static String getDoctorSignature(Doctor doctor) {
		try {
			if (doctor.isAllowDigitalSignature()) {
				String signatureImageId = doctor.getSignatureImageId();
				if (StringUtils.isNotBlank(signatureImageId)) {
					ImageResource imageResource = ImageResourceDAO.getInstance().findById(signatureImageId);
					if (imageResource != null) {
						return Base64.getEncoder().encodeToString(imageResource.getImageBytes());
					}
				}
			}
		} catch (Exception e) {
			PosLog.error(LabTestReportUtil.class, e);
		}
		return null;
	}

	public static LabTestResultReportModel createModels(TicketItem ticketItem) {
		return createModels(Arrays.asList(ticketItem), false);
	}

	public static LabTestResultReportModel createModels(List<TicketItem> ticketItems, boolean isThreeColumn) {
		List<TestItem> testItems = new ArrayList();
		LabTestResultReportModel testResultReportModel = new LabTestResultReportModel(ticketItems.get(0));

		for (TicketItem ticketItem : ticketItems) {
			List<TestItem> tItems = ticketItem.getTestItems();
			if (tItems != null) {
				for (TestItem item : tItems) {
					if (isThreeColumn && !item.isShowTestItem()) {
						continue;
					}
					String normalValue = item.getNormalValue();
					if (normalValue != null) {
						normalValue = normalValue.replaceAll("\n", "<br/>"); //$NON-NLS-1$ //$NON-NLS-2$
						normalValue = normalValue.replaceAll("µ", "µ &nbsp;"); //$NON-NLS-1$//$NON-NLS-2$
						item.setNormalValue(normalValue);
					}
					testItems.add(item);
				}
			}
		}

		testResultReportModel.setRows(testItems);
		return testResultReportModel;
	}
}
