package com.floreantpos.report;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.math.BigDecimal;
import java.net.URL;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.table.TableModel;

import org.apache.commons.lang.WordUtils;
import org.apache.commons.lang3.StringUtils;
import org.castor.core.util.Base64Encoder;

import com.floreantpos.POSConstants;
import com.floreantpos.PosLog;
import com.floreantpos.model.Address;
import com.floreantpos.model.BloodGroupType;
import com.floreantpos.model.Challan;
import com.floreantpos.model.Currency;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Department;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.ImageResource;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.CurrencyDAO;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.ext.PaperSize;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.ListTableModel;
import com.floreantpos.util.ImageUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;
import net.sourceforge.barbecue.Barcode;
import net.sourceforge.barbecue.BarcodeFactory;
import net.sourceforge.barbecue.BarcodeImageHandler;
import pl.allegro.finance.tradukisto.MoneyConverters;

public class SalesInvoiceUtil {

	public static JasperPrint createJasperPrint(Ticket ticket, boolean isPharma) throws JRException {
		return createJasperPrint(ticket, null, null, isPharma);
	}

	public static JasperPrint createJasperPrint(Challan challan, Ticket ticket, String copyFor, String copyText) throws JRException {
		return createJasperPrint(challan, ticket, null, copyFor, copyText, false);
	}

	public static JasperPrint createJasperPrint(Ticket ticket, String copyFor) throws JRException {
		return createJasperPrint(ticket, null, copyFor);
	}

	public static JasperPrint createJasperPrint(Ticket ticket, Map<String, Object> mapForPreview) throws JRException {
		return createJasperPrint(ticket, mapForPreview, null);
	}

	public static JasperPrint createJasperPrint(Ticket ticket, Map<String, Object> mapForPreview, String copyFor) throws JRException {
		return createJasperPrint(ticket, mapForPreview, copyFor, false);
	}

	public static JasperPrint createJasperPrint(Ticket ticket, Map<String, Object> mapForPreview, String copyFor, boolean isPharma) throws JRException {
		return createJasperPrint(null, ticket, mapForPreview, copyFor, null, false);
	}

	public static JasperPrint createJasperPrint(Challan challan, Ticket ticket, Map<String, Object> mapForPreview, String copyFor, String copyText,
			boolean isPharma) throws JRException {

		HashMap<String, Object> map = new HashMap<String, Object>();
		Outlet outlet = DataProvider.get().getOutlet();

		TableModel reportModel = null;
		if (challan != null) {
			reportModel = new ChallanReportModel(challan);
			map.put("challanNo", challan.getChallanNo()); //$NON-NLS-1$
			map.put("copyText", copyText);
			map.put("note", challan.getNote());
		}
		else {
			reportModel = createModels(ticket, copyFor, outlet.isEnableJasperPrint());
		}

		PaperSize paperSize = getPageSize(mapForPreview, isPharma, outlet);
		ReportUtil.populateMedlogicsProperties(map, outlet, paperSize);

		JasperReport header;
		if (challan == null) {
			if (paperSize == PaperSize.A5) {
				header = ReportUtil.getReport("byzlogics_report_customer_header_order_info_A5"); //$NON-NLS-1$	
			}
			else {
				header = ReportUtil.getReport("byzlogics_report_customer_header_order_info"); //$NON-NLS-1$	
			}
		}
		else {
			if (paperSize == PaperSize.A5) {
				header = ReportUtil.getReport("byzlogics_challan_report_customer_header_A5"); //$NON-NLS-1$	
			}
			else {
				header = ReportUtil.getReport("byzlogics_challan_report_customer_header"); //$NON-NLS-1$	
			}
		}
		map.put("head", header); //$NON-NLS-1$
		map.put("reportTitle", "Lab test report"); //$NON-NLS-1$
		map.put("isVisibleLabID", !isPharma); //$NON-NLS-1$
		map.put("unpaidOrder", //$NON-NLS-1$
				paperSize == PaperSize.A5 ? ReportUtil.getReport("byzlogics_upaid_ticket_order_A5") : ReportUtil.getReport("byzlogics_upaid_ticket_order")); //$NON-NLS-1$ //$NON-NLS-2$
		populateReportParameter(map, ticket, copyFor, isPharma);

		int receiptReportHeaderMargin = outlet.getReceiptReportHeaderMargin();
		int receiptReportFooterMargin = outlet.getReceiptReportFooterMargin();
		JasperReport masterReport = null;
		if (mapForPreview != null) {
			map.putAll(mapForPreview);
			receiptReportHeaderMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptHeaderMarginValue")); //$NON-NLS-1$
			receiptReportFooterMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptFooterMarginValue")); //$NON-NLS-1$
		}

		if (paperSize == PaperSize.A5) {
			if (StringUtils.isNoneBlank(copyFor)) {
				if (copyFor.contains("Challan") || copyFor.contains("Service Order")) {
					masterReport = ReportUtil.adjustReportHeaderAndFooter("byzlogics-Chalan-report_A5", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
				}
				else {
					masterReport = ReportUtil.adjustReportHeaderAndFooter("byzlogics-invoice-report_A5", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
				}
			}
			else {
				masterReport = ReportUtil.adjustReportHeaderAndFooter("byzlogics-invoice-report_A5", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
			}
		}
		else {
			if (StringUtils.isNoneBlank(copyFor)) {
				if (copyFor.contains("Challan") || copyFor.contains("Service Order")) {
					masterReport = ReportUtil.adjustReportHeaderAndFooter("byzlogics-Chalan-report", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
				}
				else {
					masterReport = ReportUtil.adjustReportHeaderAndFooter("byzlogics-invoice-report", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
				}
			}
			else {
				masterReport = ReportUtil.adjustReportHeaderAndFooter("byzlogics-invoice-report", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
			}
		}

		JasperPrint jasperPrint = JasperFillManager.fillReport(masterReport, map, new JRTableModelDataSource(reportModel));
		return jasperPrint;
	}

	public static HashMap<String, Object> createHtmlPrintMap(Challan challan, Ticket ticket, String copyFor, String copyText) {
		return createHtmlPrintMap(challan, ticket, null, copyFor, copyText, false);
	}

	public static HashMap<String, Object> createHtmlPrintMap(Ticket ticket, Map<String, Object> mapForPreview) {
		return createHtmlPrintMap(ticket, mapForPreview, null);
	}

	public static HashMap<String, Object> createHtmlPrintMap(Ticket ticket, Map<String, Object> mapForPreview, String copyFor) {
		return createHtmlPrintMap(ticket, mapForPreview, copyFor, false);
	}

	public static HashMap<String, Object> createHtmlPrintMap(Ticket ticket, Map<String, Object> mapForPreview, String copyFor, boolean isPharma) {
		return createHtmlPrintMap(null, ticket, mapForPreview, copyFor, null, false);
	}

	public static HashMap<String, Object> createHtmlPrintMap(Challan challan, Ticket ticket, Map<String, Object> mapForPreview, String copyFor, String copyText,
			boolean isPharma) {
		return createHtmlPrintMap(challan, ticket, mapForPreview, copyFor, copyText, isPharma, false);

	}

	public static HashMap<String, Object> createHtmlPrintMap(Challan challan, Ticket ticket, Map<String, Object> mapForPreview, String copyFor, String copyText,
			boolean isPharma, boolean jasper) {

		HashMap<String, Object> map = new HashMap<String, Object>();
		Outlet outlet = DataProvider.get().getOutlet();

		if (challan != null) {
			ChallanReportModel reportModel = new ChallanReportModel(challan);
			reportModel.setJasperModel(outlet.isEnableJasperPrint());
			map.put("challanNo", challan.getChallanNo()); //$NON-NLS-1$
			map.put("copyText", copyText); //$NON-NLS-1$
			map.put("note", challan.getNote()); //$NON-NLS-1$
			map.put("reportModel", reportModel); //$NON-NLS-1$
			map.put("orderItems", reportModel.getRows()); //$NON-NLS-1$
		}
		else {
			LabTestReportModel reportModel = createModels(ticket, copyFor, outlet.isEnableJasperPrint());
			map.put("reportModel", reportModel); //$NON-NLS-1$
			map.put("orderItems", reportModel.getRows()); //$NON-NLS-1$
		}
		map.put("StringUtils", StringUtils.class); //$NON-NLS-1$
		map.put("null", null); //$NON-NLS-1$

		PaperSize paperSize = getPageSize(mapForPreview, isPharma, outlet);

		ReportUtil.populateMedlogicsProperties(map, outlet, paperSize);
		loadHeaderLogoAndBarcodeForHtml(ticket, outlet, StoreDAO.getRestaurant(), map);

		boolean isPaperSizeA5 = paperSize == PaperSize.A5;
		map.put("paperSize", paperSize == null ? "A4" : paperSize.toString()); //$NON-NLS-1$ //$NON-NLS-2$	
		map.put("challan", challan != null); //$NON-NLS-1$	

		map.put("firstColValueMinWidth", isPaperSizeA5 ? "120px" : "190px"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$	
		map.put("firstColValueMaxWidth", isPaperSizeA5 ? "120px" : "190px"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$	
		map.put("midColValueMinWidth", isPaperSizeA5 && challan != null ? "70px" : "140px"); //$NON-NLS-1$ //$NON-NLS-2$ 
		map.put("midColValueMaxWidth", isPaperSizeA5 && challan != null ? "70px" : "140px"); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("lastColValueMinWidth", isPaperSizeA5 ? "130px" : "160px"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$	
		map.put("lastColValueMaxWidth", isPaperSizeA5 ? "130px" : "160px"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$	

		map.put("reportTitle", "Lab test report"); //$NON-NLS-1$
		map.put("isVisibleLabID", !isPharma); //$NON-NLS-1$
		populateReportParameter(map, ticket, copyFor, isPharma);

		int receiptReportHeaderMargin = outlet.getReceiptReportHeaderMargin();
		int receiptReportFooterMargin = outlet.getReceiptReportFooterMargin();
		if (mapForPreview != null) {
			map.putAll(mapForPreview);
			receiptReportHeaderMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptHeaderMarginValue")); //$NON-NLS-1$
			receiptReportFooterMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptFooterMarginValue")); //$NON-NLS-1$
		}
		if (receiptReportFooterMargin < 20) {
			receiptReportFooterMargin = 20;
		}
		map.put("receiptReportHeaderMargin", receiptReportHeaderMargin + "px");
		map.put("receiptReportFooterMargin", receiptReportFooterMargin + "px");

		map.put("receiptReportLeftMargin", isPaperSizeA5 ? "18px" : "60px");
		map.put("receiptReportRightMargin", isPaperSizeA5 ? "18px" : "50px");

		User owner = ticket.getOwner();
		String serverName = "";
		if (owner != null) {
			serverName = owner.getFullName();
		}
		else {
			serverName += ticket.getProperty("owner.firstname", "") + " " + ticket.getProperty("owner.lastname", ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		}
		map.put("colPrintedBy", //$NON-NLS-1$
				"Printed: " + DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd/MM/yy hh:mm a") + " " //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
						+ "by" + " " + serverName); //$NON-NLS-2$

		return map;
	}

	public static PaperSize getPageSize(Map<String, Object> mapForPreview, boolean isPharma, Outlet outlet) {
		PaperSize paperSize;
		if (mapForPreview == null) {
			PaperSize receiptPaperSize;
			if (isPharma) {
				receiptPaperSize = outlet.getPharmacyReceiptPaperSize();
			}
			else {
				receiptPaperSize = outlet.getReceiptPaperSize();
			}
			paperSize = receiptPaperSize;
		}
		else {
			paperSize = (PaperSize) mapForPreview.get("paperSize"); //$NON-NLS-1$
		}
		return paperSize;
	}

	private static void loadHeaderLogoAndBarcodeForHtml(Ticket ticket, Outlet outlet, Store store, Map<String, Object> map) {
		try {
			Barcode barcode = BarcodeFactory.createCode128(ticket.getId());
			BufferedImage bufferedImage = BarcodeImageHandler.getImage(barcode);
			ByteArrayOutputStream bos = new ByteArrayOutputStream();
			ImageIO.write(bufferedImage, "PNG", bos);
			byte[] data = bos.toByteArray();
			map.put("barcodeImage", new String(Base64Encoder.encode(data)));
		} catch (Exception e) {
			PosLog.error(LabTestReportUtil.class, e);
		}

		if (outlet.isShowPrintReceiptStoreLogo()) {
			ImageResource logoImage = outlet.getLogoImageResource();
			if (logoImage != null) {
				map.put("headerLogo", new String(Base64Encoder.encode(logoImage.getImageBytes()))); //$NON-NLS-1$
			}
		}
		//		String padImageId = outlet.getImagePreview();
		//		if (StringUtils.isNotBlank(padImageId)) {
		//			map.put("padImage", new String(Base64Encoder.encode(DataProvider.get().getImageResource(padImageId).getImageBytes()))); //$NON-NLS-1$
		//		}
		//		map.put("isShowPadImage", StringUtils.isNotBlank(padImageId)); //$NON-NLS-1$
	}

	public static String getMoneyIntoWords(String input) {
		MoneyConverters converter = MoneyConverters.ENGLISH_BANKING_MONEY_VALUE;
		return WordUtils.capitalize(converter.asWords(new BigDecimal(input)));
	}

	private static void populateReportParameter(HashMap<String, Object> map, Ticket ticket, String copyFor, boolean isPharma) {
		Customer patient = ticket.getCustomer();
		Store store = DataProvider.get().getStore();
		String departmentName = ""; //$NON-NLS-1$
		String doctorId = ticket.getDoctorId();
		if (StringUtils.isNotBlank(doctorId)) {
			String departmentId = DoctorDAO.getInstance().get(doctorId).getDoctorDepartmentId();
			Department department = DataProvider.get().getDepartmentById(departmentId);
			if (department != null) {
				departmentName = department.getName();
			}

		}

		String refName = "";
		String referrerId = ticket.getReferrerId();
		if (StringUtils.isNotBlank(referrerId)) {
			Customer referrer = CustomerDAO.getInstance().get(referrerId);
			if (referrer != null) {
				refName = referrer.getName();
			}
		}
		Outlet outlet = DataProvider.get().getOutlet();
		map.put("visibleReportHeader", outlet.isShowReceiptPrintHeader()); //$NON-NLS-1$
		map.put("visibleLabTestByCategory", outlet.isShowGroupLabTestByCategory()); //$NON-NLS-1$
		map.put("visibleLabTestPathologyRoomNo", outlet.isShowPathologyTestRoomNo()); //$NON-NLS-1$
		map.put("groupProductsByCategory", outlet.isGroupProductsByCategory()); //$NON-NLS-1$
		if (StringUtils.isNoneBlank(copyFor)) {
			if (copyFor.contains("Challan") || copyFor.contains("Service Order")) { //$NON-NLS-1$
				if (map.get("bill") == null) { //$NON-NLS-1$
					map.put("bill", copyFor); //$NON-NLS-1$
				}
				map.put("visibleLabTestByCategory", false); //$NON-NLS-1$
				map.put("visbileProductType", false); //$NON-NLS-1$
			}
			else {
				map.put("visbileProductType", true); //$NON-NLS-1$
				map.put("bill", ticket.isVoided() ? "Voided " + "Bill (" + copyFor.trim() + ")" : "Bill (" + copyFor.trim() + ")"); //$NON-NLS-1$
			}
		}
		else {
			map.put("bill", ticket.isVoided() ? "Voided " + "Bill" : "Bill"); //$NON-NLS-1$
			map.put("visbileProductType", true); //$NON-NLS-1$
		}
		map.put("barcode", ticket.getId()); //$NON-NLS-1$
		map.put("billNo", ticket.getId()); //$NON-NLS-1$
		map.put("date", DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(ticket.getCreateDate()), "dd MMM yy hh:mm a")); //$NON-NLS-1$
		//		map.put("dateReceived", "Delivery date: " + DateUtil.simplifyDateAndTime(ticket.getDeliveryDate(), "dd MMM yy hh:mm a")); //$NON-NLS-1$
		map.put("projectName", ticket.getProjectNameDisplay()); //$NON-NLS-1$
		map.put("labId", store.getUuid()); //$NON-NLS-1$

		boolean isQrCodeVisible = false;
		if (patient != null) {
			map.put("patientId", patient.getId()); //$NON-NLS-1$
			map.put("ptName", patient.getName()); //$NON-NLS-1$
			map.put("customerAddress", patient.getAddress()); //$NON-NLS-1$
			map.put("phone", patient.getMobileNo()); //$NON-NLS-1$
			map.put("sex", patient.getGender()); //$NON-NLS-1$
			Date dateOfBirth = patient.getDateOfBirth();
			if (dateOfBirth != null) {
				map.put("age", DateUtil.calculateAge(dateOfBirth)); //$NON-NLS-1$
			}

			String bloodGroup = patient.getBloodGroup();
			if (StringUtils.isNotBlank(bloodGroup)) {
				BloodGroupType bloodGroupType = BloodGroupType.fromNameString(bloodGroup);
				if (bloodGroupType != null) {
					bloodGroup = bloodGroupType.getDisplayString();
				}
				else {
					bloodGroup = "";
				}
			}
			map.put("bloodGroup", bloodGroup); //$NON-NLS-1$

			isQrCodeVisible = true;
		}

		String customerLoginURL = store.getCustomerLoginURL();
		if (StringUtils.isNotEmpty(customerLoginURL)) {
			if (StringUtils.isNotBlank(store.getStoreBaseLink()) && !customerLoginURL.startsWith(store.getStoreBaseLink())) {
				try {
					URL url = new URL(customerLoginURL);
					URL baseUrl = new URL(store.getStoreBaseLink());
					customerLoginURL = new URL(baseUrl.getProtocol(), baseUrl.getHost(), baseUrl.getPort(), url.getPath()).toExternalForm();
				} catch (Exception ignore) {
				}
			}
			map.put("qrCode", customerLoginURL + "/" + ticket.getCustomerId()); //$NON-NLS-1$
			map.put("qrCodeBottomMsg", "Scan the QR code to access the report");
		}
		else {
			isQrCodeVisible = false;
		}

		map.put("isQrCodeVisible", isQrCodeVisible);

		if (StringUtils.isNotBlank(doctorId)) {
			Doctor doctor = DoctorDAO.getInstance().get(doctorId);
			String doctorName = doctor.getName();
			String designation = doctor.getDoctorDesignation();
			if (StringUtils.isNotBlank(designation)) {
				doctorName = doctorName + " " + designation; //$NON-NLS-1$
			}
			map.put("consultant", doctorName); //$NON-NLS-1$
		}

		map.put("department", departmentName); //$NON-NLS-1$
		if (StringUtils.isNotBlank(refName)) {
			map.put("refBy", "Ref by: " + refName); //$NON-NLS-1$ //$NON-NLS-2$
		}

		Currency mainCurrency = CurrencyDAO.getInstance().getMainCurrencyByOutletId((ticket.getOutletId()));
		map.put("colNameOfTest", "<b>Item</b>"); //$NON-NLS-1$
		map.put("colQty", "<b>Quantity</b>"); //$NON-NLS-1$
		map.put("colUnit", "<b>Unit</b>"); //$NON-NLS-1$
		String currencySymbol = "<span style='font-family: Nirmala UI;'>" + mainCurrency.getSymbol() + "</span>";

		map.put("colAmount", "Amount" + " (" + currencySymbol + ")"); //$NON-NLS-1$

		map.put("grandTotal", "Grand total"); //$NON-NLS-1$
		map.put("colBillAmount", "Bill amount:"); //$NON-NLS-1$

		map.put("orderDiscount", "Discount on order:"); //$NON-NLS-1$
		map.put("orderDiscountAmount", currencySymbol + "" + NumberUtil.formatAmount(ticket.getDiscountAmount())); //$NON-NLS-1$
		map.put("billAmount", //$NON-NLS-1$
				currencySymbol + "" + NumberUtil.formatNumber(ticket.getSubtotalAmountWithoutItemDiscount())); //$NON-NLS-1$

		map.put("colTax", "Tax:"); //$NON-NLS-1$
		if (ticket.isTaxIncluded()) {
			map.put("tax", "Included"); //$NON-NLS-1$
		}
		else {
			map.put("tax", currencySymbol + "" + NumberUtil.formatAmount(ticket.getTaxAmount())); //$NON-NLS-1$
		}

		Double totalAmount = ticket.getTotalAmount();
		Double dueAmount = ticket.getDueAmount();

		if (outlet.isAllowConsolidatedPayment()) {
			List<Ticket> unpaidCustomerTickets = TicketDAO.getInstance().findUnpaidTicketsForCustomerAppendSource(ticket, false);

			map.put("isVisibleUpaidOrders", unpaidCustomerTickets != null && !unpaidCustomerTickets.isEmpty()); //$NON-NLS-1$

			if (unpaidCustomerTickets != null && !unpaidCustomerTickets.isEmpty()) {
				for (Ticket ticket2 : unpaidCustomerTickets) {
					totalAmount += ticket2.getTotalAmount();
					dueAmount += ticket2.getDueAmount();
				}
				UnpaidTicketModel unpaidDataSource = new SalesInvoiceUtil().new UnpaidTicketModel();
				unpaidDataSource.setRows(unpaidCustomerTickets);
				map.put("unpaidDataSource", new JRTableModelDataSource(unpaidDataSource)); //$NON-NLS-1$
			}

		}

		map.put("colSpecialDiscount", "Discount:"); //$NON-NLS-1$
		map.put("specialDiscount", currencySymbol + "" + NumberUtil.formatAmount(ticket.getDiscountAmount())); //$NON-NLS-1$

		map.put("colTotalAmount", "Total Amount:"); //$NON-NLS-1$
		map.put("totalAmount", currencySymbol + "" + NumberUtil.formatAmount(totalAmount + ticket.getRoundedAmount())); //$NON-NLS-1$

		map.put("colDueAmount", ReportUtil.createSpan("Due Amount:", dueAmount != 0));
		map.put("dueAmount", currencySymbol + "" + ReportUtil.createSpan(NumberUtil.formatAmount(dueAmount), dueAmount != 0)); //$NON-NLS-2$

		String serverName = "";
		User owner = ticket.getOwner();
		if (owner != null) {
			serverName = owner.getFullName();
		}
		else {
			serverName += ticket.getProperty("owner.firstname", "") + " " + ticket.getProperty("owner.lastname", ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		}
		map.put("colBillBy",
				"Printed: " + DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd MMM yy hh:mm a") + " "
						+ "by" + " " + serverName);

		String payModeDtl = null;

		List<PosTransaction> transactions = ticket.getAllocatedTransactionAndDirectTransactions();
		if (transactions != null) {

			Comparator<PosTransaction> comparator = Comparator.comparing(PosTransaction::getTransactionTime, Comparator.nullsLast(Comparator.reverseOrder()));
			transactions.sort(comparator);

			if (transactions != null && !transactions.isEmpty()) {
				payModeDtl = "<div style='text-align: left;'>"; //$NON-NLS-1$
				for (PosTransaction posTransaction : transactions) {
					String transactionString = "";
					Double amount = posTransaction.getAmount();
					String nameDisplay = posTransaction.buildPaymentTypeDisplayName();
					String simplifyDateAndTime = DateUtil.simplifyDateAndTime(posTransaction.getTransactionTime(), "dd MMM yy");

					if (posTransaction.isVoided()) {
						simplifyDateAndTime += " (VOIDED)";
					}

					if (posTransaction instanceof RefundTransaction) {
						amount = (-1) * amount;
						nameDisplay += " " + POSConstants.REFUND_BUTTON_TEXT; //$NON-NLS-1$
					}
					User user = DataProvider.get().getUserById(posTransaction.getUserId(), posTransaction.getOutletId());
					if (isPharma) {
						transactionString += simplifyDateAndTime + " " + nameDisplay + ": " + (amount < 0 ? "-" : "") + currencySymbol
								+ NumberUtil.getCurrencyFormatWithoutCurrencySymbol(amount < 0 ? amount * -1 : amount) + "<br>";
					}
					else {
						String transactionId = ticket.getId().equals(posTransaction.getTicketId()) ? StringUtils.EMPTY
								: " (" + posTransaction.getTicketId() + ") ";

						String username = user == null ? "" : " (" + user.getFirstName().trim() + ")";
						transactionString += simplifyDateAndTime + " " + nameDisplay + transactionId + username + ": " //$NON-NLS-2$
								+ (amount < 0 ? "-" : "") + currencySymbol
								+ NumberUtil.getCurrencyFormatWithoutCurrencySymbol(amount < 0 ? amount * -1 : amount) + "<br>";
					}
					if (posTransaction.isVoided() || posTransaction instanceof RefundTransaction) {
						transactionString = ReportUtil.createSpan(transactionString, true);
					}
					payModeDtl += transactionString;
				}
				payModeDtl += "</div>";
			}
		}

		map.put("colPayModeDtl", payModeDtl); //$NON-NLS-1$

		String projectDescription = ticket.getProjectNote();
		if (StringUtils.isNotBlank(projectDescription)) {
			map.put("colDescription", "Description: " + projectDescription); //$NON-NLS-1$
		}

		StringBuilder paymentInfo = new StringBuilder();
		if (StringUtils.isNotBlank(payModeDtl)) {
			paymentInfo.append("Payments ----------------- "); //$NON-NLS-1$
		}
		paymentInfo.append(payModeDtl);
		if (StringUtils.isNotBlank(payModeDtl)) {
			map.put("additionalPaymentProperties", paymentInfo.toString()); //$NON-NLS-1$
		}

		map.put("colPrint", "Print: " + DateUtil.formatDateWithTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()))); //$NON-NLS-1$
		Double totalAmountWord = ticket.getTotalAmount();
		String pAmount = totalAmountWord.toString();
		if (!NumberUtil.isFractional(totalAmountWord)) {
			pAmount = String.valueOf(totalAmountWord.intValue());
		}

		String words = getMoneyIntoWords(pAmount);
		map.put("colInWord", "In word: " + words); //$NON-NLS-1$ //$NON-NLS-2$

		if (ticket.isVoided()) {
			map.put("paidMsg", "VOIDED"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		else if (NumberUtil.round(dueAmount) > 0) {
			map.put("paidMsg", "DUE"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		else {
			map.put("paidMsg", "PAID"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		map.put("roomNo", "Room No"); //$NON-NLS-1$//$NON-NLS-2$
		map.put("colPaidAmount", "Paid amount:"); //$NON-NLS-1$//$NON-NLS-2$
		map.put("paidAmount", currencySymbol + "" + NumberUtil.formatAmount(ticket.getPaidAmountAfterRefund())); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("visibleCategoryTotal", outlet.isShowCategoryTotal()); //$NON-NLS-1$

		map.put("colUnit", "Unit"); //$NON-NLS-1$ //$NON-NLS-2$

		StringBuilder extraOrderInfo = new StringBuilder();
		extraOrderInfo.append("Invoice No: " + ticket.getId());
		extraOrderInfo.append("<br>");
		if (StringUtils.isNotBlank(serverName)) {
			extraOrderInfo.append("Server: " + serverName);
			extraOrderInfo.append("<br>");
		}
		extraOrderInfo.append(
				"Printed: " + DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd MMM yy hh:mm a"));
		map.put("additionalOrderInfo", extraOrderInfo.toString()); //$NON-NLS-1$

		map.put("ticketHeader", "Invoice");
		map.put("footerMessage", "Powerd by Medlogics");

		map.put("headerLine1", store != null ? store.getName() : ""); //$NON-NLS-1$
		map.put("discount", "Discount"); //$NON-NLS-1$ //$NON-NLS-2$

		map.put("colSerialNumber", "Serial number"); //$NON-NLS-1$ //$NON-NLS-2$

		addStoreInfo(map, store, outlet);

		if (!NumberUtil.isZero(ticket.getRoundedAmount())) {
			map.put("colRounding", "Rounding: "); //$NON-NLS-1$ //$NON-NLS-2$
			map.put("roundingAmount", NumberUtil.formatAmount(ticket.getRoundedAmount())); //$NON-NLS-1$
		}

	}

	public static void addStoreInfo(HashMap<String, Object> map, Store store, Outlet outlet) {
		if (outlet == null) {
			return;
		}

		StringBuilder storeInfo = new StringBuilder();
		storeInfo.append(outlet.getAddressLine1());
		if (StringUtils.isNotBlank(outlet.getAddressLine2())) {
			storeInfo.append(", " + outlet.getAddressLine2());
		}
		storeInfo.append("<br>");
		Address address = outlet.getAddress();
		String websiteAddress = outlet.getWebsiteAddress();
		if (address != null && StringUtils.isNotBlank(address.getTelephone())) {
			storeInfo.append("Hotline: " + address.getTelephone());
		}
		if (StringUtils.isNotBlank(websiteAddress)) {
			if (StringUtils.isNotBlank(address.getTelephone())) {
				storeInfo.append(", ");
			}
			storeInfo.append("Web: " + "<a style='color: black' href='" + websiteAddress + "'>" //$NON-NLS-1$
					+ websiteAddress + "</a>");
		}
		else {
			storeInfo.append("<br>");
		}

		map.put("headerLine2", storeInfo.toString());

		if (outlet.isShowPrintReceiptStoreLogo()) {
			ImageIcon storeLogo = outlet.getStoreLogo();
			if (storeLogo != null) {
				ImageIcon imageIcon = ImageUtil.pngToJpg(storeLogo);
				if (imageIcon != null) {
					map.put("storeLogo", imageIcon.getImage()); //$NON-NLS-1$
				}
			}
		}

	}

	private static LabTestReportModel createModels(Ticket ticket, String copyFor, boolean enableJasperPrint) {
		LabTestReportModel reportModel = new LabTestReportModel();
		reportModel.setJasperModel(enableJasperPrint);
		List<TicketItem> ticketItems = new ArrayList<>(ticket.getTicketItems());

		for (TicketItem item : ticketItems) {
			if (ProductType.match(item.getProductType(), ProductType.SERVICES)) {
				item.setCategoryName("Consumable services");
			}
		}

		//		Comparator<TicketItem> productTypeComparator = Comparator.comparing(TicketItem::getProductType, Comparator.nullsLast(Comparator.reverseOrder()));
		//		Comparator<TicketItem> comparator = Comparator.comparing(TicketItem::getCategoryName, Comparator.nullsFirst(Comparator.naturalOrder()));
		//		ticketItems.sort(productTypeComparator.thenComparing(comparator));
		//ticketItems.sort(comparator.thenComparing(productTypeComparator));

		List<TicketItem> services = ticketItems.stream().filter(item -> ProductType.match(item.getProductType(), ProductType.SERVICES))
				.collect(Collectors.toList());
		List<TicketItem> goods = ticketItems.stream().filter(item -> ProductType.match(item.getProductType(), ProductType.GOODS)).collect(Collectors.toList());

		ticketItems.removeAll(services);
		ticketItems.removeAll(goods);
		ticketItems.addAll(goods);
		ticketItems.addAll(services);

		List<TicketItem> withoutVoidedItems = ticketItems.stream().filter(item -> !item.isVoided()).collect(Collectors.toList());

		reportModel.setRows(withoutVoidedItems);
		return reportModel;
	}

	public class UnpaidTicketModel extends ListTableModel {

		public UnpaidTicketModel() {
			super(new String[] { "ticketId", "amount" });
		}

		@Override
		public Object getValueAt(int rowIndex, int columnIndex) {
			Ticket ticket = (Ticket) rows.get(rowIndex);
			switch (columnIndex) {
				case 0:
					return ticket.getId();
				case 1:
					return ticket.getDueAmount();
			}
			return null;
		}
	}

}
