/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;
import org.json.JSONArray;
import org.json.JSONObject;

import com.floreantpos.PosLog;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.DoctorTimeSchedule;
import com.floreantpos.model.DurationType;
import com.floreantpos.model.OutdoorDoctorProfile;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.OutdoorDoctorProfileDAO;
import com.floreantpos.model.dao.OutletDAO;
import com.floreantpos.model.util.ServiceChargeType;
import com.floreantpos.util.OrgJsonUtil;

public class UpdateDBTo465 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo465(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void update() {
		updateDoctorSchedules();
	}

	private void updateDoctorSchedules() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$
		try (Session session = sessionFactory.openSession()) {
			updateDoctorSchedules(session);
		}
		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}

	private void updateDoctorSchedules(Session session) {
		Criteria criteria = session.createCriteria(Doctor.class);
		criteria.add(Restrictions.eq(Doctor.PROP_DELETED, Boolean.FALSE));
		criteria.add(Restrictions.eq(Doctor.PROP_HAS_SCHEDULE, Boolean.TRUE));
		List<Doctor> doctors = criteria.list();
		if (doctors == null) {
			doctors = new ArrayList<>();
		}
		PosLog.info(UpdateDBTo465.class, "Found total scheduled doctors: " + doctors.size()); //$NON-NLS-1$
		Transaction transaction = session.beginTransaction();
		List<Outlet> outlets = OutletDAO.getInstance().findAll(session);
		if (outlets != null && outlets.size() > 0) {
			for (Outlet outlet : outlets) {
				for (Doctor doctor : doctors) {
					List<DoctorTimeSchedule> oldSchedules = getOldSchedules(doctor);
					if (oldSchedules != null && oldSchedules.size() > 0) {
						OutdoorDoctorProfile doctorProfile = getOutdoorProfile(doctor.getId(), outlet.getId(), session);
						if (doctorProfile == null) {
							doctorProfile = new OutdoorDoctorProfile();
							DoctorTimeSchedule doctorTimeSchedule = oldSchedules.get(0);
							doctorProfile.setOutdoorFee(doctorTimeSchedule.getOutdoorFee());
							doctorProfile.setRevisitFee(doctorTimeSchedule.getRevisitFee());
							doctorProfile.setRevisitInterval(doctorTimeSchedule.getRevisitInterval());
							doctorProfile.setRevisitDurationType(doctorTimeSchedule.getRevisitDurationType());
							doctorProfile.setDoctorChargeRate(doctorTimeSchedule.getDoctorChargeRate());
							doctorProfile.setDoctorChargeType(doctorTimeSchedule.getDoctorChargeType());
							doctorProfile.setDoctorId(doctor.getId());
							doctorProfile.setDoctorName(doctor.getName());
							doctorProfile.setCreateDate(new Date());
							doctorProfile.setOutletId(outlet.getId());
							doctorProfile.setSchedules(oldSchedules);
							OutdoorDoctorProfileDAO.getInstance().save(doctorProfile, session);
						}
					}
				}
			}
			for (Doctor doctor : doctors) {
				doctor.setHasSchedule(false);
				DoctorDAO.getInstance().saveOrUpdate(doctor, session);
			}
		}
		transaction.commit();
	}

	private OutdoorDoctorProfile getOutdoorProfile(String doctorId, String outletId, Session session) {
		if (StringUtils.isBlank(doctorId)) {
			return null;
		}
		Criteria criteria = session.createCriteria(OutdoorDoctorProfile.class); //$NON-NLS-1$
		criteria.add(Restrictions.eq(OutdoorDoctorProfile.PROP_ACTIVE, Boolean.TRUE));
		criteria.add(Restrictions.eq(OutdoorDoctorProfile.PROP_DOCTOR_ID, doctorId));
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq(OutdoorDoctorProfile.PROP_OUTLET_ID, outletId));
		}
		criteria.setMaxResults(1);
		return (OutdoorDoctorProfile) criteria.uniqueResult();

	}

	private List<DoctorTimeSchedule> getOldSchedules(Doctor doctor) {
		List<DoctorTimeSchedule> dataList = new ArrayList<>();
		JSONObject jsonProperties = doctor.getPropertiesContainer();
		if (jsonProperties == null || !jsonProperties.has("appointmentTimes")) { //$NON-NLS-1$
			return dataList;
		}
		JSONArray jsonArray = jsonProperties.getJSONArray("appointmentTimes"); //$NON-NLS-1$
		for (int i = 0; i < jsonArray.length(); i++) {
			JSONObject jsonObject = jsonArray.getJSONObject(i);
			DoctorTimeSchedule data = new DoctorTimeSchedule();
			data.setShiftName(OrgJsonUtil.getString(jsonObject, DoctorTimeSchedule.PROP_SHIFT_NAME));
			data.setStartTime(OrgJsonUtil.getString(jsonObject, DoctorTimeSchedule.PROP_START_TIME));
			data.setEndTime(OrgJsonUtil.getString(jsonObject, DoctorTimeSchedule.PROP_END_TIME));
			data.setConsultationTime(OrgJsonUtil.getInt(jsonObject, DoctorTimeSchedule.PROP_CONSULTATION_TIME));
			Double consultationFee = OrgJsonUtil.getDouble(jsonObject, DoctorTimeSchedule.PROP_CONSULTATION_FEES);
			data.setConsultationFees(consultationFee);
			data.setOutdoorFee(consultationFee);
			data.setRevisitFee(consultationFee);
			data.setRevisitInterval(7);
			data.setRevisitDurationType(DurationType.DAY.name());
			data.setDoctorChargeRate(100D);
			data.setDoctorChargeType(ServiceChargeType.PERCENTAGE.name());
			data.setDefaultShift(i == 0);

			data.setPatientCount(OrgJsonUtil.getInt(jsonObject, DoctorTimeSchedule.PROP_PATIENT_COUNT));
			data.setId(OrgJsonUtil.getString(jsonObject, DoctorTimeSchedule.PROP_ID));
			data.setDoctorId(doctor.getId());
			List<Integer> days = new ArrayList<>();
			JSONArray daysArray = jsonObject.has("days") ? jsonObject.getJSONArray("days") : null;
			for (int j = 0; j < daysArray.length(); j++) {
				days.add(daysArray.getInt(j));
			}
			data.setDays(days);
			dataList.add(data);
		}
		return dataList;
	}
}