package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.Bed;
import com.floreantpos.model.BedFeature;
import com.floreantpos.model.Floor;
import com.floreantpos.model.Room;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginatedListModel;

public class BedDAO extends BaseBedDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public BedDAO() {
	}

	public void loadData(PaginatedListModel<Bed> tableModel, String name, Floor floor, Room room) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(name)) {
				criteria.add(Restrictions.eq(Bed.PROP_NUMBER, name));
			}

			if (floor != null || room != null) {
				DetachedCriteria subQuery = DetachedCriteria.forClass(Room.class);
				Disjunction or = Restrictions.disjunction();
				if (floor != null) {
					or.add(Restrictions.eq(Room.PROP_FLOOR_ID, floor.getId()));
				}
				if (room != null) {
					or.add(Restrictions.eq(Room.PROP_ID, room.getId()));
				}
				subQuery.add(or);
				subQuery.setProjection(Property.forName(Room.PROP_ID));
				criteria.add(Property.forName(Bed.PROP_ROOM_ID).in(subQuery));
			}
			if (tableModel.getPageSize() > 0) {
				tableModel.setNumRows(rowCount(criteria));
				criteria.setFirstResult(tableModel.getCurrentRowIndex());
				criteria.setMaxResults(tableModel.getPageSize());
			}
			criteria.addOrder(Order.asc(Bed.PROP_ROOM_ID));
			criteria.addOrder(Order.asc(Bed.PROP_NUMBER));
			tableModel.setData(criteria.list());
		}
	}

	public List<Bed> getBeds(String filter, String bedType, String bedFeatureId, String roomId) {
		List<String> roomIds = null;
		if (StringUtils.isNotBlank(bedType)) {
			roomIds = RoomDAO.getInstance().getRoomIds(bedType);
			if ((roomIds == null || roomIds.isEmpty())) {
				return new ArrayList<Bed>();
			}
		}
		try (Session session = GenericDAO.getInstance().createNewSession()) {
			Criteria criteria = createCommonCriteria(session, filter, roomIds, bedFeatureId, roomId);
			criteria.addOrder(Order.asc(Bed.PROP_ROOM_ID));
			criteria.addOrder(Order.asc(Bed.PROP_NUMBER));
			return criteria.list();
		}
	}

	private Criteria createCommonCriteria(Session session, String filter, List<String> roomIds, String bedFeatureId, String roomId) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		GenericDAO.getInstance().addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(Bed.PROP_OUTLET_ID, DataProvider.get().getCurrentOutletId()));
		criteria.add(Restrictions.eq(Bed.PROP_ACTIVE, Boolean.TRUE));
		if (StringUtils.isNotBlank(filter)) {
			criteria.add(Restrictions.ilike(Bed.PROP_NUMBER, filter, MatchMode.ANYWHERE));
		}
		if (StringUtils.isNotBlank(bedFeatureId)) {
			criteria.createAlias("features", "f");
			criteria.add(Restrictions.eq("f." + BedFeature.PROP_ID, bedFeatureId));
		}

		if (StringUtils.isNotBlank(roomId)) {
			criteria.add(Restrictions.eq(Bed.PROP_ROOM_ID, roomId));
		}
		else if (roomIds != null && roomIds.size() > 0) {
			criteria.add(Restrictions.in(Bed.PROP_ROOM_ID, roomIds));
		}
		return criteria;
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public List<Bed> findAll(Floor floor, Room room, String value) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (floor != null || room != null) {
				DetachedCriteria subQuery = DetachedCriteria.forClass(Room.class);
				if (room != null) {
					subQuery.add(Restrictions.eq(Room.PROP_ID, room.getId()));
				}
				else if (floor != null) {
					subQuery.add(Restrictions.eq(Room.PROP_FLOOR_ID, floor.getId()));
				}
				subQuery.setProjection(Property.forName(Room.PROP_ID));
				criteria.add(Property.forName(Bed.PROP_ROOM_ID).in(subQuery));
			}
			criteria.add(Restrictions.or(Restrictions.eq(Bed.PROP_ACTIVE, Boolean.TRUE), Restrictions.isNull(Bed.PROP_ACTIVE)));
			criteria.addOrder(Order.asc(Bed.PROP_ROOM_ID));
			criteria.addOrder(Order.asc(Bed.PROP_NUMBER));
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session session) {
		Bed data = (Bed) obj;
		if (data == null) {
			throw new PosException("Data not found.");
		}
		data.setDeleted(Boolean.TRUE);
		saveOrUpdate(data, session);
	}

	public List<Bed> findByRoomId(String id) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Bed.PROP_OUTLET_ID, DataProvider.get().getCurrentOutletId()));
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Bed.PROP_ROOM_ID, id));
			criteria.addOrder(Order.asc(Bed.PROP_NUMBER));
			return criteria.list();
		}
	}

	public List<String> findBookingIdsByRoomId(String roomId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Bed.PROP_ROOM_ID, roomId));
			ProjectionList projectionList = Projections.projectionList();
			criteria.setProjection(projectionList.add(Property.forName(Bed.PROP_BOOKING_ID), Bed.PROP_BOOKING_ID));
			criteria.addOrder(Order.asc(Bed.PROP_NUMBER));
			return criteria.list();
		}
	}

	public void releaseBed(TicketItem ticketItem, String bookingId, Session session) {
		String bedId = ticketItem.getBedId();
		if (StringUtils.isBlank(bedId)) {
			return;
		}
		Bed bed = BedDAO.getInstance().get(bedId, session);
		if (bed == null || !bed.isOccupied() || (bed.getBookingId() != null && !bed.getBookingId().equals(bookingId))) {
			return;
		}
		bed.setOccupied(false);
		bed.putCustomerName(null);
		bed.setBookingId(null);
		BedDAO.getInstance().saveOrUpdate(bed, session);
	}
}