package com.floreantpos.services.report;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import org.hibernate.Session;

import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.dao.GenericDAO;
import com.floreantpos.model.ext.PaperSize;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.report.ReportConstants;
import com.floreantpos.report.ReportItem;
import com.floreantpos.report.ReportUtil;
import com.floreantpos.report.SalesReportModel;
import com.floreantpos.report.SalesReportUtil;
import com.floreantpos.util.CurrencyUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

import net.sf.jasperreports.engine.JREmptyDataSource;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;

public class SalesTaxReportService {
	private Date startDate;
	private Date toDate;
	private Outlet outlet;

	public SalesTaxReportService(Date startDate, Date toDate, Outlet outlet) {
		this.startDate = startDate;
		this.toDate = toDate;
		this.outlet = outlet;
	}

	public JasperPrint createSalesTaxReportJasperPrint() throws JRException {
		SalesReportModel dailySalesReportModel = createSalesTaxReportModels();
		return createSalesTaxReportJasperPrint(dailySalesReportModel);
	}

	public JasperPrint createSalesTaxReportJasperPrint(SalesReportModel dailySalesReportModel) throws JRException {
		JasperReport dailySalesReport = ReportUtil.getReport(PaperSize.getReportNameAccording2Size(ReportConstants.SALES_TAX_SUB_REPORT));

		HashMap<String, Object> map = new HashMap<String, Object>();
		ReportUtil.populateRestaurantProperties(map, outlet);
		ReportUtil.populateReportHeader(map, outlet, startDate, toDate, null, true);
		populateReportParameter(map);

		map.put("dailySalesDataSource", new JRTableModelDataSource(dailySalesReportModel)); //$NON-NLS-1$
		map.put("dailySalesReport", dailySalesReport); //$NON-NLS-1$

		JasperReport masterReport = ReportUtil.getReport(PaperSize.getReportNameAccording2Size(ReportConstants.SALES_TAX_REPORT));
		JasperPrint print = JasperFillManager.fillReport(masterReport, map, new JREmptyDataSource());
		return print;
	}

	private void populateReportParameter(HashMap<String, Object> map) {
		String reportTitle = POSConstants.SALES_TAX_REPORT;
		String colName = POSConstants.DATE;

		map.put("reportTitle", reportTitle); //$NON-NLS-1$ 
		map.put("colName", ReportUtil.reportLabelWithBoldTag(colName));//$NON-NLS-1$
		map.put("currencySymbol", CurrencyUtil.getCurrencySymbolWithBracket()); //$NON-NLS-1$

		map.put("colId", ReportUtil.reportLabelWithBoldTag(POSConstants.ID));//$NON-NLS-1$
		map.put("colQty", ReportUtil.reportLabelWithBoldTag(POSConstants.QTY));//$NON-NLS-1$
		map.put("colUnit", ReportUtil.reportLabelWithBoldTag(POSConstants.UNIT));//$NON-NLS-1$
		map.put("colSales", ReportUtil.reportLabelWithBoldTag(POSConstants.UNIT_PRICE));//$NON-NLS-1$
		map.put("colDiscount", ReportUtil.reportLabelWithBoldTag(POSConstants.DISCOUNT));//$NON-NLS-1$
		map.put("colCost", ReportUtil.reportLabelWithBoldTag(POSConstants.COST));//$NON-NLS-1$
		map.put("colCostPer", ReportUtil.reportLabelWithBoldTag(POSConstants.COST_PERCENTAGE));//$NON-NLS-1$
		map.put("colNetTotal", ReportUtil.reportLabelWithBoldTag(POSConstants.NET_SALES));//$NON-NLS-1$
		map.put("colTax", ReportUtil.reportLabelWithBoldTag(POSConstants.TAX));//$NON-NLS-1$	
		map.put("colSC", ReportUtil.reportLabelWithBoldTag(POSConstants.S_C));//$NON-NLS-1$
		map.put("colGrossT", ReportUtil.reportLabelWithBoldTag(POSConstants.TOTAL));//$NON-NLS-1$

		map.put("groupTotal", ReportUtil.reportLabelWithBoldTag(POSConstants.GROUP_TOTAL));//$NON-NLS-1$
		map.put("lblItemTotal", ReportUtil.reportLabelWithBoldTag(POSConstants.GRAND_TOTAL));//$NON-NLS-1$
		map.put("lblModifierTotal", ReportUtil.reportLabelWithBoldTag(POSConstants.MODIFIER_TOTAL + ": "));//$NON-NLS-1$ //$NON-NLS-2$
		map.put("lblGrandTotal", ReportUtil.reportLabelWithBoldTag(POSConstants.GRAND_TOTAL));//$NON-NLS-1$

		map.put("lblGrossSales", ReportUtil.reportLabelWithBoldTag(POSConstants.GROSS_SALES)); //$NON-NLS-1$
		map.put("lblServiceCharge", ReportUtil.reportLabelWithBoldTag(Messages.getString("SalesReportService.0"))); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("lblReturnAmount", ReportUtil.reportLabelWithBoldTag(POSConstants.RETURN)); //$NON-NLS-1$
		map.put("lblProfit", ReportUtil.reportLabelWithBoldTag(POSConstants.GROSS_PROFIT)); //$NON-NLS-1$
		map.put("lblProfitPer", ReportUtil.reportLabelWithBoldTag(POSConstants.PROFIT + " %")); //$NON-NLS-1$ //$NON-NLS-2$

		map.put("itemSection", POSConstants.ITEMS); //$NON-NLS-1$ 
	}

	public SalesReportModel createSalesTaxReportModels() {

		List<Date> betweenDates = POSUtil.getMonthBetweenDates(startDate, toDate);

		ArrayList<ReportItem> reportItemList = new ArrayList<>();

		GenericDAO dao = new GenericDAO();
		try (Session session = dao.createNewSession();) {
			session.setDefaultReadOnly(true);

			for (Date reportDate : betweenDates) {
				Date startOfDay;
				Date endOfDay;
				ReportItem reportItem = new ReportItem();

				startOfDay = DateUtil.startOfMonth(reportDate);
				endOfDay = DateUtil.endOfMonth(reportDate);

				reportItem.setName(DateUtil.format("MMM yyyy", startOfDay)); //$NON-NLS-1$

				SalesReportUtil salesReportUtil = new SalesReportUtil(startOfDay, endOfDay);
				salesReportUtil.setOutlet(outlet);

				double calculateGrossSales = salesReportUtil.calculateGrossSalesWithOutIncludedTax(session);
				double calculateDiscountAmount = salesReportUtil.calculateDiscountAmount(session);
				double calculateReturnAmount = salesReportUtil.calculateReturnAmountWithOutIncludedTax(session);
				calculateDiscountAmount = NumberUtil.isZero(calculateDiscountAmount) ? 0.0 : (-1) * calculateDiscountAmount;
				calculateReturnAmount = NumberUtil.isZero(calculateReturnAmount) ? 0.0 : (-1) * calculateReturnAmount;
				double netTotalAmount = calculateGrossSales + calculateDiscountAmount + calculateReturnAmount;
				double calculateServiceChargeAmount = salesReportUtil.calculateTotalServiceChargeAmount(session);
				double calculateSalesTaxAmount = salesReportUtil.calculateSalesTaxAmount(session);

				reportItem.setGrossSale(calculateGrossSales);
				reportItem.setDiscount(calculateDiscountAmount);
				reportItem.setReturnAmount(calculateReturnAmount);
				reportItem.setNetTotal(netTotalAmount);
				reportItem.setServiceCharge(calculateServiceChargeAmount);
				reportItem.setTaxTotal(calculateSalesTaxAmount);
				//reportItem.setTotal(netTotalAmount + calculateServiceChargeAmount + calculateSalesTaxAmount);
				reportItem.setGrossTotal(netTotalAmount + calculateServiceChargeAmount + calculateSalesTaxAmount);
				reportItem.calculate();
				reportItemList.add(reportItem);
			}
		}

		SalesReportModel dailySalesReportModel = new SalesReportModel();
		dailySalesReportModel.setItems(reportItemList);
		return dailySalesReportModel;
	}

}
