/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import org.hibernate.Criteria;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.StoreBalance;
import com.floreantpos.model.dao.OutletDAO;
import com.floreantpos.model.dao.StoreBalanceDAO;
import com.floreantpos.model.dao.StoreDAO;

public class UpdateDBTo472 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo472(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void update() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		try (Session session = sessionFactory.openSession()) {
			Transaction transaction = session.beginTransaction();

			Store store = StoreDAO.getInstance().get("1", session);
			Double storeCashBalance = store.getCashBalance();
			PosLog.info(getClass(), "Store Cash Balance: " + storeCashBalance);

			double totalOutletBalance = 0;
			for (String outletId : OutletDAO.getInstance().findAllOutletIds()) {

				Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, outletId));
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.STORE_BALANCE.name()));
				criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));
				Object uniqueResult = criteria.uniqueResult();
				double amount = 0;
				if (uniqueResult != null) {
					amount = (Double) uniqueResult;
				}

				PosLog.info(getClass(), "Outlet ID: " + outletId + " | Balance from transactions: " + amount);

				totalOutletBalance += amount;

				StoreBalance storeBalance = StoreBalanceDAO.getInstance().getStoreBalance(outletId, session);
				if (storeBalance == null) {
					storeBalance = new StoreBalance();
					storeBalance.setCreateDate(StoreDAO.getServerTimestamp());
					storeBalance.setOutletId(outletId);
				}
				storeBalance.setCashBalance(amount);
				StoreBalanceDAO.getInstance().saveOrUpdate(storeBalance, session);
			}

			PosLog.info(getClass(), "Total Balance from All Outlets: " + totalOutletBalance);

			// Check if store balance == sum of outlet balances (with tolerance)
			double tolerance = 0.01;
			if (Math.abs(storeCashBalance - totalOutletBalance) <= tolerance) {
				PosLog.info(getClass(), "✅ Store Cash Balance matches total outlet balances.");
			}
			else {
				PosLog.info(getClass(), "❌ Store Cash Balance does NOT match total outlet balances.");
				PosLog.info(getClass(), "Difference: " + (storeCashBalance - totalOutletBalance));
			}

			String sql = String.format("update %s.balance_update_transaction set account_number = outlet_id from %s.store s where account_number = s.id;",
					schemaName, schemaName);
			SQLQuery sqlQuery = session.createSQLQuery(sql);
			PosLog.info(getClass(), "updated query: " + sqlQuery + "");
			int update = sqlQuery.executeUpdate();
			PosLog.info(getClass(), "updated " + update + " BALANCE_UPDATE_TRANSACTION table's account_number column with corresponding store's def_outlet_id"); //$NON-NLS-1$ //$NON-NLS-2$
			transaction.commit();
		}

		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}

}