package com.floreantpos.hl7;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Session;

import com.floreantpos.PosLog;
import com.floreantpos.hl7.model.Result;
import com.floreantpos.hl7.model.Test;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.MachineResult;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.dao.GlobalConfigDAO;
import com.floreantpos.model.dao.InventoryUnitDAO;
import com.floreantpos.model.dao.MachineResultDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.dao.TicketItemDAO;
import com.google.gson.Gson;

public class HL7Util {

	private final static SimpleDateFormat mythicFormatter = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss");
	private Session session;
	private String storeId;
	private String outletId;

	public HL7Util() {
	}

	public HL7Util(Session session, String storeId, String outletId) {
		this.session = session;
		this.storeId = storeId;
		this.outletId = outletId;

	}

	public static String cleanTestName(String testName) {
		//String[] chars = { "%", "#", "^" };
		String[] chars = { "^" }; //$NON-NLS-1$
		//System.out.println("cleaning: " + testName);
		for (int i = 0; i < chars.length; i++) {
			int index = testName.indexOf(chars[i]);
			if (index > 0) {
				return testName.substring(0, index);
			}
		}

		return testName;
	}

	public synchronized void updateUnit(Map<String, InventoryUnit> unitsMap, Result result, String unitName) throws Exception {
		if (StringUtils.isNotBlank(unitName)) {
			InventoryUnit inventoryUnit = unitsMap.get(unitName);
			if (inventoryUnit == null) {
				inventoryUnit = InventoryUnitDAO.getInstance().findByName(unitName, session);
				if (inventoryUnit == null) {
					inventoryUnit = new InventoryUnit();
					inventoryUnit.setConversionRate(1.0);
					inventoryUnit.setName(unitName);
					if (unitName.length() > 10) {
						inventoryUnit.setCode(String.format("%.10s", unitName)); //$NON-NLS-1$
					}
					else {
						inventoryUnit.setCode(unitName);
					}

					InventoryUnitDAO.getInstance().save(inventoryUnit, session);
				}
				unitsMap.put(unitName, inventoryUnit);
			}

			result.unit = inventoryUnit.getId();
			result.inventoryUnit = inventoryUnit;
		}
	}

	public synchronized Test saveMachineResult(Test test, String postData) throws Exception {
		String sampleId = test.sampleId;
		if (StringUtils.isBlank(sampleId)) {
			PosLog.error(HL7Util.class, "Sample id not found for machine data");
			sampleId = "00000";
		}
		PosLog.debug(HL7Util.class, String.format("Sample id: %s", sampleId));

		MachineResult machineResult = new MachineResult();
		machineResult.setStoreId(storeId);

		if (StringUtils.isNotBlank(test.date)) {
			machineResult.setResultDate(convertToMithicFormatterDate(test.date, test.time));
		}
		else {
			machineResult.setResultDate(new Date());
		}
		machineResult.setSampleId(sampleId);
		machineResult.setOutletId(outletId);
		machineResult.setResultJson(new Gson().toJson(test));
		machineResult.putMachineRawData(postData);

		MachineResultDAO.getInstance().saveOrUpdate(machineResult, session);

		PosLog.debug(HL7Util.class, "Result has been stored.");

		TicketItem ticketItemByID = null;

		boolean isSampleIdMatchWithInvoice = GlobalConfigDAO.getInstance().isSampleIdMatchWithInvoice(session);
		if (isSampleIdMatchWithInvoice) {
			Ticket ticket = TicketDAO.getInstance().loadFullTicket(sampleId, outletId, session);
			if (ticket != null) {
				List<TicketItem> ticketItems = ticket.getTicketItems();
				if (ticketItems != null) {
					for (TicketItem ticketItem : ticketItems) {
						if (ticketItem.getName().toLowerCase().startsWith("cbc")) {
							ticketItemByID = ticketItem;
							PosLog.debug(HL7Util.class, String.format("CBC item %s found from invoice %s.", ticketItem.getId(), sampleId));
							break;
						}
					}
				}
			}

			if (ticketItemByID == null) {
				PosLog.debug(HL7Util.class, String.format("CBC item not found from invoice %s.", sampleId));
				return test;
			}
		}
		else {
			ticketItemByID = TicketItemDAO.getInstance().findByLabTest(sampleId, session);
			if (ticketItemByID == null) {
				PosLog.debug(HL7Util.class, String.format("Ticket item for sample %s not found", sampleId));
				return test;
			}
		}

		
		new MachineResultUtil(storeId, outletId).addResultToTicketItem(test, ticketItemByID);

		TicketItemDAO.getInstance().saveOrUpdate(ticketItemByID, session);

		return test;
	}

	private Date convertToMithicFormatterDate(String date, String time) {
		if (StringUtils.isNotBlank(date)) {
			String dateTimeString = date + " " + time;
			try {
				TimeZone timeZone = TimeZone.getTimeZone("Asia/Dhaka"); //$NON-NLS-1$
				//				if (StringUtils.isNotBlank(outlet.getProperty(WorkingHours.TIME_ZONE))) {
				//					timeZone = TimeZone.getTimeZone(outlet.getProperty(WorkingHours.TIME_ZONE));
				//				}
				//				else {
				//					timeZone = TimeZone.getTimeZone("Asia/Dhaka"); //$NON-NLS-1$
				//				}
				mythicFormatter.setTimeZone(timeZone);
				return mythicFormatter.parse(dateTimeString.trim());
			} catch (Exception e) {
			}
		}
		return new Date();
	}

	public Session getSession() {
		return session;
	}

	public void setSession(Session session) {
		this.session = session;
	}

	public String getStoreId() {
		return storeId;
	}

	public void setStoreId(String storeId) {
		this.storeId = storeId;
	}

	public String getOutletId() {
		return outletId;
	}

	public void setOutletId(String outletId) {
		this.outletId = outletId;
	}

}
