package com.floreantpos.hl7;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Session;

import com.floreantpos.hl7.model.Result;
import com.floreantpos.hl7.model.Test;
import com.floreantpos.jdbc.OroMultitenantConnectionProvider;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.MachineResult;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemResult;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.dao.InventoryUnitDAO;
import com.floreantpos.model.dao.MachineResultDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.TestItemDAO;
import com.floreantpos.model.ext.LabWorkStatus;
import com.floreantpos.model.util.InventoryUnitConversionRule;
import com.floreantpos.util.NumberUtil;
import com.google.gson.Gson;

public class MachineResultUtil {

	private String storeId;
	private String outletId;

	public MachineResultUtil(String storeId, String outletId) {
		super();
		this.storeId = storeId;
		this.outletId = outletId;
	}

	public synchronized void addResultToTicketItem(Test test, TicketItem ticketItem) throws Exception {
		Map<String, Result> machineResultmap = test.resultmap;
		List<TestItem> menuItemTestItems = getMenuItemTestItems(ticketItem.getMenuItemId());
		for (TestItem testItem : menuItemTestItems) {
			String machineCode = testItem.getMachineCode();
			if (StringUtils.isBlank(machineCode)) {
				continue;
			}

			Result machineResult = machineResultmap.get(machineCode);
			if (machineResult == null) {
				continue;
			}
			testItem.putOriginalMachineResult(machineResult.result);
			if (StringUtils.isBlank(testItem.getUnit())) {
				testItem.setUnit(machineResult.unit);
			}
			else {
				convertMachineResultToItemUnit(testItem, machineResult);
			}
			testItem.setResult(machineResult.result);
			testItem.putMachineName(test.machine);
			//mythic data
			if (StringUtils.isNotBlank(machineResult.normalLow) || StringUtils.isNotBlank(machineResult.normalHigh)) {
				testItem.setNormalValue(machineResult.normalLow + " - " + machineResult.normalHigh);
			}
			machineResultmap.remove(machineCode);
		}

		int sortOrder = menuItemTestItems.size();
		List<Result> remainResults = new ArrayList<Result>(machineResultmap.values());
		for (int i = 0; i < remainResults.size(); i++) {
			Result machineResult = remainResults.get(i);

			String machineCode = machineResult.testName;

			TestItem testItem = TestItemDAO.getInstance().findByMachineCode(machineCode);
			if (testItem == null) {
				testItem = new TestItem();
				testItem.setName(machineCode);
				testItem.setMachineCode(machineCode);
			}
			else {
				testItem.setLastSyncTime(null);
				testItem.setLastUpdateTime(null);
			}

			testItem.putMachineName(test.machine);
			testItem.setId(machineResult.resultNo);
			testItem.putOriginalMachineResult(machineResult.result);
			String testItemUnit = testItem.getUnit();
			if (StringUtils.isBlank(testItemUnit)) {
				testItem.setUnit(machineResult.unit);
			}
			else {
				convertMachineResultToItemUnit(testItem, machineResult);
			}
			testItem.setResult(machineResult.result);

			//mythic data
			if (StringUtils.isNotBlank(machineResult.normalLow) || StringUtils.isNotBlank(machineResult.normalHigh)) {
				testItem.setNormalValue(machineResult.normalLow + " - " + machineResult.normalHigh);
			}

			testItem.putShowTestItem(false);
			testItem.setSortOrder(++sortOrder);
			menuItemTestItems.add(testItem);
			//parsedResult += String.format("Result no: %s, name=%s, value: %s, unit: %s", result.resultNo, result.testName, result.result, result.unit);
			//parsedResult += "\n";
		}

		List<TestItemResult> testItemResults = ticketItem.getTestItemResults();
		testItemResults.clear();
		TestItemResult testItemResult = new TestItemResult();
		testItemResult.setSortOrder(testItemResults.size());
		testItemResult.setTestItems(menuItemTestItems);
		testItemResults.add(testItemResult);
		ticketItem.setTestItemResults(testItemResults);

		ticketItem.setTestItems(menuItemTestItems);
		ticketItem.putMachineResultReceived(true);
		ticketItem.putMachineResultTime(new Date());
		ticketItem.setLabWorkStatusValue(LabWorkStatus.RESULT_RECORDED);
	}

	public synchronized void addResultToTicketItem(String testId, TicketItem ticketItem) throws Exception {
		OroMultitenantConnectionProvider.initTenant(storeId);
		MachineResult machineResult = MachineResultDAO.getInstance().getOrderHistoryBySampleId(testId);
		addResultToTicketItem(ticketItem, machineResult);
	}

	public void addResultToTicketItem(TicketItem ticketItem, MachineResult machineResult) throws Exception {
		ticketItem.setTestItemResults(Collections.emptyList());
		if (machineResult != null && ticketItem != null) {
			String resultJson = machineResult.getResultJson();
			Test test = new Gson().fromJson(resultJson, Test.class);
			addResultToTicketItem(test, ticketItem);
			ticketItem.putMachineResultTime(machineResult.getResultDate());
		}
		else {
			ticketItem.putMachineResultReceived(false);
			ticketItem.putMachineResultTime(null);
		}
	}

	public synchronized List<TestItem> getMenuItemTestItems(String menuItemId) throws Exception {
		OroMultitenantConnectionProvider.initTenant(storeId);

		try (Session session = MenuItemDAO.getInstance().createNewSession()) {
			MenuItem menuItem = MenuItemDAO.getInstance().loadInitialized(menuItemId, session);

			List<TestItem> testItems = menuItem.getTestItems();
			if (testItems != null) {
				int sortorder = 1;
				for (TestItem testItem : testItems) {
					if (testItem == null) {
						continue;
					}
					testItem.setSortOrder(sortorder);
					sortorder++;
				}
			}
			return testItems;
		}
	}

	public synchronized void convertMachineResultToItemUnit(TestItem testItem, Result result) {
		try {
			String unitid = testItem.getUnit();
			if (org.apache.commons.lang.StringUtils.isBlank(unitid)) {
				return;
			}
			OroMultitenantConnectionProvider.initTenant(storeId);
			InventoryUnit inventoryUnit = InventoryUnitDAO.getInstance().findInventoryUnitById(unitid);
			if (inventoryUnit == null) {
				return;
			}
			if (inventoryUnit.getConversionRate() > 0) {
				double newRsult = Double.valueOf(result.result);

				String inventoryUnitConversionRuleName = inventoryUnit.getProperty(InventoryUnit.PROP_CONVERSION_RULE,
						InventoryUnitConversionRule.DIVISION.getName());
				InventoryUnitConversionRule inventoryUnitConversionRule = InventoryUnitConversionRule.fromName(inventoryUnitConversionRuleName);
				if (inventoryUnitConversionRule == InventoryUnitConversionRule.MULTIPLICATION) {
					newRsult = newRsult * 1D / inventoryUnit.getConversionRate();
				}
				else {
					newRsult = newRsult / inventoryUnit.getConversionRate();
				}
				result.result = NumberUtil.formatNumber(newRsult);
			}
		} catch (Exception e) {
			//ignore
		}
	}
}
