package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.Subqueries;
import org.hibernate.sql.JoinType;
import org.hibernate.transform.Transformers;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.Bed;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.DoctorVisit;
import com.floreantpos.model.PatientBookingStatus;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.util.ServiceChargeType;
import com.floreantpos.model.util.ServiceType;
import com.floreantpos.swing.PaginatedListModel;

public class DoctorVisitDAO extends BaseDoctorVisitDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public DoctorVisitDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		Serializable save = super.save(obj, s);
		performPostSaveOperations((DoctorVisit) obj, s);
		return save;
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.update(obj, s);
		performPostSaveOperations((DoctorVisit) obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		DoctorVisit doctorVisit = (DoctorVisit) obj;
		doctorVisit.setDeleted(true);
		update(doctorVisit, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		doCheckValidation(obj);
		super.saveOrUpdate(obj, s);
		performPostSaveOperations((DoctorVisit) obj, s);
	}

	public static void doCheckValidation(Object obj) {
		DoctorVisit doctorVisit = (DoctorVisit) obj;
		String visitReason = doctorVisit.getVisitReason();
		if (StringUtils.isNotBlank(visitReason) && visitReason.length() > 255) {
			throw new PosException(String.format(Messages.getString("value_too_long"), DoctorVisit.PROP_VISIT_REASON, 255)); //$NON-NLS-1$
		}

	}

	public void performPostSaveOperations(DoctorVisit doctorVisit, Session session) {
		PatientHistoryDAO.getInstance().saveDoctorVisitHistory(doctorVisit, session);
	}

	public void saveOrUpdateWithTicket(DoctorVisit doctorVisit) throws Exception {
		if (StringUtils.isBlank(doctorVisit.getDoctorId())) {
			throw new PosException("The visited doctor field cannot be empty!");
		}
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			Doctor doctor = DoctorDAO.getInstance().get(doctorVisit.getDoctorId(), session);
			if (doctor == null) {
				throw new PosException("Doctor not found by id: " + doctorVisit.getDoctorId());
			}
			boolean isNewDoctorVisit = StringUtils.isBlank(doctorVisit.getId());
			if (isNewDoctorVisit) {
				save(doctorVisit, session);
			}
			else {
				saveOrUpdate(doctorVisit, session);
			}

			BookingInfo bookingInfo = doctorVisit.getBooking();
			Ticket bookingTicket = bookingInfo.getTicket();
			bookingTicket = TicketDAO.getInstance().loadFullTicket(bookingTicket.getId(), bookingTicket.getOutletId(), session);

			TicketItem dvTicketItem = new TicketItem();
			dvTicketItem.setProductType(ProductType.SERVICES.name());
			dvTicketItem.setServiceType(ServiceType.CONSULTATION_FEE.name());
			dvTicketItem.setService(true);
			dvTicketItem.setTicketDiscountApplicable(false);
			dvTicketItem.setName("Consultant fee");
			dvTicketItem.putDoctorVisitItem(true);
			dvTicketItem.putShowInAdmissionItemsGrid(true);
			dvTicketItem.setQuantity(1d);
			dvTicketItem.setUnitPrice(doctor.getHospitalRoundFee());
			dvTicketItem.setLabDoctorRequired(true);
			dvTicketItem.setLabDoctorId(doctor.getId());
			dvTicketItem.putDoctorFeeRate(doctor.getHospitalRoundFee());
			dvTicketItem.putDoctorFeeType(ServiceChargeType.FIXEDAMOUNT.name());
			dvTicketItem.setAllowDrFeeBeforeLabwork(true);
			dvTicketItem.putVisitedDoctorId(doctor.getId());
			dvTicketItem.putVisitedDoctorName(doctor.getName());
			dvTicketItem.putDoctorVisitId(doctorVisit.getId());

			bookingTicket.addToticketItems(dvTicketItem);
			bookingTicket.calculatePrice();
			TicketDAO.getInstance().saveOrUpdate(bookingTicket, session);

			tx.commit();
		}
	}

	@SuppressWarnings("unchecked")
	public void loadDoctorVisit(PaginatedListModel<DoctorVisit> dataModel, String patientId, String roomId, String bedId, String doctorId, String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(DoctorVisit.class);
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(DoctorVisit.PROP_OUTLET_ID, outletId));
			}
			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq(DoctorVisit.PROP_PATIENT_ID, patientId));
			}
			createRoomAndBedCriteria(roomId, bedId, criteria, DoctorVisit.PROP_ADMISSION_ID, PatientBookingStatus.ACTIVE);
			if (StringUtils.isNotBlank(doctorId)) {
				criteria.add(Restrictions.eq(DoctorVisit.PROP_DOCTOR_ID, doctorId));
			}
			dataModel.setNumRows(rowCount(criteria));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(DoctorVisit.PROP_ID), DoctorVisit.PROP_ID);
			projectionList.add(Projections.property(DoctorVisit.PROP_ADMISSION_ID), DoctorVisit.PROP_ADMISSION_ID);
			projectionList.add(Projections.property(DoctorVisit.PROP_PATIENT_ID), DoctorVisit.PROP_PATIENT_ID);
			projectionList.add(Projections.property(DoctorVisit.PROP_DOCTOR_ID), DoctorVisit.PROP_DOCTOR_ID);
			projectionList.add(Projections.property(DoctorVisit.PROP_CREATED_DATE), DoctorVisit.PROP_CREATED_DATE);
			projectionList.add(Projections.property(DoctorVisit.PROP_NEXT_FOLLOW_UP_DATE), DoctorVisit.PROP_NEXT_FOLLOW_UP_DATE);
			projectionList.add(Projections.property(DoctorVisit.PROP_VISIT_DATE), DoctorVisit.PROP_VISIT_DATE);
			projectionList.add(Projections.property(DoctorVisit.PROP_VISIT_NO), DoctorVisit.PROP_VISIT_NO);
			projectionList.add(Projections.property(DoctorVisit.PROP_PROPERTIES), DoctorVisit.PROP_PROPERTIES);
			criteria.setProjection(projectionList);

			criteria.addOrder(Order.desc(DoctorVisit.PROP_CREATED_DATE));
			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());
			criteria.setResultTransformer(Transformers.aliasToBean(DoctorVisit.class));
			dataModel.setData(criteria.list());
		}
	}

	public void loadLastDoctorVisits(PaginatedListModel<DoctorVisit> dataModel, String patientId, String roomId, String bedId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(DoctorVisit.class, "doctorVisitMain");
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq("doctorVisitMain." + DoctorVisit.PROP_PATIENT_ID, patientId));
			}

			createRoomAndBedCriteria(roomId, bedId, criteria, DoctorVisit.PROP_ADMISSION_ID);

			// Subquery for getting the latest visitDate for each patientId and admissionId
			DetachedCriteria subquery = DetachedCriteria.forClass(DoctorVisit.class, "sub")
					.setProjection(Projections.projectionList().add(Projections.max("sub." + DoctorVisit.PROP_VISIT_DATE))
							.add(Projections.groupProperty("sub." + DoctorVisit.PROP_PATIENT_ID))
							.add(Projections.groupProperty("sub." + DoctorVisit.PROP_ADMISSION_ID)));

			// Apply the subquery condition
			criteria.add(Subqueries.propertiesIn(new String[] { "doctorVisitMain." + DoctorVisit.PROP_VISIT_DATE,
					"doctorVisitMain." + DoctorVisit.PROP_PATIENT_ID, "doctorVisitMain." + DoctorVisit.PROP_ADMISSION_ID }, subquery));

			dataModel.setNumRows(rowCount(criteria));
			criteria.addOrder(Order.desc("doctorVisitMain." + DoctorVisit.PROP_CREATED_DATE));
			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());

			dataModel.setData(criteria.list());
		}
	}

	public static void createRoomAndBedCriteria(String roomId, String bedId, Criteria criteria, String propertyName) {
		createRoomAndBedCriteria(roomId, bedId, criteria, propertyName, null);
	}

	public static void createRoomAndBedCriteria(String roomId, String bedId, Criteria criteria, String propertyName, PatientBookingStatus status) {
		if (StringUtils.isNotBlank(roomId) || StringUtils.isNotBlank(bedId) || status != null) {

			DetachedCriteria criteriaBoooking = DetachedCriteria.forClass(BookingInfo.class, "b_info"); //$NON-NLS-1$
			criteriaBoooking.createAlias("b_info.beds", "beds", JoinType.LEFT_OUTER_JOIN); //$NON-NLS-1$ //$NON-NLS-2$

			criteriaBoooking.setProjection(Projections.property(BookingInfo.PROP_BOOKING_ID));

			if (status != null) {
				criteriaBoooking.add(Restrictions.eq(BookingInfo.PROP_STATUS, status.name()));
			}

			if (StringUtils.isNotBlank(bedId) && bedId.equalsIgnoreCase("NO_BED")) {
				criteriaBoooking.add(Restrictions.sizeEq("beds", 0));
			}
			else if (StringUtils.isNotBlank(roomId) || StringUtils.isNotBlank(bedId)) {
				DetachedCriteria bedDetachedCriteria = DetachedCriteria.forClass(Bed.class);
				bedDetachedCriteria.setProjection(Projections.property(Bed.PROP_ID));

				if (StringUtils.isNotBlank(roomId)) {
					bedDetachedCriteria.add(Restrictions.eq(Bed.PROP_ROOM_ID, roomId));
				}

				if (StringUtils.isNotBlank(bedId)) {
					bedDetachedCriteria.add(Restrictions.eq(Bed.PROP_ID, bedId));
				}

				criteriaBoooking.add(Property.forName("beds.id").in(bedDetachedCriteria)); //$NON-NLS-1$
			}

			criteria.add(Property.forName(propertyName).in(criteriaBoooking));
		}
	}

	public DoctorVisit getLastDoctorVisit(String bookingId) {
		if (StringUtils.isBlank(bookingId)) {
			return null;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(DoctorVisit.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(DoctorVisit.PROP_ADMISSION_ID, bookingId));
			criteria.addOrder(Order.desc(DoctorVisit.PROP_CREATED_DATE));
			criteria.setMaxResults(1);
			return (DoctorVisit) criteria.uniqueResult();
		}
	}

	public List<DoctorVisit> getLatestDoctorVisits(String bookingId, int maxResult) {
		if (StringUtils.isBlank(bookingId)) {
			return null;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(DoctorVisit.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(DoctorVisit.PROP_ADMISSION_ID, bookingId));
			criteria.setMaxResults(maxResult);
			criteria.addOrder(Order.desc(DoctorVisit.PROP_CREATED_DATE));
			return criteria.list();
		}
	}
}