package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.Appointment;
import com.floreantpos.model.AppointmentStatus;
import com.floreantpos.model.Prescription;

public class PrescriptionDAO extends BasePrescriptionDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PrescriptionDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		Serializable save = super.save(obj, s);
		performPostSaveOperations((Prescription) obj, s);
		return save;
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
		performPostSaveOperations((Prescription) obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
		performPostSaveOperations((Prescription) obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof Prescription) {
			Prescription purchaseOrder = (Prescription) obj;
			purchaseOrder.setDeleted(Boolean.TRUE);
			super.update(purchaseOrder, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	public void performPostSaveOperations(Prescription prescription, Session session) {
		PatientHistoryDAO.getInstance().savePrescriptionHistory(prescription, session);
	}

	public void savePrescriptionAndAppointment(Prescription prescription, String appoinmentId) {
		try (Session session = createNewSession()) {
			Transaction transaction = session.beginTransaction();

			save(prescription, session);

			if (StringUtils.isNotBlank(appoinmentId)) {
				Appointment appointment = session.get(Appointment.class, appoinmentId);
				if (appointment != null && !Arrays.asList(AppointmentStatus.CANCELLED.name(), AppointmentStatus.PATIENT_NOT_AVAILABLE.name())
						.contains(appointment.getStatus())) {
					appointment.setStatus(AppointmentStatus.COMPLETED.name());
					appointment.putPrescriptionId(prescription.getId());
					session.saveOrUpdate(appointment);
				}

				prescription.setAppoinmentId(appoinmentId);
			}

			update(prescription, session);

			transaction.commit();
		}
	}

	public List<Prescription> getPrescriptionsByPatientId(String patientId) {
		if (StringUtils.isBlank(patientId)) {
			throw new PosException("Patient ID is required");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Prescription.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Prescription.PROP_PATIENT_ID, patientId));
			return criteria.list();
		}
	}
}