package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.Chemotherapy;
import com.floreantpos.model.Customer;
import com.orocube.medlogics.chemo.ChemotherapyStatus;

public class ChemotherapyDAO extends BaseChemotherapyDAO {

	public ChemotherapyDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Chemotherapy bean = (Chemotherapy) obj;
		if (bean == null) {
			throw new PosException("Chemotherapy not found!");
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<Chemotherapy> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

	/**
	 * Find all chemotherapy records for a specific patient
	 * @param patient the patient
	 * @return list of chemotherapy records
	 */
	public List<Chemotherapy> findByPatient(Customer patient) {
		if (patient == null || patient.getId() == null) {
			throw new IllegalArgumentException("Patient must not be null and must have an ID");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Chemotherapy.PROP_PATIENT_ID, patient.getId()));
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

	/**
	 * Find all chemotherapy records for a specific patient by ID
	 * @param patientId the patient ID
	 * @return list of chemotherapy records
	 */
	public List<Chemotherapy> findByPatientId(String patientId) {
		if (patientId == null) {
			throw new IllegalArgumentException("Patient ID must not be null");
		}
		Customer patient = CustomerDAO.getInstance().get(patientId);
		if (patient == null) {
			throw new PosException("Patient not found with ID: " + patientId);
		}
		return findByPatient(patient);
	}

	/**
	 * Find chemotherapy records by status
	 * @param status the chemotherapy status
	 * @return list of chemotherapy records
	 */
	public List<Chemotherapy> findByStatus(ChemotherapyStatus status) {
		if (status == null) {
			throw new IllegalArgumentException("Status must not be null");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Chemotherapy.PROP_STATUS, status.name()));
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

	/**
	 * Find ongoing chemotherapy records for a patient
	 * @param patient the patient
	 * @return list of ongoing chemotherapy records
	 */
	public List<Chemotherapy> findOngoingByPatient(Customer patient) {
		if (patient == null || patient.getId() == null) {
			throw new IllegalArgumentException("Patient must not be null and must have an ID");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Chemotherapy.PROP_PATIENT_ID, patient.getId()));
			criteria.add(Restrictions.eq(Chemotherapy.PROP_STATUS, ChemotherapyStatus.ONGOING.name()));
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

	/**
	 * Find all chemotherapy records for a specific outlet
	 * @param outletId the outlet ID
	 * @return list of chemotherapy records
	 */
	public List<Chemotherapy> findByOutlet(String outletId) {
		if (outletId == null) {
			throw new IllegalArgumentException("Outlet ID must not be null");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Chemotherapy.PROP_OUTLET_ID, outletId));
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

	/**
	 * Count total chemotherapy records for a patient
	 * @param patient the patient
	 * @return count of chemotherapy records
	 */
	public Long countByPatient(Customer patient) {
		if (patient == null || patient.getId() == null) {
			throw new IllegalArgumentException("Patient must not be null and must have an ID");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Chemotherapy.PROP_PATIENT_ID, patient.getId()));
			criteria.setProjection(org.hibernate.criterion.Projections.rowCount());
			return (Long) criteria.uniqueResult();
		}
	}
}
