package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuItemModifierPage;
import com.floreantpos.model.MenuItemModifierSpec;
import com.floreantpos.model.MenuModifier;
import com.floreantpos.model.ModifierGroup;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ModifierGroupDAO extends BaseModifierGroupDAO {
	private static final String EMPTY_NEWLINE_STRING = "\n"; //$NON-NLS-1$

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public ModifierGroupDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public List<ModifierGroup> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	private String constructExceptionDetailsByMenuitem(ModifierGroup modifierGroup, List<String> foreignItemNames) {
		if (foreignItemNames != null && !foreignItemNames.isEmpty()) {
			StringBuilder builder = new StringBuilder(
					Messages.getString("DataImportAction.10") + " " + modifierGroup.getName() + " " + Messages.getString("DiscountDAO.3")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < foreignItemNames.size(); i++) {
				String message = (i + 1) + "." + " " + foreignItemNames.get(i); //$NON-NLS-1$ //$NON-NLS-2$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	@Override
	protected void delete(Object obj, Session session) {
		ModifierGroup modifierGroup = (ModifierGroup) obj;
		if (modifierGroup == null) {
			throw new PosException(Messages.getString("ModifierGroupDAO.0")); //$NON-NLS-1$
		}

		StringBuilder details = new StringBuilder();
		List<String> foreignItemNames = getMenuItemByModifierGroup(modifierGroup, session);
		if (foreignItemNames != null && !foreignItemNames.isEmpty()) {
			details.append(this.constructExceptionDetailsByMenuitem(modifierGroup, foreignItemNames));
			details.append(EMPTY_NEWLINE_STRING);
		}

		String message = Messages.getString("ModifierGroupDAO.1"); //$NON-NLS-1$
		String detailsMessage = details.toString();
		if (StringUtils.isNotBlank(detailsMessage)) {
			throw new PosException(message, detailsMessage);
		}

		modifierGroup.setDeleted(true);
		update(modifierGroup, session);
		//deleteModifierGroupFromModifierSpec(modifierGroup, session);

	}

	private List<String> getMenuItemByModifierGroup(ModifierGroup modifierGroup, Session session) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		criteria.setProjection(Projections.distinct(Projections.property(MenuItem.PROP_NAME)));
		criteria.createAlias("menuItemModiferSpecs", "specs"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("specs." + MenuItemModifierSpec.PROP_MODIFIER_GROUP_ID, modifierGroup.getId())); //$NON-NLS-1$
		addDeletedFilter(criteria, MenuItem.class);
		return criteria.list();
	}

	private void deleteModifierGroupFromModifierSpec(ModifierGroup modifierGroup, Session session) {

		Criteria criteria = session.createCriteria(MenuItemModifierSpec.class);
		criteria.add(Restrictions.eq(MenuItemModifierSpec.PROP_MODIFIER_GROUP_ID, modifierGroup.getId()));
		List<MenuItemModifierSpec> itemModifierSpecs = criteria.list();
		if (itemModifierSpecs == null || itemModifierSpecs.isEmpty()) {
			return;
		}
		for (MenuItemModifierSpec itemModifierSpec : itemModifierSpecs) {
			removeMenuItemModifierSpecFromMenuItem(itemModifierSpec, session);
			MenuItemModifierSpecDAO.getInstance().delete(itemModifierSpec, session);
		}
	}

	private void removeMenuItemModifierSpecFromMenuItem(MenuItemModifierSpec modifierSpec, Session session) {

		Criteria criteria = session.createCriteria(MenuItem.class);
		criteria.createAlias("menuItemModiferSpecs", "modifierSpecs"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("modifierSpecs." + MenuItem.PROP_ID, modifierSpec.getId())); //$NON-NLS-1$
		List<MenuItem> menuItems = criteria.list();
		if (menuItems == null || menuItems.isEmpty()) {
			return;
		}
		for (MenuItem menuItem : menuItems) {
			menuItem.getMenuItemModiferSpecs().remove(modifierSpec);
			MenuItemDAO.getInstance().update(menuItem, session);
		}

	}

	public List<ModifierGroup> findNormalModifierGroups() {
		Session session = null;
		Criteria criteria = null;

		try {
			session = createNewSession();
			criteria = session.createCriteria(ModifierGroup.class);
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(ModifierGroup.PROP_NAME).ignoreCase());
			return criteria.list();
		} finally {
			session.close();
		}
	}

	/*
	 * 
	 * Note: This method is used to save ModifierGroup.PROP_PIZZA_MODIFIER_GROUP 
	 * property true for pizza modifier group.
	 * 
	 */

	public void updateModifierGroupBooleanPropertyValue(boolean pizzaGroup) {
		Session session = null;
		Criteria criteria = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			criteria = session.createCriteria(ModifierGroup.class);
			criteria.createAlias("modifiers", "m");//$NON-NLS-1$ //$NON-NLS-2$

			if (pizzaGroup)
				criteria.add(Restrictions.eq("m.pizzaModifier", Boolean.TRUE)); //$NON-NLS-1$
			else
				criteria.add(Restrictions.ne("m.pizzaModifier", Boolean.TRUE)); //$NON-NLS-1$

			List list = criteria.list();
			if (list != null && list.size() > 0) {
				for (Iterator iterator = list.iterator(); iterator.hasNext();) {
					ModifierGroup group = (ModifierGroup) iterator.next();
					saveOrUpdate(group, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
		} finally {
			session.close();
		}
	}

	public ModifierGroup findModifierGroupByName(String name) {
		try (Session session = createNewSession()) {
			return findModifierGroupByName(name, session);
		}
	}

	public ModifierGroup findModifierGroupByName(String name, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(ModifierGroup.PROP_NAME, name).ignoreCase());
		criteria.setMaxResults(1);
		return (ModifierGroup) criteria.uniqueResult();
	}

	public void checkDuplicateNameExist(String existingId, String name) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(ModifierGroup.PROP_NAME, name).ignoreCase());
			if (StringUtils.isNotEmpty(existingId)) {
				criteria.add(Restrictions.ne(ModifierGroup.PROP_ID, existingId));
			}
			addDeletedFilter(criteria);
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null && rowCount.intValue() > 0) {
				throw new DuplicateDataException(String.format(Messages.getString("GenericDAO.20") + " %s " + Messages.getString("GenericDAO.22"), name)); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
		}

	}

	public List<ModifierGroup> findAllUnSyncModifierGroup() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			Criterion falseCriteria = Restrictions.eq(ModifierGroup.PROP_CLOUD_SYNCED, Boolean.FALSE);
			Criterion nullCriteria = Restrictions.isNull(ModifierGroup.PROP_CLOUD_SYNCED);
			criteria.add(Restrictions.or(falseCriteria, nullCriteria));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void updateModifierGroupSync(List<String> ids, Boolean synced) {

		if (ids == null || ids.isEmpty())
			return;

		String whereCondition = "("; //$NON-NLS-1$
		for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
			String id = (String) iterator.next();
			whereCondition += "'" + id + "'"; //$NON-NLS-1$ //$NON-NLS-2$
			if (iterator.hasNext())
				whereCondition += ","; //$NON-NLS-1$
		}
		whereCondition += ")"; //$NON-NLS-1$
		//@formatter:off
		Transaction tx=null;
		Session session = null;
		try {
			session = getSession();
			tx=session.beginTransaction();
		String hqlString = "update ModifierGroup set %s=:synced where %s in"+whereCondition; //$NON-NLS-1$
		hqlString = String.format(hqlString, ModifierGroup.PROP_CLOUD_SYNCED, ModifierGroup.PROP_ID);
		//@formatter:on
			Query query = session.createQuery(hqlString);
			query.setParameter("synced", synced); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}

	/**
	 * Search Modifier Group By Name (MatchMode.ANYWHERE)
	 */
	public void findModifierGroupByName(PaginationSupport paginationSupport, String itemName, boolean pizzaType) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());

			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(ModifierGroup.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}

			criteria.setProjection(null);
			criteria.addOrder(Order.asc(ModifierGroup.PROP_NAME));

			paginationSupport.setRows(criteria.list());
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateModifierGroups(List<ModifierGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		saveOrUpdateModifierGroups(dataList, updateLastUpdateTime, updateSyncTime, false, false);
	}

	public void saveOrUpdateModifierGroups(List<ModifierGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, boolean saveNewDataOnly,
			boolean forceUpdate) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			Set<MenuModifier> modifiers = new HashSet<MenuModifier>();
			for (Iterator<ModifierGroup> iterator = dataList.iterator(); iterator.hasNext();) {
				ModifierGroup item = (ModifierGroup) iterator.next();
				List<MenuModifier> groupModifiers = item.getModifiers();
				if (groupModifiers != null && groupModifiers.size() > 0) {
					modifiers.addAll(groupModifiers);
				}
			}
			if (modifiers.size() > 0) {
				List<String> nonExistingModifierIds = MenuModifierDAO.getInstance().getNonExistingModifierIds(modifiers);
				if (nonExistingModifierIds != null && nonExistingModifierIds.size() > 0) {
					PosLog.error(getClass(), "The following modifiers are not present in db: " + nonExistingModifierIds); //$NON-NLS-1$
					for (MenuModifier menuModifier : modifiers) {
						if (nonExistingModifierIds.contains(menuModifier.getId())) {
							menuModifier.setUpdateLastUpdateTime(false);
							menuModifier.setUpdateSyncTime(false);
							MenuModifierDAO.getInstance().save(menuModifier, session);
							PosLog.debug(getClass(), menuModifier.getName() + "(" + menuModifier.getId() + ") saved."); //$NON-NLS-1$ //$NON-NLS-2$
						}
					}
				}
			}
			for (Iterator<ModifierGroup> iterator = dataList.iterator(); iterator.hasNext();) {
				ModifierGroup item = (ModifierGroup) iterator.next();
				ModifierGroup existingModifierGroup = get(item.getId(), session);
				if (existingModifierGroup != null && !forceUpdate
						&& (saveNewDataOnly || !BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingModifierGroup.getLastUpdateTime()))) {
					PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
					continue;
				}
				BaseDataServiceDao.get().saveOrSetVersionModifierGroup(session, updateLastUpdateTime, updateSyncTime, item, existingModifierGroup);
				saveOrUpdate(item, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void initialize(ModifierGroup modifierGroup) {
		if (modifierGroup == null || modifierGroup.getId() == null) {
			return;
		}
		if (Hibernate.isInitialized(modifierGroup.getModifiers()) && Hibernate.isInitialized(modifierGroup.getModifierPages())) {
			return;
		}
		Session session = null;

		try {
			session = createNewSession();
			session.refresh(modifierGroup);
			Hibernate.initialize(modifierGroup.getModifiers());
			Hibernate.initialize(modifierGroup.getModifierPages());
		} finally {
			closeSession(session);
		}
	}

	public void findModifierGroups(PaginationSupport model, String itemName) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(itemName))
				criteria.add(Restrictions.ilike(ModifierGroup.PROP_NAME, itemName, MatchMode.ANYWHERE));

			model.setNumRows(rowCount(criteria));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			criteria.addOrder(Order.asc(ModifierGroup.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(ModifierGroup.PROP_NAME).ignoreCase());
			model.setRows(criteria.list());
		}
	}

	public boolean isChanged(ModifierGroup modifierGroup) {
		if (StringUtils.isBlank(modifierGroup.getId())) {
			return true;
		}

		try (Session session = createNewSession()) {
			ModifierGroup savedGroup = get(modifierGroup.getId(), session);
			if (savedGroup == null) {
				return true;
			}

			if (modifierGroup.getMinQuantity() != savedGroup.getMinQuantity()) {
				return true;
			}
			if (modifierGroup.getMaxQuantity() != savedGroup.getMaxQuantity()) {
				return true;
			}
			if (modifierGroup.getEnable() != savedGroup.getEnable()) {
				return true;
			}
			if (modifierGroup.isAutoShow() != savedGroup.isAutoShow()) {
				return true;
			}
			if (modifierGroup.isEnableGroupPricing() != savedGroup.isEnableGroupPricing()) {
				return true;
			}
			if (!StringUtils.equals(modifierGroup.getInstruction(), savedGroup.getInstruction())) {
				return true;
			}
			if (!StringUtils.equals(modifierGroup.getProperty(MenuItemModifierSpec.JSON_PROP_PRICE_RULES),
					savedGroup.getProperty(MenuItemModifierSpec.JSON_PROP_PRICE_RULES))) {
				return true;
			}
			List<MenuModifier> modifiers = new ArrayList<MenuModifier>(modifierGroup.getModifiers());
			List<MenuModifier> savedModifiers = savedGroup.getModifiers();

			Comparator<MenuModifier> comparator = new Comparator<MenuModifier>() {
				@Override
				public int compare(MenuModifier o1, MenuModifier o2) {
					if (o1.getId() == null || o2.getId() == null) {
						return 1;
					}

					return o1.getId().compareToIgnoreCase(o2.getId());
				}
			};
			Collections.sort(modifiers, comparator);
			Collections.sort(savedModifiers, comparator);

			boolean modifiersSame = CollectionUtils.isEqualCollection(modifiers, savedModifiers);
			if (!modifiersSame) {
				return true;
			}
			Set<MenuItemModifierPage> modifierPages = savedGroup.getModifierPages();
			if (modifiers.size() > 0 && (modifierPages.isEmpty() || modifierPages.size() > 1)) {
				return true;
			}
		}

		return false;
	}
}
