package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.NullPrecedence;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosException;
import com.floreantpos.model.Patient;
import com.floreantpos.model.SurgeryInfo;
import com.floreantpos.model.Ticket;
import com.floreantpos.swing.PaginatedListModel;

public class SurgeryInfoDAO extends BaseSurgeryInfoDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		SurgeryInfo bean = (SurgeryInfo) obj;
		if (bean == null) {
			throw new PosException("Surgery Info not found!");
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<SurgeryInfo> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(SurgeryInfo.class);
			addDeletedFilter(criteria);

			criteria.addOrder(Order.asc(SurgeryInfo.PROP_REQUESTED_DATE));
			return criteria.list();
		}
	}

	public void loadSurgeryInfo(PaginatedListModel<SurgeryInfo> dataModel) {
		loadSurgeryInfo(dataModel, null);
	}

	public void loadSurgeryInfo(PaginatedListModel<SurgeryInfo> dataModel, Patient patient) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(SurgeryInfo.class);
			addDeletedFilter(criteria);
			if (patient != null) {
				criteria.add(Restrictions.eq(SurgeryInfo.PROP_PATIENT_ID, patient.getId()));
			}

			dataModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());
			criteria.addOrder(Order.asc(SurgeryInfo.PROP_REQUESTED_DATE));
			dataModel.setData(criteria.list());
		}
	}

	public void saveSurgeryInfoWithTicket(Ticket ticketToOrder, SurgeryInfo surgeryInfo) {
		try (Session session = createNewSession()) {
			Transaction tx = session.beginTransaction();

			boolean isNewOtOrder = StringUtils.isBlank(ticketToOrder.getId());
			if (!isNewOtOrder) {
				ticketToOrder.putOTRequestId(surgeryInfo.getId());
				ticketToOrder.putOTRequestStatus(surgeryInfo.getSurgeryStatus());
				TicketDAO.getInstance().saveOrUpdate(ticketToOrder, session);
			}

			TicketDAO.getInstance().saveOrUpdate(ticketToOrder, session);

			surgeryInfo.setOrderId(ticketToOrder.getId());

			if (StringUtils.isBlank(surgeryInfo.getId())) {
				save(surgeryInfo, session);
			}
			else {
				saveOrUpdate(surgeryInfo, session);
			}

			if (isNewOtOrder) {
				ticketToOrder.putOTRequestId(surgeryInfo.getId());
				ticketToOrder.putOTRequestStatus(surgeryInfo.getSurgeryStatus());
				TicketDAO.getInstance().saveOrUpdate(ticketToOrder, session);
			}

			tx.commit();
		}
	}

	public SurgeryInfo getSurgeryInfoByAdmissionId(String admissionId) {
		if (StringUtils.isBlank(admissionId)) {
			return null;
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(SurgeryInfo.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(SurgeryInfo.PROP_ADMISSION_ID, admissionId));
			criteria.setMaxResults(1);
			return (SurgeryInfo) criteria.uniqueResult();
		}
	}

	public List<SurgeryInfo> getSurgeryInfos(String doctorId, String patientId, String orderId, int offset, int limit) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(SurgeryInfo.class);

			//			ProjectionList projectionList = Projections.projectionList();
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_ID), SurgeryInfo.PROP_ID);
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_PATIENT_ID), SurgeryInfo.PROP_PATIENT_ID);
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_CREATE_DATE), SurgeryInfo.PROP_CREATE_DATE);
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_REQUESTED_DATE), SurgeryInfo.PROP_REQUESTED_DATE);
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_REQUESTED_TIME), SurgeryInfo.PROP_REQUESTED_TIME);
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_APPROVED_DATE), SurgeryInfo.PROP_APPROVED_DATE);
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_APPROVED_TIME), SurgeryInfo.PROP_APPROVED_TIME);
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_ORDER_ID), SurgeryInfo.PROP_ORDER_ID);
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_SURGERY_STATUS), SurgeryInfo.PROP_SURGERY_STATUS);
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_SURGERY_PROCEDURE_STATUS), SurgeryInfo.PROP_SURGERY_PROCEDURE_STATUS);
			//			projectionList.add(Projections.property(SurgeryInfo.PROP_PROPERTIES), SurgeryInfo.PROP_PROPERTIES);
			//
			//			criteria.setProjection(projectionList);
			if (doctorId != null) {
				criteria.createAlias("surgeons", "dr"); //$NON-NLS-1$ //$NON-NLS-2$
				criteria.add(Restrictions.eq("dr.id", doctorId)); //$NON-NLS-1$
			}
			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq(SurgeryInfo.PROP_PATIENT_ID, patientId));
			}
			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.ilike(SurgeryInfo.PROP_ORDER_ID, orderId.trim(), MatchMode.END));
			}
			addDeletedFilter(criteria);
			criteria.setFirstResult(offset);
			criteria.setMaxResults(limit);
			criteria.addOrder(Order.desc(SurgeryInfo.PROP_CREATE_DATE).nulls(NullPrecedence.LAST));
			//criteria.setResultTransformer(Transformers.aliasToBean(SurgeryInfo.class));
			return criteria.list();
		}
	}

	public int getSurgeryRowCount(String doctorId, String patientId, String orderId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(SurgeryInfo.class);
			criteria.setProjection(Projections.rowCount());
			if (doctorId != null) {
				criteria.createAlias("surgeons", "dr");
				criteria.add(Restrictions.eq("dr.id", doctorId)); //$NON-NLS-1$
			}
			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq(SurgeryInfo.PROP_PATIENT_ID, patientId));
			}
			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.ilike(SurgeryInfo.PROP_ORDER_ID, orderId.trim(), MatchMode.END));
			}
			addDeletedFilter(criteria);
			Number uniqueResult = (Number) criteria.uniqueResult();
			return uniqueResult == null ? 0 : uniqueResult.intValue();
		}
	}

}