package com.floreantpos.model;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BasePrescriptionItem;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

@XmlRootElement
@JsonIgnoreProperties(ignoreUnknown = true, value = { "prescription", "menuItem" })
@XmlAccessorType(XmlAccessType.FIELD)
public class PrescriptionItem extends BasePrescriptionItem implements PropertyContainer2 {
	private static final long serialVersionUID = 1L;
	
	private transient MenuItem menuItem;
	private transient com.google.gson.JsonObject propertiesContainer;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public PrescriptionItem() {
	}

	/**
	 * Constructor for primary key
	 */
	public PrescriptionItem(java.lang.String id) {
		super(id);
	}

	/**
	 * Constructor for required fields
	 */
	public PrescriptionItem(java.lang.String id, com.floreantpos.model.Prescription prescription) {

		super(id, prescription);
	}

	/*[CONSTRUCTOR MARKER END]*/

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	@XmlTransient
	@JsonIgnore
	public MenuItem getMenuItem() {
		if (menuItem == null) {
			String itemId = getMenuItemId();
			if (StringUtils.isEmpty(itemId)) {
				return null;
			}
			menuItem = MenuItemDAO.getInstance().loadInitialized(itemId);
		}

		return menuItem;
	}

	public void setMenuItem(MenuItem menuItem) {
		this.menuItem = menuItem;
		String menuItemId = null;
		if (menuItem != null) {
			menuItemId = menuItem.getId();
		}
		super.setMenuItemId(menuItemId);
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public String getMedicationDurationDisplay(boolean showBanglaTextOnPrescription) {
		double medicationDuration = NumberUtil.parseDouble(getProperty("medicine.time.interval"));
		//if (medicationDuration <= 0) {
		//	return StringUtils.EMPTY;
		//}
		String displayDuration = StringUtils.EMPTY;
		if (medicationDuration > 0) {
			if (showBanglaTextOnPrescription) {
				displayDuration = " " + NumberUtil.convertToBanglaNumber(NumberUtil.formatNumberIfNeeded(medicationDuration));
			}
			else {
				displayDuration = " " + NumberUtil.formatNumberIfNeeded(medicationDuration);
			}
		}

		MedicationDurationType medDurationType = MedicationDurationType.fromString(getProperty("medication.duration.type"));
		if (medDurationType == null) {
			return displayDuration;
		}
		switch (medDurationType) {
			case TIME:
				displayDuration = showBanglaTextOnPrescription ? " এখনি" : " now";
				break;

			case DAY:
				if (medicationDuration > 0) {
					if (showBanglaTextOnPrescription) {
						displayDuration += " দিন";
					}
					else {
						displayDuration += (medicationDuration > 1 ? " Days" : " Day");
					}
				}
				break;

			case MONTH:
				if (medicationDuration > 0) {
					if (showBanglaTextOnPrescription) {
						displayDuration += " মাস";
					}
					else {
						displayDuration += (medicationDuration > 1 ? " Months" : " Month");
					}
				}
				break;

			case INFINITY:
				if (showBanglaTextOnPrescription) {
					displayDuration = " (চলবে)";
				}
				else {
					displayDuration = " (Continue)";
				}
				break;

			default:
				break;
		}
		return displayDuration;
	}

	public String getMedTakingTime() {
		return getProperty("medicine.time", "");
	}

	public void putMedTakingTime(String medTakingTime) {
		addProperty("medicine.time", medTakingTime); //$NON-NLS-1$
	}

	public boolean isMorning() {
		return getBooleanProperty("medicine.time.morning", false); //$NON-NLS-1$
	}

	public void putMorning(boolean isMorning) {
		addProperty("medicine.time.morning", String.valueOf(isMorning)); //$NON-NLS-1$
	}

	public boolean isNoon() {
		return getBooleanProperty("medicine.time.noon", false); //$NON-NLS-1$
	}

	public void putNoon(boolean isNoon) {
		addProperty("medicine.time.noon", String.valueOf(isNoon)); //$NON-NLS-1$
	}

	public boolean isAfterNoon() {
		return getBooleanProperty("medicine.time.after_noon", false); //$NON-NLS-1$
	}

	public void putAfterNoon(boolean isAfterNoon) {
		addProperty("medicine.time.after_noon", String.valueOf(isAfterNoon)); //$NON-NLS-1$
	}

	public boolean isNight() {
		return getBooleanProperty("medicine.time.night", false); //$NON-NLS-1$
	}

	public void putNight(boolean isNight) {
		addProperty("medicine.time.night", String.valueOf(isNight)); //$NON-NLS-1$
	}

	public String getMedicineTimeNote() {
		return getProperty("medicine.time.notes", "");
	}

	public void putMedicineTimeNote(String medTimeNote) {
		addProperty("medicine.time.notes", medTimeNote); //$NON-NLS-1$
	}

	public double getTakingQty() {
		return getDoubleProperty("medicine.quantity");
	}

	public void putTakingQty(double takingQty) {
		addProperty("medicine.quantity", String.valueOf(takingQty)); //$NON-NLS-1$
	}

	public String getTakingQtyUnit() {
		return getProperty("medicine.quantity.unit", "");
	}

	public void putTakingQtyUnit(String takingQtyUnit) {
		addProperty("medicine.quantity.unit", takingQtyUnit); //$NON-NLS-1$
	}

	public double getMedicationDuration() {
		return getDoubleProperty("medicine.time.interval");
	}

	public void putMedicationDuration(double medicineDosageInterval) {
		addProperty("medicine.time.interval", String.valueOf(medicineDosageInterval)); //$NON-NLS-1$
	}

	public String getMedicationDurationType() {
		return getProperty("medication.duration.type");
	}

	public void putMedicationDurationType(MedicationDurationType type) {
		addProperty("medication.duration.type", type != null ? type.name() : null); //$NON-NLS-1$
	}

	public String getItemNameDisplayWithInstruction(boolean showInDetailView, Boolean showBanglaTextOnPrescription) {
		String medicineNames = "";
		MenuItem menuItem = getMenuItem();
		String menuItemName = menuItem.getDisplayName();

		if (showInDetailView) {
			medicineNames += "<ul class=\"list_ul\">"; //$NON-NLS-1$
			medicineNames += "<li class=\"list_item\">" + menuItemName + "</li>"; //$NON-NLS-1$ //$NON-NLS-2$
		}
		else {
			medicineNames += "<div>&#x2022; " + menuItemName + "</div>"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		return medicineNames += getInstructionDisplay(showInDetailView, showBanglaTextOnPrescription);
	}

	public String getInstructionDisplay(boolean showInDetailView, Boolean showBanglaTextOnPrescription) {
		String instructionTxt = StringUtils.EMPTY;

		Boolean morning = isMorning();
		Boolean noon = isNoon();
		Boolean afterNoon = isAfterNoon();
		Boolean night = isNight();

		String divStartTag;
		if (showInDetailView) {
			divStartTag = "<div class=\"list_item\">"; //$NON-NLS-1$
		}
		else {
			divStartTag = "<div>"; //$NON-NLS-1$
		}
		instructionTxt += divStartTag; //$NON-NLS-1$

		if (morning || noon || afterNoon || night) {
			StringBuilder medTimeBuilder = new StringBuilder();
			medTimeBuilder.append(showBanglaTextOnPrescription ? morning ? "১" : "০" : morning ? "1" : "0"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			medTimeBuilder.append(" + "); //$NON-NLS-1$

			medTimeBuilder.append(showBanglaTextOnPrescription ? noon ? "১" : "০" : noon ? "1" : "0"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			medTimeBuilder.append(" + "); //$NON-NLS-1$

			if (afterNoon) {
				medTimeBuilder.append(showBanglaTextOnPrescription ? afterNoon ? "১" : "০" : afterNoon ? "1" : "0"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				medTimeBuilder.append(" + "); //$NON-NLS-1$
			}
			medTimeBuilder.append(showBanglaTextOnPrescription ? night ? "১" : "০" : night ? "1" : "0"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			medTimeBuilder.append(" "); //$NON-NLS-1$
			instructionTxt += medTimeBuilder.toString();
		}
		MedicationTakingInstruction takingInstruction = MedicationTakingInstruction.fromString(getMedTakingTime());
		String medicineTime = takingInstruction == null ? StringUtils.EMPTY : takingInstruction.toString();
		if (StringUtils.isNotBlank(medicineTime) && MedicationTakingInstruction.ANY != takingInstruction) {
			if (showBanglaTextOnPrescription) {
				if (MedicationTakingInstruction.BEFORE_MEAL == takingInstruction) {
					medicineTime = "খাবারের আগে"; //$NON-NLS-1$
				}
				else if (MedicationTakingInstruction.AFTER_MEAL == takingInstruction) {
					medicineTime = "খাবারের পর"; //$NON-NLS-1$
				}
				else if (MedicationTakingInstruction.IMMEDIATE == takingInstruction) {
					medicineTime = "এখনি"; //$NON-NLS-1$
				}
			}

			instructionTxt += " " + medicineTime; //$NON-NLS-1$
		}

		if (!(MedicationTakingInstruction.IMMEDIATE == takingInstruction)) {
			String medicationDurationDisplay = getMedicationDurationDisplay(showBanglaTextOnPrescription);
			if (StringUtils.isNotBlank(medicationDurationDisplay)) {
				instructionTxt += (divStartTag.equals(instructionTxt) ? StringUtils.EMPTY : ", ") + medicationDurationDisplay;
			}
		}
		instructionTxt += "</div>"; //$NON-NLS-1$

		if (showInDetailView) {
			instructionTxt += "<div class=\"list_item\">"; //$NON-NLS-1$
		}
		else {
			instructionTxt += "<div>"; //$NON-NLS-1$
		}
		String medicineQuantity = NumberUtil.formatNumberIfNeeded(getTakingQty());
		if (StringUtils.isNotBlank(medicineQuantity) && !medicineQuantity.equals("0")) { //$NON-NLS-1$
			if (showBanglaTextOnPrescription) {
				instructionTxt += " " + NumberUtil.convertToBanglaNumber(medicineQuantity); //$NON-NLS-1$
			}
			else {
				instructionTxt += " " + medicineQuantity; //$NON-NLS-1$
			}

			String medicineQuantityUnit = getTakingQtyUnit();
			if (StringUtils.isNotBlank(medicineQuantityUnit)) {
				instructionTxt += " " + medicineQuantityUnit; //$NON-NLS-1$
			}
			else {
				if (showBanglaTextOnPrescription) {
					instructionTxt += " টি"; //$NON-NLS-1$
				}
				else {
					instructionTxt += " Piece"; //$NON-NLS-1$
				}
			}
			if (takingInstruction != null && !(MedicationTakingInstruction.IMMEDIATE == takingInstruction)) {
				if (showBanglaTextOnPrescription) {
					instructionTxt += " প্রতিবার";
				}
				else {
					instructionTxt += " each time";
				}
			}
		}
		instructionTxt += "</div>"; //$NON-NLS-1$

		if (showInDetailView) {
			instructionTxt += "</ul>"; //$NON-NLS-1$
		}

		String notes = getMedicineTimeNote();
		if (StringUtils.isNotBlank(notes)) {
			instructionTxt += "<div>" + notes + "</div>"; //$NON-NLS-1$ //$NON-NLS-2$
		}
		instructionTxt += "</br>"; //$NON-NLS-1$
		return instructionTxt;
	}

	public boolean isDurationContinue() {
		return getBooleanProperty("duration.continue", false);
	}

	public void putDurationContinue(boolean isDurationContinue) {
		addProperty("duration.continue", String.valueOf(isDurationContinue));
	}

	public boolean isDefaultItem() {
		return getBooleanProperty("default.item", false);
	}

	public void putDefaultItem(boolean isDefaultItem) {
		addProperty("default.item", String.valueOf(isDefaultItem));
	}

	public boolean isLabTest() {
		ProductType productType = ProductType.fromString(getProductType());
		if (ProductType.PATHOLOGY == productType || ProductType.RADIOLOGY == productType) {
			return true;
		}
		return false;
	}
	
	@Override
	@JsonIgnore
	public Prescription getPrescription() {
		return super.getPrescription();
	}
}