package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.extension.cardconnect.SecuredConstants;
import com.floreantpos.model.Address;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Store;
import com.floreantpos.model.WorkingHours;
import com.floreantpos.util.CopyUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class OutletDAO extends BaseOutletDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public OutletDAO() {

	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime((Outlet) obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime((Outlet) obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime((Outlet) obj);
		super.saveOrUpdate(obj, s);
	}

	public Outlet initialize(Outlet outlet) {
		if (outlet == null || outlet.getId() == null)
			return outlet;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(outlet);

			Hibernate.initialize(outlet.getDepartments());

			return outlet;
		} finally {
			closeSession(session);
		}
	}

	public List<Outlet> findAll() {
		return findAll(false);
	}

	public List<Outlet> findAll(boolean showDeleted) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			if (!showDeleted) {
				this.addDeletedFilter(criteria);
			}
			criteria.addOrder(Order.asc(Outlet.PROP_NAME).ignoreCase());
			return criteria.list();
		}
	}

	public List<Outlet> saveOrUpdateOutlets(List<Outlet> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return null;

		List<Outlet> updatedOutlets = new ArrayList<>();
		Transaction tx = null;
		Session session = null;
		try {
			Store store = StoreDAO.getRestaurant();
			session = createNewSession();
			tx = session.beginTransaction();
			for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
				Outlet item = (Outlet) iterator.next();
				Outlet existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					if (StringUtils.isBlank(item.getAddressLine1()) && StringUtils.isBlank(item.getCountry()) && StringUtils.isNotBlank(item.getAddressId())) {
						Address address = AddressDAO.getInstance().get(item.getAddressId());
						if (address != null) {
							CopyUtil.copyAddress(address, item);
						}
						//TODO: address and outlet address property not equal with request
						if (store.getDefaultOutletId() != null && store.getDefaultOutletId().equals(item.getId())) {
							item.setAddressLine1(store.getAddressLine1());
							item.setAddressLine2(store.getAddressLine2());
							item.setAddressLine3(store.getAddressLine3());
						}
					}
					copyCloudProperties(existingItem, item);
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
				updatedOutlets.add(item);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
		return updatedOutlets;
	}

	private void copyCloudProperties(Outlet cloudOutlet, Outlet clientOutlet) {
		try {
			List<String> cloudProperties = new ArrayList<String>();
			cloudProperties.add(Store.PROP_ONLINE_ORDER_TYPE);
			cloudProperties.add(Store.PROP_ONLINE_ORDER_PAYMENT_CASH_ON_DELIVERY);
			cloudProperties.add(Store.PROP_ONLINE_ORDER_PAYMENT_CARD_GATEWAY);
			cloudProperties.add(Store.PROP_ONLINE_ORDER_PAYMENT_CARD);
			cloudProperties.add(Store.PROP_ONLINE_ORDER_AUTHNET_MERCHANTE_ID);
			cloudProperties.add(Store.PROP_ONLINE_ORDER_AUTHNET_API_LOGIN_ID);
			cloudProperties.add(Store.PROP_ONLINE_ORDER_AUTHNET_SANDBOX_MODE);
			cloudProperties.add(Store.STORE_PROP_SUBSCRIPTION_PLAN);
			cloudProperties.add(Store.STORE_PROP_SUBSCRIPTION_TICKET_LIMIT);
			cloudProperties.add(SecuredConstants.ONLINEORDER_CARDCONNECT_HOSTSYSTEM);
			cloudProperties.add(SecuredConstants.ONLINEORDER_CARDCONNECT_MERCHANT_ID);
			cloudProperties.add(SecuredConstants.ONLINEORDER_CARDCONNECT_USERNAME);
			cloudProperties.add(SecuredConstants.ONLINEORDER_CARDCONNECT_PASSWORD);

			if (!clientOutlet.getBooleanProperty("overrideLocalSettings", false)) { //$NON-NLS-1$
				Set<String> cloudPropertyNames = cloudOutlet.getPropertyStore().keySet();
				for (String cloudPropertyName : cloudPropertyNames) {
					if (cloudPropertyName.contains(WorkingHours.OPEN) || cloudPropertyName.contains(WorkingHours.START_TIME)
							|| cloudPropertyName.contains(WorkingHours.END_TIME)) {

						if (StringUtils.isBlank(clientOutlet.getProperty(cloudPropertyName))) {
							clientOutlet.addProperty(cloudPropertyName, cloudOutlet.getProperty(cloudPropertyName));
						}
					}
				}
			}
			else {
				clientOutlet.removeProperty("overrideLocalSettings"); //$NON-NLS-1$
			}

			for (String cloudProperty : cloudProperties) {
				clientOutlet.addProperty(cloudProperty, cloudOutlet.getProperty(cloudProperty));
			}

			clientOutlet.setEnableOnlineOrdering(cloudOutlet.isEnableOnlineOrdering());
			clientOutlet.setDeliveryAvailable(cloudOutlet.isDeliveryAvailable());
			clientOutlet.setPickUpAvailable(cloudOutlet.isPickUpAvailable());
			clientOutlet.setLatitude(cloudOutlet.getLatitude());
			clientOutlet.setLongitude(cloudOutlet.getLongitude());
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public List getOutletsWithSameAddressId(String fieldValue) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Outlet.class);
			criteria.add(Restrictions.eq(Outlet.PROP_ADDRESS_ID, fieldValue).ignoreCase());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public boolean isOutletSyncable(String outletId) {

		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (outletId != null) {
				criteria.add(Restrictions.eq(Outlet.PROP_ID, outletId));
			}

			Criterion nullUpdateTime = Restrictions.isNull(Outlet.PROP_LAST_UPDATE_TIME);
			Criterion nullSyncTime = Restrictions.isNull(Outlet.PROP_LAST_SYNC_TIME);
			Criterion gtQuery = Restrictions.gtProperty(Outlet.PROP_LAST_UPDATE_TIME, Outlet.PROP_LAST_SYNC_TIME);

			criteria.add(Restrictions.or(nullUpdateTime, nullSyncTime, gtQuery));
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);

			List result = criteria.list();
			if (result.size() == 0) {
				return false;
			}
			return true;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveOrUpdateOutlet(Outlet bean, Address address) {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			if (address != null) {
				AddressDAO.getInstance().saveOrUpdate(address, session);
				bean.setAddress(address);
			}
			saveOrUpdate(bean, session);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	@SuppressWarnings("unchecked")
	public List<String> findAllOutletIds() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			criteria.setProjection(Projections.property(Outlet.PROP_ID));
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	//cloud sync method
	public void saveOrUpdateOutlet(Outlet item, String currentOutletId, boolean forceSave, boolean updateLastUpdateTime, boolean updateSyncTime,
			Session session) throws Exception {
		Outlet existingItem = get(item.getId());
		if (existingItem != null) {
			if (!forceSave && !BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
				PosLog.debug(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
				return;
			}
			long version = existingItem.getVersion();
			PropertyUtils.copyProperties(existingItem, item);
			existingItem.setVersion(version);
			existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
			existingItem.setUpdateSyncTime(updateSyncTime);
			//copyOutletToLocalStoreIfNeeeded(existingItem, currentOutletId, session);
			update(existingItem, session);
		}
		else {
			item.setVersion(0);
			item.setUpdateLastUpdateTime(updateLastUpdateTime);
			item.setUpdateSyncTime(updateSyncTime);
			//copyOutletToLocalStoreIfNeeeded(item, currentOutletId, session);
			save(item, session);
		}
	}

	public void saveOrUpdateWithMenuItems(Outlet outlet, Map<String, MenuItem> oTDefaultItemMap) {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			saveOrUpdate(outlet, session);

			MenuItemDAO instance = MenuItemDAO.getInstance();

			Set<String> keySet = oTDefaultItemMap.keySet();
			for (String otItemKey : keySet) {
				MenuItem menuItem = oTDefaultItemMap.get(otItemKey);
				if (StringUtils.isBlank(menuItem.getId())) {
					menuItem.setId(otItemKey);
					instance.save(menuItem, session);
				}
				else {
					instance.saveOrUpdate(menuItem, session);
				}
			}

			tx.commit();
		}
	}

}